# Simple barcode scanner using zbar and borg camera in a subdialog
# (or tcluvc on POSIX, or tclwmf on Windows)
#
# Usage:
#
#	set result [zbartool::read .toplevel ?camera?]
#
# Optional parameter camera is by default 0 for the front looking
# camera, but can be set to 1 for the face looking camera (AndroWish,
# in case it is available). Otherwise it can be the index or name of
# a tcluvc or tclwmf camera for the other platforms. The result is
# an empty list if nothing was decoded and/or the user hit the back
# button or escape key. The first successful decode stops the camera
# and displays the result below the image. The camera can be restarted
# by tapping on the image. The back button or escape key ends the
# dialog and returns whatever was the last decode result (which can
# be empty, see above).
#
# chw February 2019

package require Tk
package require zbar

namespace eval zbartool {

    variable img	;# current captured image
    variable img2	;# decoded image
    variable done	;# true when done
    variable result	;# decode result, can be empty
    variable fset	;# set camera focus flag (droid)
    variable droid	;# true when AndroWish
    variable win	;# true on Windows platform

    if {[info command "sdltk"] eq "sdltk" && [sdltk android]} {
	set droid 1
	set win 0
    } else {
	set droid 0
    }

    if {$droid} {
	package require Borg
    } elseif {$::tcl_platform(platform) eq "windows"} {
	set win 1
	package require tclwmf
    } else {
	set win 0
	package require tcluvc
    }

    proc read {w {cam 0}} {
	variable img
	variable img2
	variable done
	variable result
	variable fset
	variable droid
	variable win
	if {[winfo exists $w]} {
	    return -code error "toplevel $w exists"
	}
	toplevel $w
	if {$droid} {
	    borg camera stop
	    borg camera close
	    if {[catch {borg camera open $cam} ret]} {
		destroy $w
		return -code error "open camera: $ret"
	    } elseif {!$ret} {
		destroy $w
		return -code error "camera not found"
	    }
	    set sw [winfo screenwidth $w]
	    set sh [winfo screenheight $w]
	    if {$sh > $sw} {
		set tmp $sh
		set sh $sw
		set sw $sh
	    }
	    set pvw 640
	    set pvh 480
	    # assume preview sizes are sorted descending
	    foreach parm [split [dict get [borg camera parameters] \
				     preview-size-values] ","] {
		scan $parm "%dx%d" pw ph
		if {$sh > $sw} {
		    if {$pw < $sw * 0.6} {
			set pvw $pw
			set pwv $ph
			break
		    }
		} else {
		    if {$ph < $sh * 0.6} {
			set pvw $pw
			set pvh $ph
			break
		    }
		}
	    }
	    borg camera parameters preview-frame-rate 15
	    borg camera parameters preview-size "${pvw}x${pvh}"
	    scan [dict get [borg camera parameters] preview-size] \
		"%dx%d" width height
	} elseif {$win} {
	    if {[string is integer $cam]} {
		# each device entry has two elements
		set name [lindex [wmf devices] [expr {$cam * 2}]]
	    } else {
		# assume cam is given as the symbolic link
		set name $cam
	    }
	    if {[catch {
		wmf open $name [list [namespace current]::capture_wmf $w]
	    } dev]} {
		destroy $w
		return -code error "open camera: $dev"
	    }
	    array set fmts [wmf listformats $dev]
	    foreach i [lsort -integer [array names fmts]] {
		array set fmt $fmts($i)
		if {[scan $fmt(frame-size) "%dx%d" width height] == 2} {
		    if {$width == 640} {
			wmf format $dev $i
			break
		    }
		    if {$height == 480} {
			wmf format $dev $i
			break
		    }
		}
	    }
	    set i [wmf format $dev]
	    array set fmt $fmts($i)
	    if {[scan $fmt(frame-size) "%dx%d" width height] != 2} {
		wmf close $dev
		destroy $w
		return -code "cannot get image width/height"
	    }
	} else {
	    if {[string is integer $cam]} {
		# each device entry has three elements
		set name [lindex [uvc devices] [expr {$cam * 3}]]
	    } else {
		# assume cam is given as VID:PID:BUS.DEV
		set name $cam
	    }
	    if {[catch {
		uvc open $name [list [namespace current]::capture_uvc $w]
	    } dev]} {
		destroy $w
		return -code error "open camera: $dev"
	    }
	    array set fmts [uvc listformats $dev]
	    foreach i [lsort -integer [array names fmts]] {
		array set fmt $fmts($i)
		if {[scan $fmt(frame-size) "%dx%d" width height] == 2} {
		    if {$width == 640} {
			uvc format $dev $i 15
			break
		    }
		    if {$height == 480} {
			uvc format $dev $i 15
			break
		    }
		}
	    }
	    lassign [uvc format $dev] i fps
	    array set fmt $fmts($i)
	    if {[scan $fmt(frame-size) "%dx%d" width height] != 2} {
		uvc close $dev
		destroy $w
		return -code "cannot get image width/height"
	    }
	}
	if {$droid} {
	    set imgpad 0
	    set imgexpand 1
	    set imgrelief groove
	    $w configure -background black
	    wm attributes $w -fullscreen 1
	    bind $w <Key-Break> [list [namespace current]::done %W]
	    bind $w <<DidEnterBackground>> [list [namespace current]::pause %W]
	} else {
	    set width2 [expr {$width + 10}]
	    set imgpad 5
	    set imgexpand 0
	    set imgrelief sunken
	    wm geometry $w "${width2}x${width}"
	    bind $w <Escape> [list [namespace current]::done %W]
	}
	wm title $w "ZBar Tool"
	wm protocol $w WM_DELETE_WINDOW [list [namespace current]::done $w]
	set parent [winfo toplevel [winfo parent $w]]
	wm transient $w $parent
	set img [image create photo]
	set img2 [image create photo]
	label $w.img -image $img -width $width -height $height \
	    -relief $imgrelief -borderwidth 4
	pack $w.img -side top -expand $imgexpand -padx $imgpad -pady $imgpad
	$img configure -width $width -height $height
	label $w.data -height 6
	pack $w.data -side bottom -fill x -pady 20
	grab $w
	if {$droid} {
	    $w.data configure -background black -foreground red
	    bind $w <<ImageCapture>> \
		[list [namespace current]::capture_droid $w %x]
	    bind $w <Configure> \
		[list [namespace current]::rotate_droid $w $width $height]
	    set fset 0
	    bind $w.img <1> [list [namespace current]::start_stop $w]
	    borg camera start
	} elseif {$win} {
	    tk busy $parent
	    bind $w.img <1> [list [namespace current]::start_stop $w $dev]
	    wmf start $dev
	} else {
	    tk busy $parent
	    bind $w.img <1> [list [namespace current]::start_stop $w $dev]
	    uvc start $dev
	}
	focus $w
	set result ""
	set done 0
	vwait [namespace current]::done
	# watch out for Tk being gone
	catch {grab release $w}
	if {$droid} {
	    pause $w
	    borg camera close
	} elseif {$win} {
	    catch {tk busy forget $parent}
	    pause $w $dev
	    wmf close $dev
	} else {
	    catch {tk busy forget $parent}
	    pause $w $dev
	    uvc close $dev
	}
	catch {
	    destroy $w
	    image delete $img
	    image delete $img2
	}
	return $result
    }

    proc capture_droid {w flag} {
	variable img
	variable img2
	variable fset
	if {$flag} {
	    if {[catch {borg camera greyimage $img}]} {
		# Tk might be gone
		done $w
		return
	    }
	    if {![catch {
		zbar::async_decode $img \
		    [list [namespace current]::decoded_droid $w]
	    } err]} {
		$img2 copy $img -compositingrule set
	    }
	    if {!$fset} {
		# focus mode seems only to change on active capture
		borg camera parameters focus-mode continuous-picture
		set fset 1
	    }
	}
    }

    proc capture_wmf {w dev ind} {
	variable img
	variable img2
	if {$ind ne "capture"} {
	    return
	}
	if {[catch {wmf image $dev $img}]} {
	    # Tk might be gone
	    done $w
	    return
	}
	if {![catch {
	    zbar::async_decode $img \
		[list [namespace current]::decoded_wmf $w $dev]
	} err]} {
	    $img2 copy $img -compositingrule set
	}
    }

    proc capture_uvc {w dev} {
	variable img
	variable img2
	if {[catch {uvc image $dev $img}]} {
	    # Tk might be gone
	    done $w
	    return
	}
	if {![catch {
	    zbar::async_decode $img \
		[list [namespace current]::decoded_uvc $w $dev]
	} err]} {
	    $img2 copy $img -compositingrule set
	}
    }

    proc decoded {w time type data} {
	variable result
	set result [encoding convertfrom iso8859-1 $data]
	set pdata $result
	regsub -all {[[:cntrl:]]} $pdata " " pdata
	set prdata ""
	set linebrk 40
	set n 0
	set cont "\n"
	while {[string length $pdata]} {
	    incr n
	    if {$n > 6} {
		set cont "..."
	    }
	    append prdata [string range $pdata 0 ${linebrk}-1] $cont
	    set pdata [string range $pdata $linebrk end]
	    if {$n > 6} {
		break
	    }
	}
	$w.data configure -text $prdata
    }

    proc decoded_droid {w time type data} {
	variable img
	variable img2
	if {$type eq ""} {
	    return
	}
	if {[borg camera state] eq "capture"} {
	    borg camera stop
	    $img copy $img2 -compositingrule set
	    set show 1
	    borg vibrate 100
	    borg beep
	    tailcall decoded $w $time $type $data
	}
    }

    proc decoded_wmf {w dev time type data} {
	variable img
	variable img2
	if {$type eq ""} {
	    return
	}
	if {[wmf state $dev] eq "capture"} {
	    wmf stop $dev
	    $img copy $img2 -compositingrule set
	    tailcall decoded $w $time $type $data
	}
    }

    proc decoded_uvc {w dev time type data} {
	variable img
	variable img2
	if {$type eq ""} {
	    return
	}
	if {[uvc state $dev] eq "capture"} {
	    uvc stop $dev
	    $img copy $img2 -compositingrule set
	    tailcall decoded $w $time $type $data
	}
    }

    proc rotate_droid {w iw ih} {
	variable img
	variable fset
	borg camera stop
	borg camera orientation [dict get [borg displaymetrics] rotation]
	if {[borg camera orientation] % 180} {
	    $w.img configure -width $iw -height $ih
	} else {
	    $w.img configure -width $ih -height $iw
	}
	$img blank
	$img configure -width 1 -height 1
	$img configure -width 0 -height 0
	borg camera start
	set fset 0
	$w.data configure -text ""
	set result ""
    }

    proc start_stop {w {dev {}}} {
	variable result
	variable droid
	variable win
	variable fset
	set start 0
	if {$droid} {
	    if {[borg camera state] ne "capture"} {
		borg camera start
		set fset 0
		set start 1
	    } else {
		borg camera stop
	    }
	} elseif {$win} {
	    if {[wmf state $dev] ne "capture"} {
		wmf start $dev
		set start 1
	    } else {
		wmf stop $dev
	    }
	} else {
	    if {[uvc state $dev] ne "capture"} {
		uvc start $dev
		set start 1
	    } else {
		uvc stop $dev
	    }
	}
	if {$start} {
	    $w.data configure -text ""
	    set result ""
	}
    }

    proc pause {w {dev {}}} {
	variable droid
	variable win
	zbar::async_decode stop
	if {$droid} {
	    borg camera stop
	} elseif {$win} {
	    wmf stop $dev
	} else {
	    uvc stop $dev
	}
    }

    proc done {w} {
	variable done
	set done 1
    }

}

package provide zbartool 0.1
