# VCD is the Verilog Value Change Dump file format.
# It's often used in EDA tools to trace values over
# time and can be displayed with tools like GTKWave.
#
# This package can be used to generate VCD files.
# It is derived from https://wiki.tcl-lang.org/page/VCD
#
# This is a non OO version for Tcl < 8.6
#
# chw May 2023
#
# Example:
#
#  VCD create log test.vcd
#  log comment "Some comments from the user"
#  log date
#  log scope module values
#  log define_variable integer 32 var1
#  log define_variable wire 1 wire1
#  log define_variable real 64 rvar2
#  log define_variable string 1 svar3
#  log upscope
#  log timescale 100 ns
#  log enddefinitions
#  log simulation_time 10
#  log value var1 99
#  log value wire1 1
#  log simulation_time 20
#  log value svar3 "Hello world!"
#  log value rvar2 9.99
#  log value wire1 0
#  log simulation_time 30
#  log destroy

package provide vcdnooo 0.1
package provide vcd 0.1

namespace eval ::VCD {

    variable nowide
    set nowide [catch {string is wideinteger 1}]
    variable xblist [list \
	0 0000 1 0001 2 0010 3 0011 4 0100 5 0101 6 0110 7 0111 \
	8 1000 9 1001 a 1010 b 1011 c 1100 d 1101 e 1110 f 1111 \
    ]

    proc vtrace_handler {name reference varname args} {
	if {[lindex $args 0] ne {}} {
	    lassign $args elem
	    append varname ( $elem )
	}
	upvar 1 $varname var
	catch {simulation_time $name}
	if {![info exists elem] && [array exists var]} {
	    if {[catch {value $name $reference [set var()]} err]} {
		puts stderr $err
	    }
	    return
	}
	if {[catch {value $name $reference [set var]} err]} {
	    puts stderr $err
	}
    }

    proc ptrace_handler {name reference cmd args} {
	set op [lindex $args end]
	if {$op eq "enter"} {
	    set value 1
	} else {
	    set value 0
	}
	catch {simulation_time $name}
	if {[catch {value $name $reference $value} err]} {
	    puts stderr $err
	}
    }

    variable h		;# list of handles/names
    variable f		;# file channel array
    variable idc	;# id counters
    variable vtr	;# var traces
    variable ptr	;# proc traces
    variable cls	;# close flags for channels
    variable ids	;# name to identifier map
    variable typ	;# types of VCD vars
    variable wid	;# width of VCD vars
    variable clkcmd	;# microseconds or clicks

    set h {}
    array set f {}
    array set idc {}
    array set vtr {}
    array set ptr {}
    array set cls {}
    array set ids {}
    array set typ {}
    array set wid {}

    proc create {name first args} {
	variable h
	set idx [lsearch -exact $name $h]
	if {$idx >= 0} {
	    destroy $name
	}
	variable f
	variable idc
	variable vtr
	variable ptr
	variable clk0
	variable cls
	variable pos
	variable clkcmd
	set isc 0
	set pos($name) -1
	set idc($name) [list 32 32 32 32]
	set cls($name) 1
	if {$first eq "-channel"} {
	    set isc 1
	    foreach {chan c} $args {
		break
	    }
	    set f($name) $chan
	    if {$c eq "-close"} {
		set cls($name) 1
	    } else {
		set cls($name) 0
	    }
	} elseif {$first eq "-file"} {
	    set first [lindex $args 0]
	}
	if {!$isc} {
	    set f($name) [open $first w]
	}
	if {![info exists clkcmd]} {
	    set clkcmd microseconds
	    if {[catch {clock microseconds}]} {
		# assuming this gives microseconds
		set clkcmd clicks
	    }
	}
	set clk0($name) [clock $clkcmd]
	set cmd [subst {
	    proc [list $name] {cmd args} {
		eval [namespace current]::handler [list $name] \$cmd \$args
	    }
	}]
	namespace eval :: $cmd
	lappend h $name
    }

    proc destroy {name} {
	variable f
	variable vtr
	variable ptr
	variable cls
	variable pos
	variable idc
	variable ids
	variable typ
	variable wid
	foreach reference [array names vtr $name,*] {
	    trace remove variable $vtr($reference) write \
		[list ::VCD::vtrace_handler $name $reference]
	    unset vtr($reference)
	}
	foreach reference [array names ptr $name,*] {
	    trace remove execution $ptr($reference) {enter leave} \
		[list ::VCD::ptrace_handler $name $reference]
	    unset ptr($reference)
	}
	if {$cls($name)} {
	    close $f($name)
	}
	unset cls($name)
	unset f($name)
	unset pos($name)
	unset idc($name)
	foreach reference [array names ids $name,*] {
	    unset ids($reference)
	}
	foreach reference [array names typ $name,*] {
	    unset typ($reference)
	}
	foreach reference [array names wid $name,*] {
	    unset wid($reference)
	}
	variable h
	set idx [lsearch -exact $name $h]
	if {$idx >= 0} {
	    set h [lreplace $h $idx $idx]
	}
	set cmd [subst {catch {rename [list $name] {}}}]
	namespace eval :: $cmd
    }

    proc get_id {name} {
	variable idc
	foreach {c0 c1 c2 c3} $idc($name) {
	    break
	}
	incr c0
	if {$c0 >= 127} {
	    incr c1
	    set c0 33
	}
	if {$c1 >= 127} {
	    incr c2
	    set c1 33
	}
	if {$c2 >= 127} {
	    incr c3
	    set c2 33
	}
	if {$c3 >= 127} {
	    return -code error "Out of id's"
	}
	set id ""
	if {$c3 > 32} {
	    append id [format %c $c3]
	}
	if {$c2 > 32} {
	    append id [format %c $c2]
	}
	if {$c1 > 32} {
	    append id [format %c $c1]
	}
	if {$c0 > 32} {
	    append id [format %c $c0]
	}
	set idc($name) [list $c0 $c1 $c2 $c3]
	return $id
    }

    proc comment {name txt} {
	variable f
	puts $f($name) "\$comment\n$txt\n\$end"
    }

    proc date {name} {
	variable f
	puts $f($name) "\$date [clock format [clock seconds]] \$end"
    }

    proc timescale {name number scale} {
	variable f
	puts $f($name) "\$timescale $number $scale \$end"
    }

    proc define_variable {name type width reference} {
	variable ids
	variable typ
	variable wid
	variable f
	set ids($name,$reference) [get_id $name]
	set typ($name,$reference) $type
	if {$type eq "real"} {
	    set width 64
	} elseif {$type eq "string"} {
	    set width 1
	}
	set wid($name,$reference) $width
	# no colons in reference for gtkwave
	set refnc [string map {:: _ : _} $reference]
	puts $f($name) \
	    "\$var $type $width $ids($name,$reference) $refnc \$end"
    }

    proc enddefinitions {name} {
	variable f
	variable pos
	puts $f($name) "\$enddefinitions \$end"
	catch {flush $f($name)}
	if {[catch {tell $f} pos($name)]} {
	    set pos($name) -1
	}
    }

    proc scope {name type identifier} {
	variable f
	puts $f($name) "\$scope $type $identifier \$end"
    }

    proc upscope {name} {
	variable f
	puts $f($name) "\$upscope \$end"
    }

    proc version {name txt} {
	variable f
	puts $f($name) "\$version\n$txt\n\$end"
    }

    proc simulation_time {name {time {}}} {
	variable f
	variable clk0
	variable clkcmd
	if {$time eq {}} {
	    set time [expr {[clock $clkcmd] - $clk0($name)}]
	}
	puts $f($name) "\#$time"
    }

    proc flush {name} {
	variable f
	flush $f($name)
    }

    proc restart {name} {
	variable f
	variable pos
	variable clk0
	variable clkcmd
	if {$pos($name) >= 0} {
	    catch {flush $f($name)}
	    if {![catch {chan truncate $f($name) $pos($name)}]} {
		catch {seek $f($name) 0 end}
		set clk0($name) [clock $clkcmd]
	    }
	}
    }

    proc value {name reference value} {
	variable ids
	variable typ
	variable wid
	variable f
	variable nowide
	variable xblist
	switch -exact -- $typ($name,$reference) {
	    "integer" {
		if {[string is integer -strict $value]} {
		    if {$value == 0} {
			puts $f($name) "b0 $ids($name,$reference)"
		    } else {
			set v [string map $xblist [format %08x $value]]
			puts $f($name) "b[string trimleft $v 0] $ids($name,$reference)"
		    }
		} elseif {!$nowide && [string is wideinteger -strict $value]} {
		    if {$value == 0} {
			puts $f($name) "b0 $ids($name,$reference)"
		    } else {
			set v [string map $xblist [format %016lx $value]]
			puts $f($name) "b[string trimleft $v 0] $ids($name,$reference)"
		    }
		} else {
		    return -code error error "Value '$value' for reference '$reference' is not an integer"
		}
	    }
	    "real" {
		if {![string is double -strict $value]} {
		    return -code error error "Value '$value' for reference '$reference' is not a double"
		}
		puts $f($name) "r[format %.16g $value] $ids($name,$reference)"
	    }
	    "string" {
		set clist [split $value {}]
		set vlist {}
		foreach c [lrange $clist 0 79] {
		    scan $c %c v
		    if {$v <= 0x20 || $v > 0x7e || $v == 0x25} {
			lappend vlist [format %%%02x $v]
		    } else {
			lappend vlist $c
		    }
		}
		if {[llength $clist] > 80} {
		    lappend vlist . . .
		}
		puts $f($name) "s[join $vlist {}] $ids($name,$reference)"
	    }
	    default {
		if {$value != 0 && $value != 1 &&
		    $value != "x" && $value != "z"} {
		    return -code error error "Value '$value' for reference '$reference' must be 0, 1, x or z"
		}
		puts $f($name) "$value$ids($name,$reference)"
	    }
	}
    }

    proc handler {name cmd args} {
	eval [list $cmd] [list $name] $args
    }

    proc var_trace {name reference {var {}}} {
	variable ids
	variable vtr
	if {![info exists ids($name,$reference)]} {
	    return -code error error "Reference '$reference' not found"
	}
	if {$var eq {}} {
	    set var $reference
	}
	if {[string range $var 0 1] ne "::"} {
	    set var ::$var
	}
	trace remove variable $var write \
	    [list [namespace current]::vtrace_handler $name $reference]
	trace add variable $var write \
	    [list [namespace current]::vtrace_handler $name $reference]
	set vtr($name,$reference) $var
	if {[info exists $var]} {
	    simulation_time $name
	    value $name $reference [set $var]
	}
    }

    proc proc_trace {name reference {proc {}}} {
	variable ids
	variable ptr
	if {![info exists ids($name,$reference)]} {
	    return -code error error "Reference '$reference' not found"
	}
	if {$proc eq {}} {
	    set proc $reference
	}
	trace remove execution $proc {enter leave} \
	    [list [namespace current]::ptrace_handler $name $reference]
	trace add execution $proc {enter leave} \
	    [list [namespace current]::ptrace_handler $name $reference]
	set ptr($name,$reference) $proc
    }

}

proc ::VCD {cmd args} {
    eval ::VCD::$cmd $args
}
