# VCD is the Verilog Value Change Dump file format.
# It's often used in EDA tools to trace values over
# time and can be displayed with tools like GTKWave.
#
# This package can be used to generate VCD files.
# It is derived from https://wiki.tcl-lang.org/page/VCD
#
# chw April 2021
#
# Example:
#
#  VCD create log test.vcd
#  log comment "Some comments from the user"
#  log date
#  log scope module values
#  log define_variable integer 32 var1
#  log define_variable wire 1 wire1
#  log define_variable real 64 rvar2
#  log define_variable string 1 svar3
#  log upscope
#  log timescale 100 ns
#  log enddefinitions
#  log simulation_time 10
#  log value var1 99
#  log value wire1 1
#  log simulation_time 20
#  log value svar3 "Hello world!"
#  log value rvar2 9.99
#  log value wire1 0
#  log simulation_time 30
#  log destroy

package require TclOO
package provide vcd 0.1

namespace eval ::VCD {
    proc vtrace_handler {obj reference varname args} {
	if {[lindex $args 0] ne {}} {
	    lassign $args elem
	    append varname ( $elem )
	}
	upvar 1 $varname var
	catch {$obj simulation_time}
	if {![info exists elem] && [array exists var]} {
	    if {[catch {$obj value $reference [set var()]} err]} {
		puts stderr $err
	    }
	    return
	}
	if {[catch {$obj value $reference [set var]} err]} {
	    puts stderr $err
	}
    }
    proc ptrace_handler {obj reference cmd args} {
	set op [lindex $args end]
	if {$op eq "enter"} {
	    set value 1
	} else {
	    set value 0
	}
	catch {$obj simulation_time}
	if {[catch {$obj value $reference $value} err]} {
	    puts stderr $err
	}
    }
}

oo::class create VCD {
    constructor {first args} {
	variable f
	variable c0 32
	variable c1 32
	variable c2 32
	variable c3 32
	variable vtr
	variable ptr
	variable clk0
	variable isc
	variable cls
	variable pos
	set isc 0
	set pos -1
	set cls 1
	if {$first eq "-channel"} {
	    set isc 1
	    lassign $args f cls
	    if {$cls eq "-close"} {
		set cls 1
	    } else {
		set cls 0
	    }
	} elseif {$first eq "-file"} {
	    lassign $args first
	}
	if {!$isc} {
	    set f [open $first w]
	}
	array set vtr {}
	array set ptr {}
	set clk0 [clock microseconds]
    }
    destructor {
	variable f
	variable vtr
	variable ptr
	variable cls
	foreach reference [array names vtr] {
	    trace remove variable $vtr($reference) write \
		[list ::VCD::vtrace_handler [self] $reference]
	}
	foreach reference [array names ptr] {
	    trace remove execution $ptr($reference) {enter leave} \
		[list ::VCD::ptrace_handler [self] $reference]
	}
	if {$cls} {
	    close $f
	}
    }
    method get_id {} {
	variable c0
	variable c1
	variable c2
	variable c3
	incr c0
	if {$c0 >= 127} {
	    incr c1
	    set c0 33
	}
	if {$c1 >= 127} {
	    incr c2
	    set c1 33
	}
	if {$c2 >= 127} {
	    incr c3
	    set c2 33
	}
	if {$c3 >= 127} {
	    return -code error "Out of id's"
	}
	set id ""
	if {$c3 > 32} {
	    append id [format %c $c3]
	}
	if {$c2 > 32} {
	    append id [format %c $c2]
	}
	if {$c1 > 32} {
	    append id [format %c $c1]
	}
	if {$c0 > 32} {
	    append id [format %c $c0]
	}
	return $id
    }
    method comment {txt} {
	variable f
	puts $f "\$comment\n$txt\n\$end"
    }
    method date {} {
	variable f
	puts $f "\$date [clock format [clock seconds]] \$end"
    }
    method timescale {number scale} {
	variable f
	puts $f "\$timescale $number $scale \$end"
    }
    method define_variable {type width reference} {
	variable ids
	variable typ
	variable wid
	variable f
	set ids($reference) [my get_id]
	set typ($reference) $type
	if {$type eq "real"} {
	    set width 64
	} elseif {$type eq "string"} {
	    set width 1
	}
	set wid($reference) $width
	# no colons in reference for gtkwave
	set refnc [string map {:: _ : _} $reference]
	puts $f "\$var $type $width $ids($reference) $refnc \$end"
    }
    method enddefinitions {} {
	variable f
	variable pos
	puts $f "\$enddefinitions \$end"
	catch {flush $f}
	if {[catch {tell $f} pos]} {
	    set pos -1
	}
    }
    method scope {type identifier} {
	variable f
	puts $f "\$scope $type $identifier \$end"
    }
    method upscope { } {
	variable f
	puts $f "\$upscope \$end"
    }
    method version {txt} {
	variable f
	puts $f "\$version\n$txt\n\$end"
    }
    method simulation_time {{time {}}} {
	variable f
	variable clk0
	if {$time eq {}} {
	    set time [expr {[clock microseconds] - $clk0}]
	}
	puts $f "\#$time"
    }
    method flush {} {
	variable f
	flush $f
    }
    method restart {} {
	variable f
	variable pos
	variable clk0
	if {$pos >= 0} {
	    catch {flush $f}
	    if {![catch {chan truncate $f $pos}]} {
		catch {seek $f 0 end}
		set clk0 [clock microseconds]
	    }
	}
    }
    method value {reference value} {
	variable ids
	variable typ
	variable wid
	variable f
	switch -exact -- $typ($reference) {
	    "integer" {
		if {[string is integer -strict $value]} {
		    if {$value == 0} {
			puts $f "b0 $ids($reference)"
		    } else {
			puts $f "b[string trimleft [format %b $value] 0] $ids($reference)"
		    }
		} elseif {[string is wideinteger -strict $value]} {
		    if {$value == 0} {
			puts $f "b0 $ids($reference)"
		    } else {
			puts $f "b[string trimleft [format %lb $value] 0] $ids($reference)"
		    }
		}
		return -code error "Value '$value' for reference '$reference' is not an integer"
	    }
	    "real" {
		if {![string is double -strict $value]} {
		    return -code error "Value '$value' for reference '$reference' is not a double"
		}
		puts $f "r[format %.16g $value] $ids($reference)"
	    }
	    "string" {
		set clist [split $value {}]
		set vlist {}
		foreach c [lrange $clist 0 79] {
		    scan $c %c v
		    if {$v <= 0x20 || $v > 0x7e || $v == 0x25} {
			lappend vlist [format %%%02x $v]
		    } else {
			lappend vlist $c
		    }
		}
		if {[llength $clist] > 80} {
		    lappend vlist . . .
		}
		puts $f "s[join $vlist {}] $ids($reference)"
	    }
	    default {
		if {$value ni {0 1 x z}} {
		    return -code error "Value '$value' for reference '$reference' must be 0, 1, x or z"
		}
		puts $f "$value$ids($reference)"
	    }
	}
    }
    method var_trace {reference {var {}}} {
	variable ids
	variable vtr
	if {![info exists ids($reference)]} {
	    return -code error "Reference '$reference' not found"
	}
	if {$var eq {}} {
	    set var $reference
	}
	if {[string range $var 0 1] ne "::"} {
	    set var ::$var
	}
	trace remove variable $var write \
	    [list ::VCD::vtrace_handler [self] $reference]
	trace add variable $var write \
	    [list ::VCD::vtrace_handler [self] $reference]
	set vtr($reference) $var
	if {[info exists $var]} {
	    my simulation_time
	    my value $reference [set $var]
	}
    }
    method proc_trace {reference {proc {}}} {
	variable ids
	variable ptr
	if {![info exists ids($reference)]} {
	    return -code error "Reference '$reference' not found"
	}
	if {$proc eq {}} {
	    set proc $reference
	}
	trace remove execution $proc {enter leave} \
	    [list ::VCD::ptrace_handler [self] $reference]
	trace add execution $proc {enter leave} \
	    [list ::VCD::ptrace_handler [self] $reference]
	set ptr($reference) $proc
    }
}
