#!/usr/bin/env tclsh
package require Tk

# Create a treeview style without an indicator to use as an advanced listbox
ttk::style configure Listbox.Treeview.Item -padding {2 0}
ttk::style layout Listbox.Treeview.Item {
    Treeitem.padding -sticky nswe -children {
        Treeitem.image -side left -sticky {}
        Treeitem.focus -side left -sticky {} -children {
            Treeitem.text -side left -sticky {}
        }
    }
}

package require upnp
package require didl

set dir [file dirname [info script]]

proc yieldm {{value ""}} {
    yieldto return -level 0 $value
}

proc main {} {
    wm title . "Media browser"
    ttk::treeview .tv -style Listbox.Treeview -height 16 \
      -yscrollcommand {.vs set}
    ttk::scrollbar .vs -command {.tv yview}
    .tv column #0 -width 500
    grid .tv .vs -sticky news
    grid columnconfigure . .tv -weight 1
    grid rowconfigure . .tv -weight 1
    icons .tv
    set urn urn:schemas-upnp-org:device:MediaServer:1

    set me [info coroutine]
    bind .tv <<TreeviewSelect>> [list $me tvselect]

    bind . <<NetworkInfo>> {ssdp rejoin}
    bind . <<TetherInfo>>  {ssdp rejoin}

    while 1 {
	.tv delete [.tv children {}]
	.tv heading #0 -text "Select a device"
	upnp discover $urn [list $me discover]
	while 1 {
	    set args [lassign [yieldm] cmd]
	    if {$cmd eq "discover"} {
		lassign $args ns
		namespace upvar $ns info info icon icon
		incr dev
		# Run in a separate coroutine so callbacks don't get mixed up
		set img [coroutine devicon$dev devicon .tv $icon]
		set id [.tv insert {} end -text [dict get $info friendlyName] \
		  -values $ns -tag ctrlpt -image $img]
		trace add variable ${ns}::info unset [list $me unset $ns $id]
	    } elseif {$cmd eq "unset"} {
		lassign $args ns id
		catch {.tv delete $id}
	    } elseif {$cmd eq "tvselect"} {
		break
	    }
	}
	# upnp discover $urn {}
	device [lindex [.tv selection] 0]
    }
}

proc device {item} {
    .tv heading #0 -text "Browse files"
    lassign [.tv item $item -values] ns
    set id 0
    while 1 {
	if {[.tv item $item -text] ne ".."} {
	    lappend path $id
	} elseif {[llength $path] > 1} {
	    set path [lreplace $path end end]
	} else {
	    return
	}
	set result [namespace eval $ns \
	  [list ContentDirectory::Browse $id BrowseDirectChildren * 0 100 ""]]
	set didl [didl parse [dict get $result Result]]
	.tv delete [.tv children {}]
	.tv insert {} end -text .. \
	  -values [lrange $path end-1 end-1] -tag object.container
	foreach tag {container item} {
	    if {[dict exists $didl $tag]} {
		foreach item [dict get $didl $tag] {
		    set id [dict get $item attributes id]
		    set values [list $id]
		    if {[dict exists $item contents res]} {
			foreach n [dict get $item contents res] {
			    lappend values [dict get $n value]
			}
		    }
		    set dict [dict get $item contents]
		    .tv insert {} end -text [dict get $dict dc:title] \
		      -tag [tag .tv [dict get $dict upnp:class]] \
		      -values $values
		}
	    }
	}
	while 1 {
	    set args [lassign [yieldm] cmd]
	    if {$cmd eq "tvselect"} {
		set item [lindex [.tv selection] 0]
		lassign [.tv item $item -values] id url
		if {$url eq ""} break
		coroutine media show $url
	    }
	    if {$cmd eq "unset"} {
		lassign $args uuid
		if {$uuid eq $ns} {
		    # The device we were browsing has gone away
		    return
		}
	    }
	}
    }
}

proc icons {w} {
    global dir
    set path [file join $dir icons 16x16]
    foreach {tag file} {
	object.container			folder.png
	object.container.album.photoAlbum	emblem-photos.png
	object.container.storageFolder		drive-harddisk.png
	object.item				text-x-generic.png
	object.item.imageItem.photo		image-x-generic.png
	object.item.videoItem			applications-multimedia.png
	object.item.audioItem.musicTrack	audio-x-generic.png
    } {
	set ls [glob -nocomplain -dir $path */$file]
	if {[llength $ls]} {
	    set name [file rootname $file]
	    set img [image create photo ::icon::$name \
	      -file [lindex $ls 0] -width 20]
	    $w tag configure $tag -image $img
	}
    }
}

proc devicon {tv dict} {
    # Specs say: At least one icon should be of type “image/png”
    if {![dict exists $dict image/png]} return
    # Check all png images
    foreach n [dict keys [dict get $dict image/png]] {
	scan $n %dx%dx%d w h d
	if {$w % 16 == 0 && $w == $h} {
	    set factor [expr {$w / 16}]
	    set url [dict get $dict image/png $n]
	    http::geturl $url -command [info coroutine]
	    # Create an empty image for the selected icon candidate
	    set img [image create photo -width 20 -height 16]
	    # Pass the image name to the caller, then wait for the image data
	    set tok [yield $img]
	    # Create a temporary image from the receive image data
	    set tmp [image create photo -data [http::data $tok]]
	    # Shrink the image to 16x16
	    $img copy $tmp -subsample $factor
	    # Cleanup some resources that are no longer needed
	    http::cleanup $tok
	    image delete $tmp
	    # Inform the treeview widget that it should update its display
	    event generate $tv <Configure>
	    # No need to look any further
	    return
	}
    }
    return
}

proc tag {w tag} {
    set tags [$w tag names]
    while {$tag ni $tags} {
	set x [string last . $tag]
	if {$x < 0} {return object.unknown}
	set tag [string replace $tag $x end]
    }
    return $tag
}

proc show {url} {
    try {
	http::geturl $url -command [info coroutine]
	set tok [yield]
	if {[winfo exists .img]} {
	    set old [.img.label cget -image]
	} else {
	    set old ""
	}
	if {[catch {image create photo -data [http::data $tok]} img]} {
	    set img $old
	} elseif {$old ne ""} {
	    image delete $old
	    .img.label configure -image $img
	} else {
	    toplevel .img
	    wm title .img "Image viewer"
	    label .img.label -image $img
	    pack .img.label
	    wm resizable .img 0 0
	}
	http::cleanup $tok
	if {$img ne ""} {
	    bind .img.label <Destroy> [info coroutine]
	    yield
	    image delete $img
	}
    } on error {err info} {
	puts [dict get $info -errorinfo]
    }
}

coroutine browse try {
    main
} on error {err info} {
    puts [dict get $info -errorinfo]
}
