# vfs.tcl --
#
# Mapping of OPCUA variables (read-only) and method calls using Tcl VFS.
#
# Copyright (c) 2022-2023 Christian Werner <chw at ch minus werner dot de>
#
# See the file "license.terms" for information on usage and redistribution of
# this file, and for a DISCLAIMER OF ALL WARRANTIES.

package require topcua
package require vfs

package provide vfs::opcua 0.1

namespace eval vfs::opcua {
    variable T	;# array indexed by brpath, values are { nodeid clspath }
    variable R	;# reverse of T, indexed by nodeid, values are brpath
    variable D	;# data cache, array indexed by nodeid, values are
		;# OPCUA variables' Value attributes
    variable U	;# array of URLs for reconnect indexed by client handle
    variable M	;# array for memchans

    array set T {}
    array set R {}
    array set D {}
    array set U {}
    array set M {}

    proc _connect {C url} {
	variable T
	variable R
	variable U
	set U($C) $url
	::opcua connect $C $url
	catch {::opcua gentypes $C}
	# omit "/Root" and namespace prefixes in method names
	catch {::opcua genstubs $C /Root/ {{/[1-9][0-9]*:} {/}}}
	set root [::opcua root]
	set tree [::opcua ptree $C]
	# omit "/" and "/Root" prefixes in brpath
	foreach {brpath nodeid clspath refid typeid parent} $tree {
	    if {$nodeid eq $root} {
		continue
	    }
	    set brpath [string trimleft $brpath /]
	    regsub -all -- {^Root/} $brpath {} brpath
	    set short $brpath
	    regsub -all -- {/[1-9][0-9]*:} $short {/} short
	    incr t($short)
	}
	foreach {brpath nodeid clspath refid typeid parent} $tree {
	    if {$nodeid eq $root} {
		continue
	    }
	    set brpath [string trimleft $brpath /]
	    regsub -all -- {^Root/} $brpath {} brpath
	    set short $brpath
	    regsub -all -- {/[1-9][0-9]*:} $short {/} short
	    if {$t($short) == 1} {
		set brpath $short
	    }
	    set T($C,$brpath) [list $nodeid $clspath]
	    set R($C,$nodeid) $brpath
	}
    }

    proc _disconnect {C} {
	variable T
	variable R
	variable D
	variable U
	::opcua disconnect $C
	array unset T $C,*
	array unset R $C,*
	array unset D $C,*
	unset U($C)
    }

    proc Mount {url local} {
	variable T
	variable R
	variable U
	set urlc $url
	if {[string first opc.tcp:// $urlc] != 0} {
	    set index [string first : $urlc]
	    if {$index >= 0} {
		incr index
		set urlc [string trimleft [string range $urlc $index end] /]
	    }
	    set urlc opc.tcp://$urlc
	}
	set C [::opcua new]
	if {![catch {vfs::filesystem info $url}]} {
	    vfs::unmount $url
	}
	if {[file pathtype $local] ne "absolute"} {
	    set local [file normalize $local]
	}
	vfs::filesystem mount $local [list [namespace current]::handler $C]
	vfs::RegisterMount $local [list [namespace current]::Unmount $C]
	_connect $C $urlc
	return $C
    }

    proc _readvar {C nodeid} {
	variable U
	foreach attempt {0 1} {
	    if {![catch {::opcua read $C $nodeid} val]} {
		return $val
	    }
	    if {$attempt < 1} {
		switch -- [lindex $::errorCode 3] {
		    BadSessionIdInvalid -
		    BadConnectionClosed {
			# try to reconnect
			set url $U($C)
			catch {_disconnect $C}
			catch {_connect $C $url}
		    }
		}
	    }
	}
	return -code error $val
    }

    proc Unmount {C local} {
	if {[file pathtype $local] ne "absolute"} {
	    set local [file normalize $local]
	}
	vfs::filesystem unmount $local
	_disconnect $C
	::opcua destroy $C
    }

    proc handler {C cmd root relative actualpath args} {
	if {$cmd eq "matchindirectory"} {
	    [namespace current]::$cmd $C $relative $actualpath {*}$args
	} else {
	    [namespace current]::$cmd $C $relative {*}$args
	}
    }

    proc attributes {C} {
	return [list "state"]
    }

    proc state {C args} {
	vfs::attributeCantConfigure "state" "readonly" $args
    }

    proc _getdir {C path actualpath {pattern *}} {
	variable R
	variable T
	if {$path eq "." || $path eq ""} {
	    set path ""
	}
	if {$pattern eq ""} {
	    if {[info exists T($C,$path)]} {
		return [list $path]
	    }
	    return [list]
	}
	set res [list]
	if {$path eq ""} {
	    set sep /
	    set strip 0
	    set depth 1
	} elseif {[info exists T($C,$path)]} {
	    set sep ""
	    set strip [string length $path]
	    set depth [llength [file split $path]]
	    incr depth 1
	}
	if {[info exists depth]} {
	    foreach name [array names R $C,*] {
		if {$strip && [string first $path $R($name)] != 0} {
		    continue
		}
		set flist [file split $R($name)]
		if {[llength $flist] != $depth} {
		    continue
		}
		if {[string match $pattern [lindex $flist end]]} {
		    lappend res \
			$actualpath$sep[string range $R($name) $strip end]
		}
	    }
	}
	return $res
    }

    proc matchindirectory {C path actualpath pattern type} {
	variable T
	set res [_getdir $C $path $actualpath $pattern]
	if {![string length $pattern]} {
	    if {![info exists T($C,$path)]} {
		return {}
	    }
	    set res [list $actualpath]
	}
	set actualpath ""
	::vfs::matchCorrectTypes $type $res $actualpath
    }

    proc stat {C name} {
	variable T
	variable D
	if {$name eq ""} {
	    return [list type directory mtime 0 size 0 mode 0555 ino -1 \
			depth 0 name "" dev -1 uid -1 gid -1 nlink 1]
	}
	if {[info exists T($C,$name)]} {
	    lassign $T($C,$name) nodeid clspath
	    if {[string match "*/Variable" $clspath]} {
		if {![info exists D($C,$nodeid)]} {
		    if {[catch {set D($C,$nodeid) [_readvar $C $nodeid]}]} {
			vfs::filesystem posixerror $::vfs::posix(EIO)
		    }
		}
		return [list type file mtime 0 mode 0444 ino -1 \
			    size [string length $D($C,$nodeid)] \
			    atime 0 ctime 0]
	    } elseif {[string match "*/Method" $clspath]} {
		return [list type file mtime 0 mode 0666 ino -1 \
			    size 0 atime 0 ctime 0]
	    }
	    return [list type directory mtime 0 size 0 mode 0555 ino -1 \
			depth 0 name $name dev -1 uid -1 gid -1 nlink 1]
	}
	vfs::filesystem posixerror $::vfs::posix(ENOENT)
    }

    proc access {C name mode} {
	variable T
	if {$name eq {} && !($mode & 2)} {
	    return 1
	}
	if {[info exists T($C,$name)]} {
	    lassign $T($C,$name) nodeid clspath
	    if {[string match "*/Variable" $clspath]} {
		if {$mode & 2} {
		    vfs::filesystem posixerror $::vfs::posix(EACCES)
		}
		return 1
	    }
	    if {[string match "*/Method" $clspath]} {
		return 1
	    }
	    if {$mode & 2} {
		vfs::filesystem posixerror $::vfs::posix(EACCES)
	    }
	    return 1
	}
	vfs::filesystem posixerror $::vfs::posix(ENOENT)
    }

    proc open {C name mode permission} {
	variable T
	variable D
	if {![info exists T($C,$name)]} {
	    vfs::filesystem posixerror $::vfs::posix(ENOENT)
	}
	switch -glob -- $mode {
	    "" - "r" {
		lassign $T($C,$name) nodeid clspath
		if {[string match "*/Method" $clspath]} {
		    vfs::filesystem posixerror $::vfs::posix(EACCES)
		}
		if {![string match "*/Variable" $clspath]} {
		    vfs::filesystem posixerror $::vfs::posix(EISDIR)
		}
		if {[catch {set D($C,$nodeid) [_readvar $C $nodeid]}]} {
		    vfs::filesystem posixerror $::vfs::posix(EACCES)
		}
		return [list [_memchan $C $nodeid 0 $D($C,$nodeid)]]
	    }
	    "w*" {
		lassign $T($C,$name) nodeid clspath
		if {[string match "*/Variable" $clspath]} {
		    vfs::filesystem posixerror $::vfs::posix(EROFS)
		}
		if {![string match "*/Method" $clspath]} {
		    vfs::filesystem posixerror $::vfs::posix(EISDIR)
		}
		return [list [_memchan $C $nodeid 1]]
	    }
	    default {
		vfs::filesystem posixerror $::vfs::posix(EROFS)
	    }
	}
    }

    proc createdirectory {C name} {
	vfs::filesystem posixerror $::vfs::posix(EROFS)
    }

    proc removedirectory {C name recursive} {
	vfs::filesystem posixerror $::vfs::posix(EROFS)
    }

    proc deletefile {C name} {
	vfs::filesystem posixerror $::vfs::posix(EROFS)
    }

    proc fileattributes {C name args} {
	switch -- [llength $args] {
	    0 {
		# list strings
		return [list]
	    }
	    1 {
		# get value
		return ""
	    }
	    2 {
		# set value
		vfs::filesystem posixerror $::vfs::posix(EROFS)
	    }
	}
    }

    proc utime {C path actime mtime} {
	vfs::filesystem posixerror $::vfs::posix(EROFS)
    }

    # Memory backed channel constructor

    proc _memchan {C nodeid ismeth {data {}}} {
	variable M
	set chan [chan creat {read write} [namespace origin _memchan_handler]]
	set M($chan,C) $C
	set M($chan,nodeid) $nodeid
	set M($chan,ismeth) $ismeth
	set M($chan,buf) $data
	return $chan
    }

    # A seek operation which set the file pointer to offset 0
    # triggers another read or method call.

    proc _memchan_handler {cmd chan args} {
	variable M
	variable R
	variable D
	switch -exact -- $cmd {
	    initialize {
		lassign $args mode
		set M($chan,pos) 0
		if {![info exists M(timer)]} {
		    set M(timer) {}
		}
		return {
		    initialize finalize watch read write seek
		    cget cgetall configure truncate
		}
	    }
	    finalize {
		unset -nocomplain M($chan,buf) M($chan,pos)
		unset -nocomplain M($chan,C) M($chan,nodeid) M($chan,ismeth)
		foreach event {read write} {
		    if {[info exists M($event,watch)]} {
			set idx [lsearch -exact M($event,watch) $chan]
			if {$idx >= 0} {
			    set M($event,watch) \
				[lreplace $M($event,watch) $idx $idx]
			}
		    }
		}
	    }
	    seek {
		lassign $args offset base
		switch -exact -- $base {
		    1 - current {
			incr offset $M($chan,pos)
		    }
		    2 - end {
			incr offset [string length $M($chan,buf)]
		    }
		}
		if {$offset < 0} {
		    return -code error \
			"error during seek on \"$chan\": invalid argument"
		} elseif {$offset > [string length $M($chan,buf)]} {
		    set extend [expr {$offset - [string length $M($chan,buf)]}]
		    append buf [binary format @$extend]
		}
		set M($chan,pos) $offset
		if {($M($chan,pos) == 0)} {
		    set eio 0
		    set C $M($chan,C)
		    set nodeid $M($chan,nodeid)
		    if {$M($chan,ismeth)} {
			set meth ::opcua::${C}::$R($C,$nodeid)
			if {[catch {info args $meth} input]} {
			    return -code error \
				"error during seek on \"$chan\": not a method"
			}
			set D($C,$nodeid) {}
			if {[llength $input] > 1} {
			    if {[catch {
				set D($C,$nodeid) [$meth {*}$M($chan,buf)]
			    }]} {
				incr eio
			    }
			} elseif {[llength $input] == 0} {
			    if {[catch {
				set D($C,$nodeid) [$meth]
			    }]} {
				incr eio
			    }
			} elseif {[catch {
			    set D($C,$nodeid) [$meth $M($chan,buf)]
			}]} {
			    incr eio
			}
		    } else {
			if {[catch {
			    set D($C,$nodeid) [_readvar $C $nodeid]
			}]} {
			    incr eio
			}
		    }
		    if {$eio} {
			return -code error \
			    "error during seek on \"$chan\": I/O error"
		    } else {
			set M($chan,buf) $D($C,$nodeid)
		    }
		}
		return $M($chan,pos)
	    }
	    read {
		lassign $args count
		set ret [string range $M($chan,buf) $M($chan,pos) \
			   [expr {$M($chan,pos) + $count - 1}]]
		incr M($chan,pos) [string length $ret]
		return $ret
	    }
	    write {
		lassign $args data
		set count [string length $data]
		if {$M($chan,pos) >= [string length $M($chan,buf)]} {
		    append M($chan,buf) $data
		} else {
		    set last [expr {$M($chan,pos) + $count - 1}]
		    set M($chan,buf) \
			[string replace $M($chan,buf) $M($chan,pos) $last $data]
		}
		incr M($chan,pos) $count
		return $count
	    }
	    cget {
		lassign $args option
		switch -exact -- $option {
		    -length {
			return [string length $M($chan,buf)]
		    }
		    -allocated {
			return [string length $M($chan,buf)]
		    }
		    -clear {
			if {$M($chan,buf) eq {}} {
			    return 1
			}
			return 0
		    }
		    default {
			return -code error "bad option \"$option\":\
should be one of -blocking, -buffering, -buffersize, -encoding,\
-eofchar, -translation, -length, -allocated, or -clear"
		    }
		}
	    }
	    cgetall {
		set len [string length $M($chan,buf)]
		set clr [expr {$len == 0}]
		return [list -length $len -allocated $len -clear $clr]
	    }
	    configure {
		lassign $args option value
		switch -exact -- $option {
		    -length {
		    }
		    -allocated {
		    }
		    -clear {
			# use -clear 1 before writing arguments
			# for next method call
			if {$value} {
			    set M($chan,buf) {}
			    set M($chan,pos) 0
			}
		    }
		    default {
			return -code error "bad option \"$option\":\
should be one of -blocking, -buffering, -buffersize, -encoding,\
-eofchar, -translation, -length, -allocated, or -clear"
		    }
		}
	    }
	    watch {
		lassign $args eventspec
		after cancel $M(timer)
		foreach event {read write} {
		    if {![info exists M($event,watch)]} {
			set M($event,watch) {}
		    }
		    set idx [lsearch -exact $M($event,watch) $chan]
		    if {$event in $eventspec} {
			if {$idx == -1} {
			    lappend M($event,watch) $chan
			}
		    } elseif {$idx != -1} {
			set M($event,watch) \
			    [lreplace $M($event,watch) $idx $idx]
		    }
		}
		set M(timer) [after 10 [list ::vfs::opcua::_memchan_timer]]
	    }
	    truncate {
		lassign $args length
		if {$length < 0} {
		    return -code error \
			"error during truncate on \"$chan\": invalid argument"
		} elseif {$length > [string length $M($chan,buf)]} {
		    set extend [expr {$length - [string length $M($chan,buf)]}]
		    append buf [binary format @$extend]
		} else {
		    set M($chan,buf) [string range $M($chan,buf) 0 $length-1]
		}
		set length [string length $M($chan,buf)]
		if {$M($chan,pos) > $length} {
		    set M($chan,pos) $length
		}
	    }
	}
    }

    # memchan channels are always writable and always readable

    proc _memchan_timer {} {
	variable M
	set more 0
	foreach event {read write} {
	    incr more [llength $M($event,watch)]
	    foreach chan $M($event,watch) {
		chan postevent $chan $event
	    }
	}
	if {$more > 0} {
	    set M(timer) [after 10 [info level 0]]
	}
    }
}
