# topcua.tcl --
#
# Library functions of a proof of concept Tcl binding to the
# open62541 OPC UA library (client and server).
#
# Copyright (c) 2018-2024 Christian Werner <chw at ch minus werner dot de>
#
# See the file "license.terms" for information on usage and redistribution of
# this file, and for a DISCLAIMER OF ALL WARRANTIES.

#############################################################################
# Common utility prodecures.

namespace eval ::opcua {

    # Internal helper wrapping translate of hierarchical path starting
    # at root object, elements separated with slashes. To be used
    # when path is known to have no embedded slashes in elements.
    # Returns the nodeid only.

    proc _hxlate {handle path} {
	set elems [root]
	foreach e [split $path "/"] {
	    lappend elems "/" $e
	}
	return [lindex [translate $handle {*}$elems] 0]
    }

    # Internal helper for quoting strings for XML

    proc _xmlquote {str} {
	return [string map {& &amp; < &lt; > &gt; "\"" &quot;} $str]
    }

    # Internal helper performing chunked "mbrowse" operations.

    proc _chunked_mbrowse {handle mbr} {
	# determine chunk size for "mbrowse"
	set max [limits $handle MaxNodesPerBrowse]
	if {$max <= 0} {
	    set max 500
	}
	set len [llength $mbr]
	set ret {}
	while {$len} {
	    if {$len > $max} {
		set nmbr [lrange $mbr 0 $max-1]
		set mbr [lrange $mbr $max end]
		incr len -$max
	    } else {
		set nmbr $mbr
		set mbr {}
		set len 0
	    }
	    if {[catch {mbrowse $handle {*}$nmbr} part]} {
		return -code error $part
	    }
	    lappend ret {*}$part
	}
	return $ret
    }

    # Internal helper performing chunked "mtranslate" operations.

    proc _chunked_mtranslate {handle mtr} {
	# determine chunk size for "mtranslate"
	set max [limits $handle MaxNodesPerTranslateBrowsePathsToNodeIds]
	if {$max <= 0} {
	    set max 500
	}
	set len [llength $mtr]
	set ret {}
	while {$len} {
	    if {$len > $max} {
		set nmtr [lrange $mtr 0 $max-1]
		set mtr [lrange $mtr $max end]
		incr len -$max
	    } else {
		set nmtr $mtr
		set mtr {}
		set len 0
	    }
	    if {[catch {mtranslate $handle {*}$nmtr} part]} {
		return -code error $part
	    }
	    lappend ret {*}$part
	}
	return $ret
    }

    # Internal helper performing chunked "mreadx" operations.

    proc _chunked_mreadx {handle mrd} {
	# determine chunk size for "mreadx"
	set max [limits $handle MaxNodesPerRead]
	if {$max <= 0} {
	    set max 500
	}
	set max [expr {$max * 3}]	;# 3-tupel per read operation
	set len [llength $mrd]
	set ret {}
	while {$len} {
	    if {$len > $max} {
		set nmrd [lrange $mrd 0 $max-1]
		set mrd [lrange $mrd $max end]
		incr len -$max
	    } else {
		set nmrd $mrd
		set mrd {}
		set len 0
	    }
	    if {[catch {mreadx $handle {} {*}$nmrd} part]} {
		return -code error $part
	    }
	    lappend ret {*}$part
	}
	return $ret
    }

    # Internal tree walker for "tree" and related procs.
    # Input argument "inlist" is made up of elements of
    #    nodeid browsename dispname nodeclass
    #    refnodeid typenodeid pnodeid ppath
    # Usage of "mbrowse" allows for minimum number of
    # calls to retrieve the tree information and to make
    # the function tail recursive.

    proc _walk {handle inlist result {mask {}} {level 0}} {
	if {$level > 128} {
	    return
	}
	upvar $result ret
	set brlist {}
	set pnlist {}
	foreach {
	    nodeid browsename dispname nodeclass
	    refnodeid typenodeid pnodeid ppath
	} $inlist {
	    if {[lsearch -exact $ppath $nodeid] >= 0} {
		# skip, already seen
		continue
	    }
	    set path $ppath
	    lappend path $nodeid
	    lappend ret $level $nodeid $browsename $dispname \
		$nodeclass $refnodeid $typenodeid $pnodeid $path
	    lappend brlist [list $nodeid Forward / {*}$mask]
	    lappend pnlist $nodeid $path
	}
	incr level
	set outlist {}
	# multi browse in chunks of "bsize" nodes
	set len [llength $brlist]
	# determine chunk size for "mbrowse"
	set bsize [limits $handle MaxNodesPerBrowse]
	if {$bsize == 0} {
	    set bsize 500
	}
	set bs2 $bsize
	incr bs2 $bsize
	while {$len} {
	    if {$len > $bsize} {
		set nbrlist [lrange $brlist 0 $bsize-1]
		set brlist [lrange $brlist $bsize end]
		set npnlist [lrange $pnlist 0 $bs2-1]
		set pnlist [lrange $pnlist $bs2 end]
		incr len -$bsize
	    } else {
		set nbrlist $brlist
		set brlist {}
		set npnlist $pnlist
		set pnlist {}
		set len 0
	    }
	    foreach br [mbrowse $handle {*}$nbrlist] {p path} $npnlist {
		if {[llength $br] == 0} {
		    continue
		}
		# sorted by display name
		foreach {n b d c r t} \
		    [lsort -dictionary -stride 6 -index 2 $br] {
		    lappend outlist $n $b $d $c $r $t $p $path
		}
	    }
	}
	if {[llength $outlist]} {
	    tailcall _walk $handle $outlist ret $mask $level
	}
    }

    # Return a list of the address space resembling the
    # tree view of UAExpert. List layout adds level column
    # to "opcua::browse ..." list:
    #
    #   level	    - 0 is Root, 1 is Objects etc.
    #   nodeid	    - the node identifier, e.g. "ns=1;i=99"
    #   browsename  - name of node for browsing
    #   dispname    - name of node for display
    #   nodeclass   - class of node, e.g. "Variable"
    #   refnodeid   - reference node identifer
    #   typenodeid  - type node identifier
    #   parent/path - parent node identifier or full path
    #
    # When "withpath" is true, "nodeid"s are not necessarily
    # unique but the last item ("path") is.

    proc tree {handle {root {}} {mask {}} {withpath 0}} {
	set ret {}
	set ref [reftype References]
	if {$root eq {}} {
	    set root [root]
	    set pn {}
	    set org [reftype HierarchicalReferences]
	    set type [_hxlate $handle \
			  Types/ObjectTypes/BaseObjectType/FolderType]
	} else {
	    set org $ref
	    set pn [parent $handle $root]
	    if {[catch {read $handle $root DataType} type]} {
		switch -- [read $handle $root NodeClass] {
		    Object {
			set type [_hxlate $handle \
				      Types/ObjectTypes/BaseObjectType]
		    }
		    DataType {
			set type [_hxlate $handle \
				      Types/DataTypes/BaseDataType]
		    }
		    VariableType {
			set type [_hxlate $handle \
				      Types/VariableTypes/BaseVariableType]
		    }
		    default {
			set type [_hxlate $handle \
				      Types/ReferenceTypes/References]
		    }
		}
	    }
	}
	# read BrowseName attribute yields QualifiedName
	set bname [read $handle $root BrowseName]
	# read DisplayName attribute yields LocalizedText
	# but we want the text part only here
	set dname [dict get [read $handle $root DisplayName] text]
	_walk $handle [list $root $bname $dname \
			[read $handle $root NodeClass] $org $type $pn {}] \
	    ret $mask
	if {$withpath} {
	    # remove duplicates by path
	    array set x {}
	    set nodup {}
	    foreach {l n b d c r t pn path} $ret {
		if {[::info exists x($path)]} {
		    continue
		}
		incr x($path)
		lappend nodup $l $n $b $d $c $r $t $path
	    }
	} else {
	    # remove duplicates by node
	    array set x {}
	    set nodup {}
	    foreach {l n b d c r t pn path} $ret {
		if {[::info exists x($n)]} {
		    continue
		}
		incr x($n)
		lappend nodup $l $n $b $d $c $r $t $pn
	    }
	}
	return $nodup
    }

    # Similar to tree, but make path like layout as in
    # "browsepath nodeid nodeclasspath refnodeid typenodeid parent ...", e.g.
    #
    # /Root	    i=84 /Object	i=35	i=61	{}
    # /Root/Objects i=85 /Object/Object	i=35	i=61	i=84
    # ...
    #
    # Browse path components with a namespace different from the root namespace
    # are written as qualified names ("nsindex:name"), e.g. "2:AnotherFolder".

    proc ptree {handle {root {}} {mask {}}} {
	set ret {}
	array set cc {}
	array set bb {}
	foreach {l n b d c r t path} [tree $handle $root $mask 1] {
	    set br {}
	    set cl {}
	    set cc($n) $c
	    set bb($n) $b
	    set pn [lindex $path end-1]
	    foreach i $path {
		lappend br $bb($i)
		lappend cl $cc($i)
	    }
	    lappend ret /[join $br /] $n /[join $cl /] $r $t $pn
	}
	return $ret
    }

    # Return a list of child nodes given parent.

    proc children {handle nodeid} {
	set ret {}
	foreach {n b d c r t} [browse $handle $nodeid Forward /] {
	    lappend ret $n
	}
	return $ret
    }

    # Return the parent node given child.

    proc parent {handle nodeid} {
	set ret {}
	foreach {n b d c r t} [browse $handle $nodeid Inverse /] {
	    set ret $n
	    # only the first item
	    break
	}
	return $ret
    }

    # Generate stubs for methods in sub-namespace derived from
    # client or server name. Argument "strip" is cut off from the begin
    # of browse paths. Argument "substs" is a list of regexp/substitution
    # elements which are applied in order on the browse path. All following
    # arguments are glob patterns for matching in browse paths before the
    # substitutions are applied. The entire address space is processed
    # by using the root node as base for retrieving the tree of names.
    # Example:
    #
    #   opcua::new client Pumps
    #   opcua::connect Pumps ...
    #   opcua::genstubs Pumps /Root/Objects/2:Pumps/2:
    #
    # writes these procs (assuming there are Pump_1 and Pump_2 objects
    # each having a Start and a Stop method with zero input arguments):
    #
    #   proc opcua::Pumps::Pump_1/Start {} ...
    #   proc opcua::Pumps::Pump_1/Stop {} ...
    #   proc opcua::Pumps::Pump_2/Start {} ...
    #   proc opcua::Pumps::Pump_2/Stop {} ...
    #
    # The result is a list of the proc names which were generated.

    proc genstubs {handle {strip {}} {substs {}} args} {
	set ret [dict get [::info frame -1]]
	if {[dict exists $ret proc] &&
	    ([dict get $ret proc] eq "::opcua::xgenstubs")} {
	    set call ::opcua::xcall
	} else {
	    set call ::opcua::call
	}
	::namespace eval ::opcua::$handle {}
	set all [expr {[llength $args] == 0}]
	set root [root]
	if {$all && ([string first /Root/Objects $strip] == 0)} {
	    # speed up for common place
	    set strip [string range $strip 5 end]
	    set root [_hxlate $handle Objects]
	}
	set plist {}
	set trlist {}
	foreach {b n c r t pn} [ptree $handle $root {Object Method}] {
	    if {($strip ne {}) && ([string first $strip $b] != 0)} {
		continue
	    }
	    if {![string match "*Object/Method" $c]} {
		continue
	    }
	    if {$all} {
		set found 1
	    } else {
		set found 0
		foreach pat $args {
		    if {[string match $pat $b]} {
			set found 1
			break
		    }
		}
	    }
	    if {$found} {
		# procname will be method path with prefix stripped ...
		set b [string range $b [string length $strip] end]
		# ... and mangled by substitutions
		set st {}
		foreach {re st} $substs {
		    regsub -all -- $re $b $st b
		}
		# we need the name, the node, and its parent for stubs
		lappend plist $b $n $pn
		lappend trlist [list $n / InputArguments]
	    }
	}
	# look for input arguments
	if {[catch {_chunked_mtranslate $handle $trlist} trdata]} {
	    # try the expensive way
	    set trdata {}
	    foreach tr $trlist {
		if {[catch {translate $handle {*}$tr} ia]} {
		    lappend trdata {}
		} else {
		    lappend trdata $ia
		}
	    }
	}
	set ret {}
	set plist1 {}
	set plist2 {}
	set rlist {}
	foreach tr $trdata {b n o} $plist {
	    lassign $tr ia _ _
	    if {$ia eq {}} {
		# method w/o input arguments
		lappend plist1 $b $n $o
		continue
	    }
	    lappend plist2 $b $n $o
	    lappend rlist $ia Value {}
	}
	# make "plist1" stubs
	foreach {b n o} $plist1 {
	    set o [list $o]		;# beware of semicolon in nodeid
	    set n [list $n]		;# beware of semicolon in nodeid
	    set pr ::opcua::${handle}::${b}
	    eval [format "proc %s {} {\n    tailcall %s %s %s %s\n}" \
			[list $pr] $call [list $handle] $o $n]
	    lappend ret $pr
	}
	# read input arguments
	if {[catch {_chunked_mreadx $handle $rlist} rdata]} {
	    return $ret
	}
	# make "plist2" stubs
	foreach {sc ia} $rdata {b n o} $plist2 {
	    set plist {}
	    set alist {}
	    if {$sc != 0} {
		# assume method w/o input arguments
	    } else {
		# go over the arguments
		foreach in $ia {
		    if {![dict exists $in Name] ||
			![dict exists $in DataType]} {
			# should not happen
			continue
		    }
		    set name [dict get $in Name]
		    set typeid [dict get $in DataType]
		    # no whitespace in argument name, otherwise
		    # we might end up with wrong names and default
		    # values or even worse, plain syntax errors
		    regsub -all {\s+} $name _ name
		    lappend plist $name
		    if {[string match "ns=*;*" $typeid]} {
			# keep identifier
			set type $typeid
		    } elseif {[catch {types name $typeid} type]} {
			# can't use string name
			set type $typeid
		    }
		    set vr [dict get $in ValueRank]
		    if {$vr >= [const VALUERANK_ONE_OR_MORE_DIMENSIONS]} {
			set type *${type}
		    } elseif {$vr == [const VALUERANK_SCALAR]} {
			set type !${type}
		    }
		    # caution: due to variable reference, need string here
		    append alist " [list $type] \$[list $name]"
		}
	    }
	    set o [list $o]		;# beware of semicolon in nodeid
	    set n [list $n]		;# beware of semicolon in nodeid
	    set pr ::opcua::${handle}::${b}
	    eval [format "proc %s %s {\n    tailcall %s %s %s %s%s\n}" \
			[list $pr] [list $plist] $call [list $handle] \
			$o $n $alist]
	    lappend ret $pr
	}
	return $ret
    }

    # Setup namespace zero struct type mappings.
    # Info is fetched recursively from "/Root/Types/DataTypes/BaseDataType".

    proc _ns0structs {handle {brlvar {}}} {
	if {$brlvar ne {}} {
	    # save browse list in caller
	    upvar $brlvar brlist
	}
	set currtypes [types list $handle]
	set stroot [_hxlate $handle Types/DataTypes/BaseDataType]
	set defs {}
	set mrd {}
	set mrb {}
	set brlist [tree $handle $stroot {Object DataType}]
	foreach {level nodeid brname dname cls refid typeid parent} $brlist {
	    # sort out: must be in namespace zero
	    if {[scan $nodeid "ns=%d;%s" ns dummy] == 2} {
		continue
	    }
	    if {$cls ne "DataType"} {
		continue
	    }
	    if {$brname in $currtypes} {
		continue
	    }
	    lappend mrd $nodeid DataTypeDefinition {}
	    lappend mrb $nodeid $brname
	}
	# speed up: read all DataTypeDefinitions with "mread"
	if {![catch {_chunked_mreadx $handle $mrd} dtd]} {
	    foreach {sc def} $dtd {nodeid brname} $mrb {
		if {$sc != 0} {
		    continue
		}
		# TBD: only plain structs for now
		if {![dict exists $def StructureType] ||
		    ([dict get $def StructureType] !=
		     [const STRUCTURETYPE_STRUCTURE])} {
		    continue
		}
		set encid [dict get $def DefaultEncodingId]
		set struct [list typedef $handle struct $brname $nodeid $encid]
		if {[llength [dict get $def Fields]] == 0} {
		    # abstract type?
		    continue
		}
		foreach field [dict get $def Fields] {
		    # TBD: handle IsOptional proper
		    set fname [dict get $field Name]
		    if {[dict get $field ValueRank] >
			[const VALUERANK_SCALAR]} {
			set fname *$fname
		    }
		    lappend struct [dict get $field DataType] $fname
		}
		lappend defs $struct
	    }
	}
	if {[llength $defs]} {
	    typedef $handle begin
	    foreach cmd $defs {
		if {[catch {{*}$cmd}]} {
		    # can fail when the reduced namespace zero
		    # is compiled into the open62541 library
		}
	    }
	    typedef $handle commit
	}
    }

    # Generate structs and enums from node information
    # for namespaces other than zero.
    # Info is fetched recursively from "/Root/Types/DataTypes/BaseDataType".

    proc _gentypes1 {handle {brlvar {}}} {
	if {$brlvar ne {}} {
	    upvar $brlvar brlist
	}
	array set currmap [types map $handle]
	set currtypes [array names currmap]
	set newtypes {}
	set stroot [_hxlate $handle Types/DataTypes/BaseDataType]
	set struid [_hxlate $handle Types/DataTypes/BaseDataType/Structure]
	set defs {}
	set mrd {}
	set mrb {}
	# sort by level that lower level's type can get a shortened
	# type name by omitting the namespace prefix
	if {![::info exists brlist]} {
	    set brlist [tree $handle $stroot {Object DataType}]
	}
	foreach {level nodeid brname dname cls refid typeid parent} \
	    [lsort -integer -stride 8 -index 0 $brlist] {
	    # sort out: only data types to be considered
	    if {$cls ne "DataType"} {
		continue
	    }
	    if {$nodeid in $currtypes} {
		continue
	    }
	    lappend mrd $nodeid DataTypeDefinition {}
	    lappend mrb $nodeid $brname $parent
	}
	# speed up: read all DataTypeDefinitions in a single operation
	if {![catch {_chunked_mreadx $handle $mrd} dtd]} {
	    set mbr {}
	    set mbp {}
	    set mrd {}
	    set mre {}
	    foreach {sc def} $dtd {nodeid brname parent} $mrb {
		# build list for multi browse operation for all
		# nodes which don't have a DataTypeDefinition
		if {$sc != 0} {
		    lappend mbr [list $nodeid Forward HasProperty Variable]
		    lappend mbp $nodeid $brname $parent
		}
	    }
	    if {[catch {_chunked_mbrowse $handle $mbr} dtlist]} {
		set dtlist {}
	    }
	    # enums/subtypes from multi browse first,
	    # for the EnumStrings and EnumValues items
	    # collect info for another multi read operation
	    foreach br $dtlist {nodeid brname parent} $mbp {
		# only consider first hit per mbrowse subresult
		lassign $br n b d c r t
		set name $brname
		# omit namespace index if type name is new
		scan $name "%d:%s" ns name
		if {($name ne $brname) && ($name in $newtypes)} {
		    set name $brname
		}
		if {$b eq "EnumStrings"} {
		    lappend mrd $n Value {}
		    lappend mre $nodeid $brname EnumStrings
		} elseif {$b eq "EnumValues"} {
		    lappend mrd $n Value {}
		    lappend mre $nodeid $brname EnumValues
		} elseif {($parent ne [types nodeid ExtensionObject]) &&
			  ($parent ne [types nodeid Variant])} {
		    # both, parent type and current type must not be abstract
		    if {[catch {
			mreadx $handle {} $parent IsAbstract {} \
			    $nodeid IsAbstract {}} isa]} {
			continue
		    }
		    lassign $isa psc pisa nsc nisa
		    if {($psc == 0) && !$pisa && ($nsc == 0) && !$nisa} {
			# neither enum nor abstract, if it's not a struct
			# it must be a simple subtype
			set pp $parent
			while {$pp ne {}} {
			    if {$pp eq $struid} {
				break
			    }
			    set pp [parent $handle $pp]
			}
			if {$pp eq {}} {
			    lappend defs [list typedef $handle subtype $name \
				$nodeid $parent]
			    lappend newtypes $name
			}
		    }
		}
	    }
	    # enums from multi read information
	    if {![catch {_chunked_mreadx $handle $mrd} enl]} {
		foreach {sc val} $enl {nodeid brname ename} $mre {
		    if {$sc != 0} {
			continue
		    }
		    set name $brname
		    # omit namespace index if type name is new
		    scan $name "%d:%s" ns name
		    if {($name ne $brname) && ($name in $newtypes)} {
			set name $brname
		    }
		    unset -nocomplain elist
		    if {$ename eq "EnumStrings"} {
			set i 0
			foreach v $val {
			    if {[dict exists $v text]} {
				set v [dict get $v text]
			    } else {
				set v "Unknown$i"
			    }
			    lappend elist $i $v
			    incr i
			}
		    } elseif {$ename eq "EnumValues"} {
			foreach ev $val {
			    if {[dict exists $ev Value]} {
				unset -nocomplain v
				set i [dict get $ev Value]
				if {[dict exists $ev DisplayName]} {
				    catch {
					set v [dict get \
					    [dict get $ev DisplayName] text]
				    }
				}
				if {![::info exists v]} {
				    set v "Unknown$i"
				}
				lappend elist $i $v
			    }
			}
		    }
		    if {[::info exists elist]} {
			set enum [list typedef $handle enum $name $nodeid]
			lappend enum [types nodeid UInt32] UInt32 {*}$elist
			lappend defs $enum
			lappend newtypes $name
		    }
		}
	    }
	    # structs/enums from DataTypeDefinition
	    foreach {sc def} $dtd {nodeid brname parent} $mrb {
		if {$sc != 0} {
		    continue
		}
		set name $brname
		# omit namespace index if type name is new
		scan $name "%d:%s" ns name
		if {($name ne $brname) && ($name in $newtypes)} {
		    set name $brname
		}
		set kind {}
		if {![dict exists $def StructureType]} {
		    # not a struct, check for enum
		    unset -nocomplain elist
		    if {[dict exists $def Fields]} {
			set kind enum
			foreach v [dict get $def Fields] {
			    lappend elist [dict get $v Value] \
				[dict get $v Name]
			}
			if {![::info exists elist]} {
			    # no fields, ignored
			    continue
			}
		    } else {
			# not an enum, ignored
			continue
		    }
		} elseif {[dict get $def StructureType] ==
			  [const STRUCTURETYPE_STRUCTURE]} {
		    # simple struct, which we support
		    set kind struct
		} elseif {[dict get $def StructureType] ==
			  [const STRUCTURETYPE_UNION]} {
		    # simple union, which we support
		    set kind union
		} elseif {[dict get $def StructureType] ==
			  [const STRUCTURETYPE_STRUCTUREWITHOPTIONALFIELDS]} {
		    # optstruct, which we support
		    set kind optstruct
		} else {
		    # all else unsupported
		    continue
		}
		switch -exact -- $kind {
		    enum {
			set enum [list typedef $handle $kind $name $nodeid]
			lappend enum [types nodeid UInt32] UInt32 {*}$elist
			lappend defs $enum
			lappend newtypes $name
		    }
		    struct - union {
			if {[llength [dict get $def Fields]] == 0} {
			    # abstract type?
			    continue
			}
			set encid [dict get $def DefaultEncodingId]
			set struct [list typedef $handle $kind $name \
			    $nodeid $encid]
			foreach field [dict get $def Fields] {
			    set fname [dict get $field Name]
			    set dtype [dict get $field DataType]
			    if {[dict get $field ValueRank] >
				[const VALUERANK_SCALAR]} {
				set fname *$fname
			    }
			    lappend struct $dtype $fname
			}
			lappend defs $struct
			lappend newtypes $name
		    }
		    optstruct {
			if {[llength [dict get $def Fields]] == 0} {
			    # abstract type?
			    continue
			}
			set encid [dict get $def DefaultEncodingId]
			set struct [list typedef $handle $kind $name \
			    $nodeid $encid]
			foreach field [dict get $def Fields] {
			    set fname [dict get $field Name]
			    set dtype [dict get $field DataType]
			    if {[dict get $field ValueRank] >
				[const VALUERANK_SCALAR]} {
				set fname *$fname
			    }
			    if {[dict get $field IsOptional]} {
				set fopt "optional"
			    } else {
				set fopt "mandatory"
			    }
			    lappend struct $fopt $dtype $fname
			}
			lappend defs $struct
			lappend newtypes $name
		    }
		}
	    }
	}
	set newtypes {}
	if {[llength $defs]} {
	    typedef $handle begin
	    set max [llength $defs]
	    set count 0
	    # multiple rounds since order of defs is undefined
	    while {1} {
		set ndefs {}
		foreach cmd $defs {
		    if {[catch {{*}$cmd} err]} {
			lappend ndefs $cmd
		    } else {
			lappend newtypes [lindex $cmd 3]
		    }
		}
		set nmax [llength $ndefs]
		if {($nmax == 0) || ($nmax == $max)} {
		    break
		}
		set defs $ndefs
		set max $nmax
		incr count
	    }
	    typedef $handle commit
	}
	return $newtypes
    }

    # Read *.bsd string for namespaces, enums, structs
    # returning a dictionary with:
    #
    #   namespaces  { prefix uri ... }
    #   enums       { enumname { bit-width itemname value ... } ... }
    #   structs     { structname { opt type fieldname ... isunion } ... }
    #
    # This function needs tDOM for XML processing!

    proc _readbsd {handle string} {
	dict set out namespaces {}
	dict set out enums {}
	dict set out structs {}
	if {[catch {
	    package require tdom
	    set doc [dom parse $string]
	}]} {
	    return $out
	}
	set root [$doc documentElement]
	# process namespaces
	array set ns {}
	foreach att [$root attributes] {
	    if {[llength $att] == 3} {
		lassign $att loc pfx uri
	    }
	    if {$uri ne ""} {
		continue
	    }
	    set nsx "xmlns:$loc"
	    if {[$root hasAttribute $nsx]} {
		set ns($pfx) [$root getAttribute $nsx]
	    }
	}
	dict set out namespaces [array get ns]
	set nsl [array names ns]
	if {("opc" ni $nsl) || ("tns" ni $nsl) || ("ua" ni $nsl)} {
	    # need opc, tns, and ua namespace prefixes
	    return -code error -errorcode {opcua Internal 0 Good} \
		"xmlns:opc, xmlns:tns, or xmlns:ua missing"
	}
	# now deal with /opc:TypeDictionary
	set typedict [lindex [$root selectNodes /opc:TypeDictionary] 0]
	# process structs/unions
	array set st {}
	foreach struct [$typedict selectNodes opc:StructuredType] {
	    set isunion 0
	    if {[$struct hasAttribute BaseType]} {
		# must be ua:ExtensionObject or ua:Union, otherwise skip
		if {[$struct getAttribute BaseType] eq "ua:Union"} {
		    set isunion 1
		} elseif {[$struct getAttribute BaseType] ne
			  "ua:ExtensionObject"} {
		    continue
		}
	    }
	    array set bits {}
	    array set swf {}
	    set stf {}
	    foreach field [$struct childNodes] {
		if {[$field nodeName] ne "opc:Field"} {
		    continue
		}
		set swname {}
		if {[$field hasAttribute SwitchField]} {
		    set swname [$field getAttribute SwitchField]
		} elseif {$isunion} {
		    # ignore field
		    continue
		}
		if {![$field hasAttribute TypeName]} {
		    # cannot process this struct
		    set stf {}
		    break
		}
		set nstype [$field getAttribute TypeName]
		set type ""
		scan $nstype {%[^:]:%s} fns type
		if {$type ne ""} {
		    switch -- $fns {
			opc - ua {
			    # use type
			}
			tns {
			    set type $nstype
			}
			default {
			    # cannot process this struct
			    set stf {}
			    break
			}
		    }
		} else {
		    set type $nstype
		}
		if {![$field hasAttribute Name]} {
		    # cannot process this struct
		    set stf {}
		    break
		}
		set fn [$field getAttribute Name]
		if {$type eq "Bit"} {
		    # optstruct indicator, not a real field
		    set bits($fn) 1
		    continue
		}
		if {$swname ne {}} {
		    set swf($fn) $swname
		}
		# handle attribute LengthField, it makes the field
		# into an array, if not empty; the field's name is
		# set to 'field' for scalar, and '*field' for array
		if {[$field hasAttribute LengthField] &&
		    ([$field getAttribute LengthField] ne "")} {
		    if {[::info exists bits($fn)]} {
			set bits(*$fn) $bits($fn)
		    }
		    if {[::info exists swf($fn)]} {
			set swf(*$fn) $swf($fn)
		    }
		    # remove the length field ("NoOf*") in the
		    # struct list immediately before this field
		    if {[lindex $stf end] eq "NoOf$fn"} {
			set stf [lrange $stf 0 end-2]
		    }
		    set fn *$fn
		}
		lappend stf $type $fn
	    }
	    set newst {}
	    foreach {type fn} $stf {
		set opt 0
		if {[::info exists swf($fn)]} {
		    set bit $swf($fn)
		    if {[::info exists bits($bit)]} {
			set opt 1
		    }
		}
		lappend newst $opt $type $fn
	    }
	    if {[llength $newst] && [$struct hasAttribute Name]} {
		lappend newst $isunion
		set st([$struct getAttribute Name]) $newst
	    }
	    unset -nocomplain bits
	    unset -nocomplain swf
	}
	# structs/unions need further checking later
	set stl [array names st]
	# process enums
	array set en {}
	foreach enum [$typedict selectNodes opc:EnumeratedType] {
	    set enf {}
	    foreach value [$enum childNodes] {
		if {[$value nodeName] ne "opc:EnumeratedValue"} {
		    continue
		}
		if {[$value hasAttribute Name] && [$value hasAttribute Value]} {
		    lappend enf \
			[$value getAttribute Name] \
			[$value getAttribute Value]
		}
	    }
	    if {[llength $enf] && [$enum hasAttribute Name] &&
		[$enum hasAttribute LengthInBits]} {
		set en([$enum getAttribute Name]) \
		    [concat [$enum getAttribute LengthInBits] $enf]
	    }
	}
	dict set out enums [array get en]
	set enl [array names en]
	# all types in namespace zero
	set ns0types [types list]
	# cross check type names in structs/unions
	foreach name $stl {
	    set newst {}
	    set isunion [lindex $st($name) end]
	    foreach {opt type field} [lrange $st($name) 0 end-1] {
		if {$type eq "CharArray"} {
		    set type "String"
		}
		# if type is a type nodeid, accept it
		if {![catch {types name $nstype}]} {
		    lappend newst $opt $type $field
		    continue
		}
		set fns ""
		set nstype $type
		scan $type {%[^:]:%s} fns nstype
		if {($nstype ne "") && ($fns eq "tns")} {
		    if {($nstype ni $stl) && ($nstype ni $enl)} {
			if {[catch {types nodeid $handle $nstype}]} {
			    # maybe in a different namespace, skip
			    set newst {}
			    break
			}
			set type $nstype
		    }
		    # check enum field and add integer base type
		    if {$nstype in $enl} {
			switch -- [lindex $en($nstype) 0] {
			    32 {
				lappend newst $opt [list $type UInt32] $field
			    }
			    16 {
				lappend newst $opt [list $type UInt16] $field
			    }
			    8 {
				lappend newst $opt [list $type Byte] $field
			    }
			    default {
				# unsupported width, skip
				set newst {}
				break
			    }
			}
		    } else {
			lappend newst $opt $type $field
		    }
		} elseif {$type ni $ns0types} {
		    # not a namespace zero type, skip struct
		    set newst {}
		    break
		} else {
		    lappend newst $opt $type $field
		}
	    }
	    if {$newst eq {}} {
		unset st($name)
	    } else {
		lappend newst $isunion
		set st($name) $newst
	    }
	}
	dict set out structs [array get st]
	# finish, clean up
	$doc delete
	return $out
    }

    # Get struct types not in namespace zero, returns list
    #
    #   { name nodeid encid bsdname ... }
    #
    # Info is fetched from "/Root/Types/DataTypes/BaseDataType/Structure"
    # and must have proper encoding and description references.

    proc _getstructs {handle {brlist {}}} {
	set nodes {}
	if {[catch {
		_hxlate $handle Types/DataTypes/BaseDataType/Structure
	} stroot]} {
	    return {}
	}
	if {$brlist eq {}} {
	    foreach {level nodeid brname dname cls refid typeid parent} \
		[tree $handle $stroot {Object DataType}] {
		if {$cls eq "DataType"} {
		    lappend nodes $nodeid $brname
		}
	    }
	} else {
	    array set isstr [list $stroot 1]
	    # sort by level that subtyped structs are not in front
	    foreach {level nodeid brname dname cls refid typeid parent} \
		[lsort -integer -stride 8 -index 0 $brlist] {
		if {$cls ne "DataType"} {
		    continue
		}
		if {$parent in [array names isstr]} {
		    lappend nodes $nodeid $brname
		    set isstr($nodeid) 1
		}
	    }
	}
	set ablist {}
	set nlist {}
	foreach {nodeid brname} $nodes {
	    # sort out: not in namespace zero
	    if {[scan $nodeid "ns=%d;%s" ns dummy] == 2} {
		if {[string match "${ns}:*" $brname]} {
		    scan $brname {%[^:]:%s} ns name
		    # struct must be non-abstract
		    lappend ablist $nodeid IsAbstract {}
		    lappend nlist $name $nodeid $brname
		}
	    }
	}
	if {[catch {_chunked_mreadx $handle $ablist} ablist]} {
	    return {}
	}
	set nlist2 {}
	set trlist1 {}
	foreach {sc ab} $ablist {name nodeid brname} $nlist {
	    if {($sc == 0) && !$ab} {
		# struct must have a "Default Binary" encoding
		# otherwise it doesn't map to an ExtensionObject
		lappend trlist1 [list $nodeid HasEncoding "Default Binary"]
		lappend nlist2 $name $nodeid $brname
	    }
	}
	if {$trlist1 eq {}} {
	    return {}
	}
	set nlist $nlist2
	if {[catch {_chunked_mtranslate $handle $trlist1} trdata]} {
	    return {}
	}
	set trlist2 {}
	foreach tr $trdata {name nodeid brname} $nlist {
	    # struct name in *.bsd is the value of the
	    # HasDescription reference of the encoding
	    lassign $tr encid _ _
	    if {$encid ne {}} {
		lappend trlist2 [list $encid HasDescription $brname]
		lappend mlist $name $nodeid $encid
	    }
	}
	if {$trlist2 eq {}} {
	    return {}
	}
	if {[catch {_chunked_mtranslate $handle $trlist2} trdata]} {
	    return {}
	}
	# prepare read of *.bsd names
	set rlist {}
	set nlist {}
	foreach tr $trdata {name nodeid encid} $mlist {
	    lassign $tr bsdid _ _
	    if {$bsdid ne {}} {
		lappend rlist $bsdid Value {}
		lappend nlist $name $nodeid $encid
	    }
	}
	if {$rlist eq {}} {
	    return {}
	}
	if {[catch {_chunked_mreadx $handle $rlist} rdata]} {
	    return {}
	}
	if {$rdata eq {}} {
	    return {}
	}
	set st {}
	foreach {sc bsdname} $rdata {name nodeid encid} $mlist {
	    if {$sc == 0} {
		lappend st $name $nodeid $encid $bsdname
	    }
	}
	return $st
    }

    # Get *.bsd info, skips namespace zero and returns list
    #
    #    { nsindex uri bsd ... }
    #
    # Info is fetched from "/Root/Types/DataTypes/OPC Binary".

    proc _getbsds {handle} {
	# make translate list, read list for XML, and namespace index list
	set bsdroot [_hxlate $handle "Types/DataTypes/OPC Binary"]
	set trlist {}
	foreach nodeid [children $handle $bsdroot] {
	    # sort out: not in namespace zero, must be numeric nodeid
	    if {[scan $nodeid "ns=%d;i=%d" ns id] == 2} {
		lappend trlist [list $nodeid HasProperty NamespaceUri]
		lappend rdlist1 $nodeid Value {}
		lappend nslist $ns
	    }
	}
	if {$trlist eq {}} {
	    return {}
	}
	if {[catch {mtranslate $handle {*}$trlist} trdata]} {
	    # try the expensive way
	    set trdata {}
	    foreach {nodeid ref brname} $trlist {
		if {[catch {translate $handle $nodeid $ref $brname} tr]} {
		    lappend trdata {}
		} else {
		    lappend trdata $tr
		}
	    }
	}
	# now find out URIs either by value or display name attributes
	foreach tr $trdata {nodeid _ _} $trlist {
	    lassign $tr urid _ _
	    if {$urid ne {}} {
		lappend rdlist2 $urid Value {}
	    } else {
		lappend rdlist2 $nodeid DisplayName {}
	    }
	}
	if {[catch {mreadx $handle {} {*}$rdlist1} xmlist] ||
	    [catch {mreadx $handle {} {*}$rdlist2} urlist]} {
	    return {}
	}
	set bsd {}
	# all lists setup, make result
	foreach {sc1 xml} $xmlist {sc2 uri} $urlist \
	    {nodeid attr index} $rdlist2 ns $nslist {
	    if {$sc1 != 0} {
		# no XML read, skip it
		continue
	    }
	    if {$sc2 != 0} {
		set uri ""
	    } elseif {$attr eq "DisplayName"} {
		if {[catch {set uri [dict get $uri text]}]} {
		    set uri ""
		}
	    }
	    lappend bsd $ns $uri $xml
	}
	return $bsd
    }

    # Dump *.bsd info given handle, this is the public interface
    # to _getbsds above which omits the namespace indices.

    proc dumpbsds {handle} {
	lmap {nsindex uri bsd} [_getbsds $handle] {
	    list $uri $bsd
	}
    }

    # Resolve deferred typedefs (TBD: needs testing)
    #
    #  name -	new name which now is fully defined
    #  hvar -	array name of already resolved names
    #  nvar -	array name of needed names, element names are target names,
    #		element values are a list of required names each
    #  lvar -	array name of deferred typedefs
    #  dvar -	output list which gets resolved typedefs appended

    proc _resolve {name hvar nvar lvar dvar} {
	upvar $hvar have
	upvar $nvar need
	upvar $lvar later
	upvar $dvar defs
	set todo {}
	foreach n [array names need] {
	    set i [lsearch -exact $need($n) $name]
	    if {$i >= 0} {
		set need($n) [lreplace $need($n) $i $i]
		if {$need($n) eq {}} {
		    unset need($n)
		    if {[::info exists later($n)]} {
			lappend defs $later($n)
			unset later($n)
			set have($n) 1
		    }
		    lappend todo $n
		}
	    }
	}
	foreach n $todo {
	    _resolve $n have need later defs
	}
    }

    # Generate ExtensionObject defs for enums and structs.
    # Method information of a server handle must be saved
    # and restored, since the "gentypes" proc recreates
    # type mappings. Optional "args" are pairwise uri
    # and xml strings provided by caller. Otherwise the
    # information is read from the server.

    proc gentypes {handle args} {
	set brlist {}
	_ns0structs $handle brlist
	if {[info $handle] ne "server"} {
	    set types [_gentypes1 $handle brlist]
	    # the hard way by XML parsing the *.bsd information
	    lappend types {*}[_gentypes $handle brlist {*}$args]
	    return [lsort -dictionary -unique $types]
	}
	set msaved [methods $handle]
	set types [_gentypes1 $handle brlist]
	set ret [catch {_gentypes $handle brlist {*}$args} result]
	if {$ret} {
	    # remember errorCode
	    set ec $::errorCode
	}
	foreach {m _ _} $msaved {
	    # try to restore method output types
	    if {[catch {
		lindex [translate $handle $m HasProperty OutputArguments] 0
	    } oid]} {
		catch {methods $handle $m {}}
	    } else {
		catch {methods $handle $m [read $handle $oid]}
	    }
	}
	if {$ret} {
	    return -code $ret -errorcode $ec $result
	}
	lappend types {*}$result
	return [lsort -dictionary -unique $types]
    }

    proc _gentypes {handle {brlvar {}} args} {
	if {$brlvar ne {}} {
	    upvar $brlvar brlist
	} else {
	    set brlist {}
	}
	foreach {name nodeid encid bsdname} [_getstructs $handle $brlist] {
	    set stn($name) $nodeid
	    set enc($name) $encid
	    set bsn($bsdname) $name
	}
	if {![array exists stn]} {
	    return
	}
	set defs {}
	if {[llength $args]} {
	    set bsds {}
	    set nslist \
		[read $handle [_hxlate $handle Objects/Server/NamespaceArray]]
	    foreach {uri bsd} $args {
		set nsindex [lsearch -exact $nslist $uri]
		if {$nsindex > 0} {
		    # omit namespace zero
		    lappend bsds $nsindex $uri $bsd
		}
	    }
	} else {
	    set bsds [_getbsds $handle]
	}
	foreach {nsindex uri bsd} $bsds {
	    set data [_readbsd $handle $bsd]
	    array set need {}
	    array set later {}
	    array set have {}
	    foreach {en ed} [dict get $data enums] {
		if {![catch {types nodeid $handle $en}] ||
		    ![catch {types nodeid $handle ${nsindex}:$en}]} {
		    # already defined, skip
		    continue
		}
		switch -- [lindex $ed 0] {
		    32 {
			set type UInt32
		    }
		    16 {
			set type UInt16
		    }
		    8 {
			set type Byte
		    }
		    default {
			# skip, not representable
			continue
		    }
		}
		if {[catch {translate $handle [root] \
			/ Types / DataTypes / BaseDataType \
				/ Enumeration / ${nsindex}:$en} id]} {
		    continue
		}
		set id [lindex $id 0]
		if {[catch {
		    translate $handle $id HasEncoding {Default Binary}
		} eid]} {
		    set eid $id
		}
		set eid [lindex $eid 0]
		lappend defs [list typedef $handle enum \
				  $en $id $eid $type \
				  {*}[lrange $ed 1 end]]
	    }
	    foreach {sn sd} [dict get $data structs] {
		if {![::info exists bsn($sn)]} {
		    # no mapping from *.bsd to address space
		    continue
		}
		# bn is address space struct name, sn is *.bsd struct name
		# Holy indirection, Batman!
		# Robin, let's use the *.bsd name in the opcua::typedef
		set bn $bsn($sn)
		if {[::info exists stn($bn)] && [::info exists stn($sn)]} {
		    # verify namespace index
		    if {[scan $stn($sn) "ns=%d;i=%d" ns id] != 2} {
			continue
		    }
		    if {$ns != $nsindex} {
			continue
		    }
		}
		if {![catch {types nodeid $handle $sn}] ||
		    ![catch {types nodeid $handle ${nsindex}:$sn}]} {
		    # already defined, skip
		    continue
		}
		set isunion [lindex $sd end]
		set isopt 0
		if {$isunion} {
		    set def [list typedef $handle union]
		} else {
		    foreach {opt type member} [lrange $sd 0 end-1] {
			if {$opt} {
			    set isopt 1
			    break
			}
		    }
		    if {$isopt} {
			set def [list typedef $handle optstruct]
		    } else {
			set def [list typedef $handle struct]
		    }
		}
		lappend def $sn $stn($bn) $enc($bn)
		foreach {opt type member} [lrange $sd 0 end-1] {
		    if {[regsub -- ^tns: $type ${nsindex}: type]} {
			# enums have two elements
			scan [lindex $type 0] "%d:%s" tns ttype
			if {![::info exists have($ttype)]} {
			    set ee {}
			    if {[llength $type] > 1} {
				# check for enum
				if {[catch {translate $handle [root] \
					/ Types / DataTypes / BaseDataType \
					/ Enumeration / [lindex $type 0]} ee]} {
				    set ee {}
				}
			    } else {
				# assume another struct defined later
			    }
			    if {($ee eq {}) && ($ttype ne $sn)} {
				lappend need($sn) $ttype
			    }
			}
			# omit namespace prefix unless type exists
			if {[catch {types nodeid $handle $type}]} {
			    set type $ttype
			}
			unset ttype
		    }
		    if {$isopt} {
			lappend def [expr {$opt ? "optional" : "mandatory"}]
		    }
		    lappend def $type $member
		}
		if {[::info exists need($sn)]} {
		    set later($sn) $def
		} else {
		    lappend defs $def
		    set have($sn) 1
		    _resolve $sn have need later defs
		}
	    }
	    # Try to fix missing types by making forward declarations
	    foreach sn [array names need] {
		lappend defs [lrange $later($sn) 0 5]
	    }
	    foreach sn [array names need] {
		lappend defs $later($sn)
	    }
	    unset -nocomplain need
	    unset -nocomplain later
	    unset -nocomplain have
	}
	# Now typedef all the shiny new things in a transaction:
	# in a client, this replaces former definitions,
	# in a server, this adds new definitions.
	set newtypes {}
	if {[llength $defs]} {
	    typedef $handle begin
	    foreach cmd $defs {
		{*}$cmd
		lappend newtypes [lindex $cmd 3]
	    }
	    typedef $handle commit
	    # the struct typedefs need a representation as
	    # variable type of data variables, add it now
	    # for all structs.
	    set BD [_hxlate $handle \
		Types/VariableTypes/BaseVariableType/BaseDataVariableType]
	    foreach cmd $defs {
		set cmd [lassign $cmd typedef dummy kind type]
		if {$kind ni {struct optstruct union}} {
		    continue
		}
		# list of types/members of struct in sdef
		set sdef [lassign $cmd nodeid dummy]
		set ns 0
		scan $nodeid "ns=%d;i=%d" ns dummy
		# delete old definition, if any
		if {![catch {
		    translate $handle $BD HasSubtype ${ns}:$type
		} oldid]} {
		    set oldid [lindex $oldid 0]
		    if {[catch {delete $handle Node $oldid 1}]} {
			# Can't get rid of old definition, skip for now.
			continue
		    }
		}
		# attributes for new variable type
		set att [attrs default VariableTypeAttributes]
		dict set att DataType [types nodeid $handle $type]
		# node identifier for new variable type
		set vtid [_nodeid $handle $ns]
		# create the new variable type
		catch {
		    # can fail depending on server
		    add $handle VariableType $vtid $BD \
			HasSubtype "${ns}:$type" {} $att
		}
		if {[info $handle] eq "server"} {
		    # for the struct members add variables as components
		    # below the VariableType node.
		    if {$kind eq "struct"} {
			mapstruct $handle $vtid
		    }
		}
	    }
	}
	return $newtypes
    }

    # Internal helper: generate next numeric node identifier
    # given namespace index.

    proc _nodeid {handle nsidx} {
	# counter for node identifiers
	upvar \#0 ::opcua::${handle}::idcount idcount
	if {![::info exists idcount($nsidx)]} {
	    set idcount($nsidx) 56788
	}
	incr idcount($nsidx)
	while {1} {
	    set nodeid "ns=${nsidx};i=$idcount($nsidx)"
	    # retry, if identifier already exists
	    if {[catch {read $handle $nodeid NodeClass}]} {
		break
	    }
	    incr idcount($nsidx)
	}
	return $nodeid
    }

    # Define types in server (enums and structs).
    # Given handle must be a server object. The namespace URI
    # nsuri must have been created in the server object before.
    # The defs parameter contains struct definitions, e.g.
    #
    #  struct KVPair {
    #    String name
    #    String value
    #  }
    #  struct RGB {
    #    UInt16 red
    #    UInt16 green
    #    UInt16 blue
    #  }
    #  struct NamedColor {
    #    String name
    #    RGB color
    #  }
    #  struct WithArray {
    #    String name
    #    String *values
    #  }
    #  enum ColorComponents { Red Green Blue }
    #  union ManyThings {
    #    RGB color
    #    WithArray names
    #    KVPair pair
    #  }
    #  optstruct OptionalStuff {
    #    optional RGB color
    #    mandatory WithArray names
    #    optional KVPair pair
    #    mandatory Int32 number
    #    optional Double number2
    #  }

    proc deftypes {handle nsuri defs} {
	if {[info $handle] ne "server"} {
	    return -code error -errorcode {opcua Internal 0 Good} \
		"not a server handle"
	}
	set nsidx [namespace $handle $nsuri]
	set nsname [lindex [split [string trimright $nsuri "/"] "/"] end]
	# get /Root/Types/DataTypes/BaseDataType/Structure
	set TFS [_hxlate $handle Types/DataTypes/BaseDataType/Structure]
	# get /Root/Types/DataTypes/BaseDataType/Structure/Union
	set TFU [_hxlate $handle Types/DataTypes/BaseDataType/Structure/Union]
	# get /Root/Types/DataTypes/BaseDataType/Enumeration
	set TFE [_hxlate $handle Types/DataTypes/BaseDataType/Enumeration]
	# get /Root/Types/ObjectTypes/BaseObjectType/DataTypeEncodingType
	set TE [_hxlate $handle \
	    Types/ObjectTypes/BaseObjectType/DataTypeEncodingType]
	# get /Root/Types/VariableTypes/BaseVariableType/BaseDataVariableType/DataTypeDescriptionType
	set DD [_hxlate $handle \
	    Types/VariableTypes/BaseVariableType/BaseDataVariableType/DataTypeDescriptionType]
	# get /Root/Types/VariableTypes/BaseVariableType/PropertyType
	set PT [_hxlate $handle \
	    Types/VariableTypes/BaseVariableType/PropertyType]
	set bsd "<opc:TypeDictionary\n"
	append bsd " xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\"\n"
	append bsd " xmlns:tns=\"$nsuri\"\n"
	append bsd " DefaultByteOrder=\"LittleEndian\"\n"
	append bsd " xmlns:opc=\"http://opcfoundation.org/BinarySchema/\"\n"
	append bsd " xmlns:ua=\"http://opcfoundation.org/UA/\"\n"
	append bsd " TargetNamespace=\"$nsuri\">\n"
	append bsd " <opc:Import "
	append bsd "Namespace=\"http://opcfoundation.org/UA/\" />\n"
	# get /Root/Types/DataTypes/OPC Binary
	set OB [_hxlate $handle "Types/DataTypes/OPC Binary"]
	# get /Root/Types/VariableTypes/BaseVariableType/BaseDataVariableType/DataTypeDictionaryType
	set DT [_hxlate $handle \
	    Types/VariableTypes/BaseVariableType/BaseDataVariableType/DataTypeDictionaryType]
	# make node for *.bsd type defs
	set att [attrs default VariableAttributes]
	dict set att DataType [types nodeid ByteString]
	dict set att Value [list ByteString ""]
	dict set att ValueRank [const VALUERANK_SCALAR]
	set tt [add $handle Variable [_nodeid $handle $nsidx] $OB \
		    HasComponent "${nsidx}:${nsname}" $DT $att]
	# define types
	foreach {kind name def} $defs {
	    if {$kind ni {struct optstruct union typedef enum}} {
		continue
	    }
	    # create a struct, union, or enum type
	    if {$kind in {struct optstruct}} {
		if {$def eq {}} {
		    # forward declaration, skip it
		    continue
		}
		set tf $TFS
	    } elseif {$kind eq "union"} {
		if {$def eq {}} {
		    # forward declaration, skip it
		    continue
		}
		set tf $TFU
	    } elseif {$kind eq "typedef"} {
		set kind subtype
		# C style syntax, e.g. "typedef UInt16 WORD"
		# name is parent, def is the final type name
		if {[catch {types nodeid $handle $name} tf]} {
		    # assume it's already a node identifier
		    set tf $name
		}
		set name $def
	    } else {
		# must be an enum definition
		set tf $TFE
	    }
	    set ty [add $handle DataType [_nodeid $handle $nsidx] $tf \
			HasSubtype "${nsidx}:$name"]
	    if {$kind ne "subtype"} {
		# add encoding object for serialization
		set te [add $handle Object [_nodeid $handle $nsidx] {} \
			    {} "Default Binary" $TE]
		# inverse reference to struct, union, or enum
		add $handle Reference $te HasEncoding $ty 0
	    }
	    # provide type description (the *.bsd string)
	    if {$kind eq "struct"} {
		append bsd \
		    " <opc:StructuredType BaseType=\"ua:ExtensionObject\""
		append bsd " Name=\"[_xmlquote $name]\">\n"
		foreach {type field} $def {
		    if {$type in [types list]} {
			# assumes we have not imported other namespaces yet
			set type "opc:$type"
		    } else {
			# assumes type from this new namespace
			set type "tns:$type"
		    }
		    if {$type eq "opc:String"} {
			set type "opc:CharArray"
		    }
		    set isarray 0
		    if {[string index $field 0] eq "*"} {
			set field [string range $field 1 end]
			set isarray 1
		    } elseif {[string index $field 0] eq "!"} {
			set field [string range $field 1 end]
		    }
		    set field [_xmlquote $field]
		    set lf ""
		    if {$isarray} {
			set lf NoOf$field
			append bsd "  <opc:Field TypeName=\"opc:Int32\""
			append bsd " Name=\"$lf\" />\n"
			set lf " LengthField=\"$lf\""
		    }
		    append bsd "  <opc:Field TypeName=\"$type\""
		    append bsd "$lf Name=\"$field\" />\n"
		}
		append bsd " </opc:StructuredType>\n"
	    } elseif {$kind eq "optstruct"} {
		append bsd \
		    " <opc:StructuredType BaseType=\"ua:ExtensionObject\""
		append bsd " Name=\"[_xmlquote $name]\">\n"
		foreach {opt type field} $def {
		    if {$type in [types list]} {
			# assumes we have not imported other namespaces yet
			set type "opc:$type"
		    } else {
			# assumes type from this new namespace
			set type "tns:$type"
		    }
		    if {$type eq "opc:String"} {
			set type "opc:CharArray"
		    }
		    set isarray 0
		    if {[string index $field 0] eq "*"} {
			set field [string range $field 1 end]
			set isarray 1
		    } elseif {[string index $field 0] eq "!"} {
			set field [string range $field 1 end]
		    }
		    set field [_xmlquote $field]
		    switch -exact -- $opt {
			o - op - opt - opti - optio - option - optiona -
			optional { set opt 1 }
			default  { set opt 0 }
		    }
		    set swf ""
		    if {$opt} {
			set swf ${field}Specified
			append bsd "  <opc:Field TypeName=\"opc:Bit\""
			append bsd " Name=\"$swf\" />\n"
			set swf " SwitchField=\"$swf\""
		    }
		    set lf ""
		    if {$isarray} {
			set lf NoOf$field
			append bsd "  <opc:Field TypeName=\"opc:Int32\""
			append bsd " Name=\"$lf\" />\n"
			set lf " LengthField=\"$lf\""
		    }
		    append bsd "  <opc:Field TypeName=\"$type\""
		    append bsd "${lf}${swf} Name=\"$field\" />\n"
		}
		append bsd " </opc:StructuredType>\n"
	    } elseif {$kind eq "union"} {
		append bsd \
		    " <opc:StructuredType BaseType=\"ua:Union\""
		append bsd " Name=\"[_xmlquote $name]\">\n"
		append bsd "  <opc:Field TypeName=\"opc:UInt32\""
		append bsd " Name=\"SwitchField\" />\n"
		set count 1
		foreach {type field} $def {
		    if {$type in [types list]} {
			# assumes we have not imported other namespaces yet
			set type "opc:$type"
		    } else {
			# assumes type from this new namespace
			set type "tns:$type"
		    }
		    if {$type eq "opc:String"} {
			set type "opc:CharArray"
		    }
		    set isarray 0
		    if {[string index $field 0] eq "*"} {
			set field [string range $field 1 end]
			set isarray 1
		    } elseif {[string index $field 0] eq "!"} {
			set field [string range $field 1 end]
		    }
		    set field [_xmlquote $field]
		    set lf ""
		    if {$isarray} {
			set lf NoOf$field
			append bsd "  <opc:Field TypeName=\"opc:Int32\""
			append bsd " Name=\"$lf\" />\n"
			set lf " LengthField=\"$lf\""
		    }
		    append bsd "  <opc:Field TypeName=\"$type\""
		    append bsd " SwitchField=\"SwitchField\""
		    append bsd " SwitchValue=\"$count\""
		    append bsd "$lf Name=\"$field\" />\n"
		    incr count
		}
		append bsd " </opc:StructuredType>\n"
	    } elseif {$kind eq "subtype"} {
		append bsd " <opc:OpaqueType"
		append bsd " Name=\"[_xmlquote $name]\" />\n"
		# subtype needs no more information
		continue
	    } else {
		append bsd \
		    " <opc:EnumeratedType BaseType=\"ua:ExtensionObject\""
		append bsd " Name=\"[_xmlquote $name]\""
		append bsd " LengthInBits=\"32\">\n"
		set enval 0
		set ensl [list LocalizedText]
		foreach ename $def {
		    append bsd "  <opc:EnumeratedValue"
		    append bsd " Name=\"[_xmlquote $ename]\""
		    append bsd " Value=\"[_xmlquote $enval]\"/>\n"
		    incr enval
		    set ens1 [types empty LocalizedText]
		    dict set ens1 text $ename
		    lappend ensl $ens1
		}
		append bsd " </opc:EnumeratedType>\n"
		set att [attrs default VariableAttributes]
		dict set att DataType [types nodeid LocalizedText]
		dict set att Value $ensl
		dict set att ValueRank [const VALUERANK_ONE_DIMENSION]
		dict set att ArrayDimensions $enval
		add $handle Variable [_nodeid $handle $nsidx] $ty \
		    HasProperty "EnumStrings" $PT $att
		# enum needs no more information
		continue
	    }
	    # add struct below namespace
	    set att [attrs default VariableAttributes]
	    dict set att DataType [types nodeid String]
	    dict set att Value [list String $name]
	    dict set att ValueRank [const VALUERANK_SCALAR]
	    set tx [add $handle Variable [_nodeid $handle $nsidx] $tt \
			HasComponent "${nsidx}:$name" $DD $att]
	    # add reference to encoding object
	    add $handle Reference $te HasDescription $tx 1
	}
	append bsd "</opc:TypeDictionary>\n"
	# store *.bsd type defs into node
	write $handle $tt ByteString $bsd
	# and store the namespace URI as property
	set att [attrs default VariableAttributes]
	dict set att DataType [types nodeid String]
	dict set att Value [list String $nsuri]
	dict set att ValueRank [const VALUERANK_SCALAR]
	add $handle Variable [_nodeid $handle $nsidx] $tt \
	    HasProperty "NamespaceUri" $PT $att
    }

    # Helper functions to initialize dicts for node attributes and
    # other data types. It wraps "dict with". The "body" parameter
    # is executed in a call frame of its own in caller's namespace
    # on a fresh and empty dict of the requested type.

    proc attr_init {attrtype body args} {
	set ns [uplevel 1 ::namespace current]
	tailcall apply [list {__ _ args} {
	    dict with __ $_
	    return $__
	} $ns] [attrs default $attrtype] $body {*}$args
    }

    proc dict_init {type body args} {
	set ns [uplevel 1 ::namespace current]
	tailcall apply [list {__ _ args} {
	    dict with __ $_
	    return $__
	} $ns] [types empty $type] $body {*}$args
    }

}

#############################################################################
# Node set loader. Support procedures are prefixed with "_ld".

namespace eval ::opcua {

    # Internal helper: get boolean XML attribute.

    proc _ld_getboolatt {node att def} {
	set ret $def
	if {[$node hasAttribute $att]} {
	    set ret [expr {[$node getAttribute $att] ? 1 : 0}]
	}
	return $ret
    }

    # Internal helper: set node reference info from XML dom node.

    proc _ld_noderefs {node dict} {
	upvar $dict d
	set ret {}
	foreach ref [$node selectNodes References/Reference] {
	    dict set r forward [_ld_getboolatt $ref IsForward 1]
	    dict set r type [$ref getAttribute ReferenceType]
	    dict set r nodeid [[$ref firstChild] data]
	    lappend ret $r
	    unset r
	}
	dict set d refs $ret
    }

    # Internal helper: set node data type definition from XML dom node.

    proc _ld_nodedefs {node dict al} {
	variable _loptsetid
	upvar $dict d
	upvar $al alias
	set ret {}
	set optset 0
	set isunion 0
	set isprim 1
	set subtype {}
	foreach def [$node selectNodes Definition] {
	    set name [$def getAttribute Name]
	    if {[$def hasAttribute IsOptionSet]} {
		set optset [$def getAttribute IsOptionSet]
	    }
	    if {[$def hasAttribute IsUnion]} {
		set isunion [$def getAttribute IsUnion]
	    }
	    set isprim 0
	    break
	}
	if {$optset || $isprim} {
	    # check for references to an integer type to decide
	    # if an enum or integer subtype is more appropriate
	    foreach ref [dict get $d refs] {
		dict with ref {
		    if {($type eq "HasSubtype") && !$forward} {
			if {$nodeid in {
				SByte Byte Int16 UInt16 Int32 UInt32
				Int64 UInt64
			}} {
			    set optset 0
			    set subtype [types nodeid $nodeid]
			    break
			}
			if {($nodeid eq [types nodeid SByte]) ||
			    ($nodeid eq [types nodeid Byte]) ||
			    ($nodeid eq [types nodeid Int16]) ||
			    ($nodeid eq [types nodeid UInt16]) ||
			    ($nodeid eq [types nodeid Int32]) ||
			    ($nodeid eq [types nodeid UInt32]) ||
			    ($nodeid eq [types nodeid Int64]) ||
			    ($nodeid eq [types nodeid UInt64])} {
			    set optset 0
			    set subtype $nodeid
			    break
			}
			if {$isprim} {
			    if {$nodeid in {String Double Float}} {
				set optset 0
				set subtype [types nodeid $nodeid]
				break
			    }
			    if {($nodeid eq [types nodeid String]) ||
				($nodeid eq [types nodeid Double]) ||
				($nodeid eq [types nodeid Float])} {
				set optset 0
				set subtype $nodeid
				break
			    }
			}
		    }
		}
	    }
	} elseif {[::info exists _loptsetid]} {
	    # check if we have "OptionSet" as supertype
	    foreach ref [dict get $d refs] {
		dict with ref {
		    if {($type eq "HasSubtype") && !$forward} {
			if {$nodeid eq $_loptsetid} {
			    set optset 1
			    break
			}
		    }
		}
	    }
	}
	if {$isprim && ($subtype ne {})} {
	    set optset 0
	    set name [$node getAttribute BrowseName]
	    lappend ret subtype $name $subtype
	}
	set nisopt 0
	foreach field [$node selectNodes Definition/Field] {
	    if {$optset} {
		# struct
		# TBD: collect valid bits
		continue
	    }
	    if {[$field hasAttribute Value]} {
		# enum
		dict set r name [$field getAttribute Name]
		dict set r value [$field getAttribute Value]
		if {$ret eq {}} {
		    if {$subtype ne {}} {
			lappend ret subtype $name $subtype
		    } else {
			lappend ret enum $name
		    }
		}
	    } elseif {[$field hasAttribute DataType]} {
		# struct/union
		dict set r name [$field getAttribute Name]
		set type [$field getAttribute DataType]
		if {[::info exists alias($type)]} {
		    set type $alias($type)
		}
		dict set r type $type
		set isopt 0
		if {[$field hasAttribute IsOptional]} {
		    set isopt [expr {[$field getAttribute IsOptional] ? 1 : 0}]
		}
		dict set r isopt $isopt
		if {$isopt} {
		    incr nisopt
		}
		set isarr 0
		if {[$field hasAttribute ValueRank]} {
		    set isarr [expr {[$field getAttribute ValueRank] > 0}]
		    if {$isarr} {
			if {[$field hasAttribute ArrayDimensions]} {
			    set dim [split \
				[$field getAttribute ArrayDimensions] ","]
			    dict set r dim $dim
			}
		    }
		}
		dict set r isarray $isarr
		if {$ret eq {}} {
		    if {$isunion} {
			lappend ret union
		    } else {
			lappend ret struct
		    }
		    lappend ret $name
		}
	    } else {
		# TBD: what now?
		continue
	    }
	    if {[lindex $ret 0] ne "subtype"} {
		lappend ret $r
	    }
	    unset r
	}
	if {$optset} {
	    # TBD: valid bits from above, alternate typedef syntax?
	    lappend ret struct $name
	    lappend ret [dict create name Value type ByteString \
		isopt 0 isarray 0]
	    lappend ret [dict create name ValidBits type ByteString \
		isopt 0 isarray 0]
	}
	if {$nisopt} {
	    # make optstruct when struct has optional fields
	    set ret [lreplace $ret 0 0 optstruct]
	}
	if {$ret ne {}} {
	    dict set d typedef $ret
	}
    }

    # Internal helper: set node's browse/display names etc. from XML dom node.

    proc _ld_nodeatts {node dict} {
	upvar $dict d
	set brname [$node getAttribute BrowseName]
	set dname {}
	foreach child [$node selectNodes DisplayName] {
	    set dname [[$child firstChild] data]
	    break
	}
	if {$dname eq {}} {
	    set n [string first ":" $brname]
	    if {$n >= 0} {
		set dname [string range $brname $n+1 end]
	    } else {
		set dname $brname
	    }
	}
	dict set d brname $brname
	dict set d dname $dname
	if {[$node hasAttribute AccessLevel]} {
	    dict set d alevel [$node getAttribute AccessLevel]
	}
	if {[$node hasAttribute UserAccessLevel]} {
	    dict set d ulevel [$node getAttribute UserAccessLevel]
	}
	if {[$node hasAttribute Historizing]} {
	    dict set d hist [$node getAttribute Historizing]
	}
	if {[$node hasAttribute EventNotifier]} {
	    dict set d event [$node getAttribute EventNotifier]
	}
	if {[$node hasAttribute InverseName]} {
	    dict set d iname [$node getAttribute InverseName]
	}
	if {[$node hasAttribute ParentNodeId]} {
	    dict set d parent [$node getAttribute ParentNodeId]
	}
	foreach child [$node selectNodes Description] {
	    set child [$child firstChild]
	    if {$child ne {}} {
		dict set d descr [$child data]
	    }
	    break
	}
    }

    # Internal helper: add parent to node dict if needed,
    # sets the first reverse reference as parent node and
    # adds dict key "refid" with the node id of the reference.

    proc _ld_fixparent {handle dict narray} {
	upvar $dict d
	if {![dict exists $d parent]} {
	    foreach r [dict get $d refs] {
		if {![dict get $r forward]} {
		    # Use only hierarchical references as parent,
		    # otherwise leave key "parent" empty.
		    if {[dict get $r type] ni {HasModellingRule
			HasTypeDefinition HasEncoding HasDescription
			GeneratesEvent NonHierarchicalReferences}} {
			dict set d parent [dict get $r nodeid]
			break
		    }
		}
	    }
	    if {![dict exists $d parent]} {
		dict set d parent {}
	    }
	}
	if {![dict exists $d refid]} {
	    set parent [dict get $d parent]
	    foreach r [dict get $d refs] {
		if {([dict get $r nodeid] eq $parent) &&
		    ![dict get $r forward]} {
		    set type [dict get $r type]
		    if {![catch {reftype $type} rtype]} {
			set type $rtype
		    }
		    dict set d refid $type
		    break
		}
	    }
	    if {![dict exists $d refid]} {
		# try parent's forward references to this node
		upvar $narray nodes
		set me [dict get $d nodeid]
		if {[::info exists nodes($parent)]} {
		    foreach r [dict get $nodes($parent) refs] {
			if {([dict get $r nodeid] eq $me) &&
			    [dict get $r forward]} {
			    set type [dict get $r type]
			    if {![catch {reftype $type} rtype]} {
				set type $rtype
			    }
			    dict set d refid $type
			    break
			}
		    }
		}
		if {![dict exists $d refid]} {
		    dict set d refid {}
		}
	    }
	}
	if {![dict exists $d typeref]} {
	    switch -- [dict get $d kind] {
		Object - Variable - VariableType {
		    foreach r [dict get $d refs] {
			if {[dict get $r type] eq "HasTypeDefinition"} {
			    dict set d typeref [dict get $r nodeid]
			    break
			}
		    }
		}
	    }
	}
	if {![dict exists $d type]} {
	    switch -- [dict get $d kind] {
		Object {
		    foreach r [dict get $d refs] {
			if {[dict get $r type] eq "HasTypeDefinition"} {
			    dict set d type [dict get $r nodeid]
			    break
			}
		    }
		}
		VariableType {
		    foreach r [dict get $d refs] {
			if {[dict get $r type] eq "HasSubtype"} {
			    if {[catch {
				read $handle [dict get $r nodeid] DataType
			    } type]} {
				set type [dict get $r nodeid]
			    }
			    dict set d type $type
			    break
			}
		    }
		}
	    }
	}
    }

    # Internal helper: fix namespace indices in node info dicts.
    #  vn:    name of dict with node info
    #  vm:    array with namespace index from->to mapping
    #  al:    array with alias sym->node mapping
    #  nkeys: list of nodeid fields to fix
    #  pkeys: list of qualified names to fix

    proc _ld_fixns {vn vm al nkeys pkeys} {
	upvar $vn n
	upvar $vm m
	upvar $al alias
	foreach k $nkeys {
	    if {[dict exists $n $k]} {
		set v [dict get $n $k]
		if {[::info exists alias($v)]} {
		    dict set n $k $alias($v)
		} else {
		    foreach mm [array names m] {
			if {[string match "ns=$mm;*" $v]} {
			    regsub -- "^ns=$mm;" $v "ns=$m($mm);" v
			    dict set n $k $v
			    break
			}
		    }
		}
	    }
	}
	foreach k $pkeys {
	    if {[dict exists $n $k]} {
		set v [dict get $n $k]
		foreach mm [array names m] {
		    if {[string match "$mm:*" $v]} {
			regsub -- "^$mm:" $v "$m($mm):" v
			dict set n $k $v
			break
		    }
		}
	    }
	}
	# special case for values of ExtensionObject
	if {[dict exists $n value] && [dict exists $n fixnsv]} {
	    set v [dict get $n value]
	    if {[dict get $n fixnsv] < 0} {
		# scalar
		set t [dict get $v DataType]
		foreach mm [array names m] {
		    if {[string match "ns=$mm;*" $t]} {
			regsub -- "^ns=$mm;" $t "ns=$m($mm);" t
			dict set v DataType $t
			dict set n value $v
			break
		    }
		}
	    } else {
		# list
		set nv {}
		foreach el $v {
		    set t [dict get $el DataType]
		    foreach mm [array names m] {
			if {[string match "ns=$mm;*" $t]} {
			    regsub -- "^ns=$mm;" $t "ns=$m($mm);" t
			    dict set el DataType $t
			    break
			}
		    }
		    lappend nv $el
		}
		dict set n value $nv
	    }
	}
	# special case for values of QualifiedName or NodeId
	if {[dict exists $n value] && [dict exists $n fixnsq]} {
	    set v [dict get $n value]
	    if {[dict get $n fixnsq] < 0} {
		# scalar
		foreach mm [array names m] {
		    if {[string match "$mm:*" $v]} {
			regsub -- "^$mm:" $v "$m($mm):" v
			dict set n value $v
			break
		    } elseif {[string match "ns=$mm;*" $v]} {
			regsub -- "^ns=$mm;" $v "ns=$m($mm);" v
			dict set n value $v
			break
		    }
		}
	    } else {
		# list
		set nv {}
		foreach el $v {
		    foreach mm [array names m] {
			if {[string match "$mm:*" $el]} {
			    regsub -- "^$mm:" $el "$m($mm):" el
			    break
			} elseif {[string match "ns=$mm;*" $el]} {
			    regsub -- "^ns=$mm;" $el "ns=$m($mm);" el
			    break
			}
		    }
		    lappend nv $el
		}
		dict set n value $nv
	    }
	}
    }

    # Internal helper: recursively convert XML node to given dict variable.

    proc _ld_xn2d {node dict} {
	upvar $dict d
	if {[$node hasChildNodes]} {
	    set val {}
	    foreach child [$node childNodes] {
		if {[$child nodeType] eq "TEXT_NODE"} {
		    dict set d [$node nodeName] [$child nodeValue]
		    return
		}
		_ld_xn2d $child val
	    }
	} elseif {[$node nodeType] eq "TEXT_NODE"} {
	    set val [$node nodeValue]
	} else {
	    set val {}
	}
	dict set d [$node nodeName] $val
    }

    # Internal helper: fix dictionary for missing stuff in
    # Argument and EnumValueType structures.

    proc _ld_x2d_fix {dict} {
	upvar $dict d
	if {[dict exists $d DataType]} {
	    catch {
		dict set d DataType [dict get $d DataType Identifier]
	    }
	}
	foreach key {Description DisplayName} {
	    set nd {}
	    set lcl ""
	    set txt ""
	    catch {set lcl [dict get $d $key Locale]}
	    catch {set txt [dict get $d $key Text]}
	    dict set nd locale $lcl
	    dict set nd text $txt
	    dict set d $key $nd
	}
	if {[dict exists $d ArrayDimensions]} {
	    set dim [dict get $d ArrayDimensions]
	    if {[llength $dim] > 1} {
		# strip off type (UInt32 typically)
		dict set d ArrayDimensions [lindex $dim 1]
	    }
	}
    }

    # Internal helper: convert XML string to dict whose value is returned.

    proc _ld_x2d {xml} {
	set d {}
	set doc [dom parse -ignorexmlns -- $xml]
	set root [$doc documentElement]
	set item [$root selectNodes {/ExtensionObject/Body[1]}]
	set item [$item firstChild]
	_ld_xn2d $item d
	set d [dict get $d [dict keys $d]]
	_ld_x2d_fix d
	$doc delete
	return $d
    }

    # Internal helper: like _ld_x2d but processing a list of items.

    proc _ld_x2d_list {xml} {
	set dl {}
	set doc [dom parse -ignorexmlns -- $xml]
	set root [$doc documentElement]
	foreach child [$root selectNodes {ExtensionObject/Body[1]}] {
	    set child [$child firstChild]
	    set d {}
	    _ld_xn2d $child d
	    set d [dict get $d [dict keys $d]]
	    _ld_x2d_fix d
	    lappend dl $d
	}
	$doc delete
	return $dl
    }

    # Internal helper: rewrite typedef and make *.bsd fragment for it

    proc _ld_rewrite_td {handle tdvar alvar} {
	upvar $tdvar typedef
	upvar $alvar alias
	set bsd {}
	set stuff [lassign $typedef kind tname]
	set ttname $tname
	# omit namespace prefix in type name attributes
	scan $ttname "%d:%s" _ ttname
	switch -exact -- $kind {
	    enum {
		set td enum
		lappend td $tname
		set bsd "<opc:EnumeratedType LengthInBits=\"32\""
		append bsd " Name=\"[_xmlquote $ttname]\">"
		foreach d $stuff {
		    dict with d {
			lappend td $name $value
			append bsd "\n <opc:EnumeratedValue"
			append bsd " Name=\"[_xmlquote $name]\""
			append bsd " Value=\"[_xmlquote $value]\">"
		    }
		}
		append bsd "\n</opc:EnumeratedType>\n"
		set typedef $td
	    }
	    struct - optstruct {
		set td $kind
		lappend td $tname
		set bsd "<opc:StructuredType BaseType=\"ua:ExtensionObject\""
		append bsd " Name=\"[_xmlquote $ttname]\">"
		foreach d $stuff {
		    dict with d {
			if {[::info exists alias($type)]} {
			    set atype $type
			    set type $alias($type)
			}
			if {$kind eq "optstruct"} {
			    lappend td \
				[expr {$isopt ? "optional" : "mandatory"}]
			}
			if {$isarray} {
			    lappend td $type *$name
			} else {
			    lappend td $type $name
			}
			if {![catch {types name $handle $type} ntype]} {
			    set type $ntype
			} elseif {[scan $type "ns=%d;%s" nsidx ntype] == 2} {
			    if {[::info exists atype]} {
				set type ${nsidx}:$atype
			    }
			}
			# TBD: proper XML namespace handling?
			if {([string first : $type] < 0) &&
			    ![string match "ns=*;*" $type]} {
			    set type "opc:$type"
			}
			if {$type eq "opc:String"} {
			    set type "opc:CharArray"
			}
			set field [_xmlquote $name]
			set swf ""
			if {$isopt} {
			    set swf ${field}Specified
			    append bsd "\n <opc:Field TypeName=\"opc:Bit\""
			    append bsd " Name=\"$swf\" >"
			    set swf " SwitchField=\"$swf\""
			}
			set lf ""
			if {$isarray} {
			    set lf NoOf$field
			    append bsd "\n <opc:Field TypeName=\"opc:Int32\""
			    append bsd " Name=\"$lf\" />"
			    set lf " LengthField=\"$lf\""
			}
			append bsd "\n <opc:Field"
			append bsd " TypeName=\"[_xmlquote $type]\""
			append bsd "${lf}${swf} Name=\"$field\" />"
		    }
		}
		append bsd "\n</opc:StructuredType>\n"
		set typedef $td
	    }
	    union {
		set td union
		lappend td $tname
		set bsd "<opc:StructuredType BaseType=\"ua:Union\""
		append bsd " Name=\"[_xmlquote $ttname]\">"
		append bsd "\n <opc:Field TypeName=\"opc:UInt32\""
		append bsd " Name=\"SwitchField\" />"
		set count 1
		foreach d $stuff {
		    dict with d {
			if {[::info exists alias($type)]} {
			    set atype $type
			    set type $alias($type)
			}
			if {$isarray} {
			    lappend td $type *$name
			} else {
			    lappend td $type $name
			}
			if {![catch {types name $handle $type} ntype]} {
			    set type $ntype
			} elseif {[scan $type "ns=%d;%s" nsidx ntype] == 2} {
			    if {[::info exists atype]} {
				set type ${nsidx}:$atype
			    }
			}
			# TBD: proper XML namespace handling?
			if {([string first : $type] < 0) &&
			    ![string match "ns=*;*" $type]} {
			    set type "opc:$type"
			}
			if {$type eq "opc:String"} {
			    set type "opc:CharArray"
			}
			append bsd "\n <opc:Field SwitchField=\"SwitchField\""
			append bsd " SwitchValue=\"$count\""
			append bsd " TypeName=\"[_xmlquote $type]\""
			if {$isarray} {
			    append bsd " LengthField=\"[_xmlquote NoOf$name]\""
			}
			append bsd " Name=\"[_xmlquote $name]\" />"
			incr count
		    }
		}
		append bsd "\n</opc:StructuredType>\n"
		set typedef $td
	    }
	}
	return $bsd
    }

    # Internal helper: perform a typedef and gather *.bsd information
    # in an array ("bsds") indexed by namespace index. Only succeeding
    # typedefs add to that array. Similary the "bsdn" array collects the
    # typedef names and encoding identifiers.

    proc _ld_tdef {handle bsds bsdn tnode typedef} {
	upvar $bsds bsdout
	upvar $bsdn bsdnam
	if {![catch {typedef $handle {*}$typedef} err]} {
	    if {[dict exists $tnode bsd_nsidx]} {
		set nsidx [dict get $tnode bsd_nsidx]
		append bsdout($nsidx) [dict get $tnode bsd]
		lassign $typedef kind name nodeid encid
		if {$kind in {optstruct struct union}} {
		    lappend bsdnam($nsidx) $name $encid
		}
	    }
	    return 1
	}
	return 0
    }

    # Internal helper: convert value from XML according to type.

    proc _ld_convval {type node data fixns} {
	upvar $fixns fixnsq
	switch -glob -- $type {
	    *ByteString {
		# try to convert from base64
		if {![catch {binary decode base64 $data} ndata]} {
		    set data $ndata
		}
	    }
	    *LocalizedText {
		if {$node ne {}} {
		    set lcl ""
		    set txt ""
		    while {$node ne {}} {
			switch -glob -- [$node nodeName] {
			    *Locale {
				catch {
				    set lcl \
					[string trim [[$node firstChild] data]]
				}
			    }
			    *Text {
				catch {
				    set txt \
					[string trim [[$node firstChild] data]]
				}
			    }
			}
			set node [$node nextSibling]
		    }
		    dict set dd locale $lcl
		    dict set dd text $txt
		    set data $dd
		}
	    }
	    *QualifiedName {
		if {$node ne {}} {
		    set idx 0
		    set nam ""
		    while {$node ne {}} {
			switch -glob -- [$node nodeName] {
			    *NamespaceIndex {
				catch {
				    scan [[$node firstChild] data]] %d idx
				}
			    }
			    *Name {
				catch {
				    set nam [string trim \
					[[$node firstChild] data]]
				}
			    }
			}
			set node [$node nextSibling]
		    }
		    if {$idx != 0} {
			set data ${idx}:$nam
			incr fixnsq
		    } else {
			set data $nam
		    }
		}
	    }
	    *NodeId {
		if {$node ne {}} {
		    set id {}
		    while {$node ne {}} {
			switch -glob -- [$node nodeName] {
			    *Identifier {
				catch {
				    set id [string trim \
					[[$node firstChild] data]]
				}
			    }
			}
			set node [$node nextSibling]
		    }
		    if {$id ne {}} {
			set data $id
			incr fixnsq
		    }
		}
	    }
	}
	return $data
    }

    # Internal helper: write value attribute into node,
    # on error recursively try super types, if any.
    # Requires dict value with full node information.

    proc _ld_wrvalue {handle node} {
	dict with node {
	    if {$islist} {
		if {[::info exists dim] && ($dim ne {})} {
		    # For 1-dim try first without explicit dimension in type
		    if {([llength $dim] == 1) && ![catch {
			    write $handle $nodeid Value *$type $value
		    }]} {
			return
		    }
		    set ty ([join $dim ,])$type
		} else {
		    set ty *$type
		}
	    } else {
		set ty $type
	    }
	}
	if {[catch {write $handle $nodeid Value $ty $value} err]} {
	    if {[read $handle $type IsAbstract]} {
		set test $value
		if {$islist} {
		    lassign $test test
		}
		switch -exact -- [read $handle $type BrowseName] {
		    Number {
			if {[string is wideinteger $test]} {
			    set xtype Int64
			} elseif {[string is double $test]} {
			    set xtype Double
			}
		    }
		    Integer {
			if {[string is wideinteger $test]} {
			    set xtype Int64
			}
		    }
		    UInteger {
			if {[string is wideinteger $test]} {
			    set xtype UInt64
			}
		    }
		}
		if {[::info exists xtype]} {
		    dict set node type $xtype
		    tailcall _ld_wrvalue $handle $node
		}
		return -code error $err
	    }
	    foreach {n b d c r t} [browse $handle $type Inverse HasSubtype] {
		dict set node type $n
		if {![catch {_ld_wrvalue $handle $node}]} {
		    return
		}
	    }
	    return -code error $err
	}
    }

    # Internal helper: create a node given handle and node dict.
    # Requires name of dict variable for upvar and optional
    # array variable to collect method information.

    proc _ld_mknode {handle dict {mvar {}}} {
	upvar $dict d
	dict with d {
	    set att [attrs default ${kind}Attributes]
	    if {[::info exists dname]} {
		dict set att DisplayName locale {}
		dict set att DisplayName text $dname
	    }
	    if {[::info exists descr]} {
		dict set att Description locale {}
		dict set att Description text $descr
	    }
	    switch -- $kind {
		Object {
		    if {![::info exists type]} {
			set type {}
		    }
		    if {[::info exists event]} {
			dict set att EventNotifier $event
		    }
		    add $handle Object \
			$nodeid $parent $refid $brname $type $att
		}
		Method {
		    # find old references to existing method(s)
		    if {[catch {
			translate $handle $parent $refid $brname
		    } mno]} {
			unset mno
		    }
		    # TBD: in/out args
		    set mns 0
		    scan $nodeid "ns=%d;" mns
		    set mname ::opcua::${handle}::impl${mns}_${dname}
		    add $handle Method $nodeid $parent $refid \
			{} $brname {} $mname $att
		    if {$mvar ne {}} {
			upvar $mvar mnames
			lappend mnames($mname) $nodeid
		    }
		    unset mns
		    # now delete old references
		    if {[::info exists mno]} {
			foreach {n _ _} $mno {
			    delete $handle Reference $parent $refid $n 1 1
			}
			unset mno
		    }
		}
		ReferenceType {
		    dict set att IsAbstract $abstract
		    dict set att Symmetric $symmetric
		    if {[::info exists iname]} {
			dict set att InverseName locale {}
			dict set att InverseName text $iname
		    }
		    add $handle ReferenceType \
			$nodeid $parent $refid $brname $att
		}
		Variable {
		    if {![::info exists typeref]} {
			set typeref {}
		    }
		    if {[::info exists type]} {
			dict set att DataType $type
		    }
		    if {[::info exists alevel]} {
			dict set att AccessLevel $alevel
		    }
		    if {[::info exists ulevel]} {
			dict set att UserAccessLevel $ulevel
		    }
		    if {[::info exists hist]} {
			dict set att Historizing $hist
		    }
		    if {[::info exists rank]} {
			dict set att ValueRank $rank
			if {($rank >= [const VALUERANK_ONE_OR_MORE_DIMENSIONS])
				&& [::info exists dim]} {
			    dict set att ArrayDimensions $dim
			}
		    }
		    add $handle Variable \
			$nodeid $parent $refid $brname $typeref $att
		    # TBD: more complex values
		    if {[::info exists type] && [::info exists value]} {
			_ld_wrvalue $handle $d
		    }
		}
		ObjectType {
		    dict set att IsAbstract $abstract
		    add $handle $kind $nodeid $parent $refid $brname $att
		}
		DataType {
		    dict set att IsAbstract $abstract
		    add $handle $kind $nodeid $parent $refid $brname $att
		    # TBD: handle XML <Definition>
		}
		VariableType {
		    if {![::info exists typeref]} {
			set typeref {}
		    }
		    if {[::info exists type]} {
			dict set att DataType $type
		    }
		    if {[::info exists rank]} {
			dict set att ValueRank $rank
			if {($rank >= [const VALUERANK_ONE_OR_MORE_DIMENSIONS])
				&& [::info exists dim]} {
			    dict set att ArrayDimensions $dim
			}
		    }
		    dict set att IsAbstract $abstract
		    add $handle VariableType \
			$nodeid $parent $refid $brname $typeref $att
		}
	    }
	}
    }

    # Capture new node identifiers while running nodeset loader.

    proc _ld_oninit {nodeid cls} {
	variable _lnodes
	set _lnodes($nodeid) 1
    }

    # Write loader trace messages using opcua log mechanism.

    proc _ld_log {msg} {
	variable _lts
	set log [log]
	if {$log eq {}} {
	    return
	}
	if {[::info exists _lts]} {
	    set msg "T=[expr {[clock milliseconds] - $_lts}]ms $msg"
	}
	::namespace eval :: $log trace loader [list $msg]
    }

    # Assign dict elements to variables, make non-existing keys
    # into empty variables.

    proc _dict_assign {dict args} {
	foreach var $args {
	    try {
		set val [dict get $dict $var]
	    } trap {TCL LOOKUP DICT} {} {
		set val {}
	    }
	    uplevel [list set $var $val]
	}
    }

    # Load nodeset from XML into a opcua server handle.
    # Optional error reports are written to the caller's variables
    # "evar" (for node information), "rvar" (for references),
    # and "tvar" (for type information).
    # Function returns a pairwise list of method names and
    # sublist of nodeids for the respective method name.

    proc loader {handle xml {evar {}} {rvar {}} {tvar {}}} {
	variable _lnodes
	variable _lts
	variable _loptsetid
	if {[info $handle] ne "server"} {
	    return -code error -errorcode {opcua Internal 0 Good} \
		"not a server handle"
	}
	if {[catch {package require tdom}]} {
	    return -code error -errorcode {opcua Internal 0 Good} \
		"required tdom package not available"
	}
	if {[lindex [state $handle] 0] ne "stopped"} {
	    return -code error -errorcode {opcua Internal 0 Good} \
		"server must be stopped"
	}
	if {$evar ne {}} {
	    upvar $evar ev
	}
	if {$rvar ne {}} {
	    upvar $rvar rv
	}
	if {$tvar ne {}} {
	    upvar $tvar tv
	}
	set _lts [clock milliseconds]
	_ld_log "Loader start, [string length $xml] XML chars"
	# ensure structure types from namespace zero are available
	_ns0structs $handle
	# capture identifiers of created nodes in _lnodes array
	array set _lnodes {}
	set oninit [oninitialize $handle]
	oninitialize $handle [::namespace current]::_ld_oninit
	# find identifier of "OptionSet" if available
	if {[catch {_hxlate $handle \
		Types/DataTypes/BaseDataType/Structure/OptionSet} _loptsetid]} {
	    unset _loptsetid
	}
	try {
	    set ret [_loader0 $handle $xml ev rv tv]
	} on error {result opts} {
	    _ld_log "Loader done, error \"$result\""
	    return -code error -options $opts $result
	} finally {
	    oninitialize $handle $oninit
	    # cleanup automatically created nodes, when _loader0
	    # fails this should remove all nodes created so far
	    foreach nodeid [array names _lnodes] {
		catch {delete $handle Node $nodeid 1}
	    }
	    unset _lnodes
	}
	_ld_log "Loader done, success"
	return $ret
    }

    proc _loader0 {handle xml {evar {}} {rvar {}} {tvar {}}} {
	variable _lnodes
	if {[catch {dom parse -ignorexmlns -- $xml} doc]} {
	    return -code error -errorcode {opcua Internal 0 Good} \
		"XML parse failed: $doc"
	}
	set root [$doc documentElement]
	_ld_log "XML document loaded"
	# load XML namespaces
	array set ns {0 http://opcfoundation.org/UA/}
	set nscount 1
	foreach nsuri [$root selectNodes /UANodeSet/NamespaceUris/Uri/text()] {
	    set nsuri [$nsuri data]
	    set ns($nscount) $nsuri
	    incr nscount
	}
	# find out our namespace
	set mainns 1
	foreach node [$root selectNodes /UANodeSet/Models/Model] {
	    set uri [$node getAttribute ModelUri]
	    foreach nscount [array names ns] {
		if {$ns($nscount) eq $uri} {
		    set mainns $nscount
		    break
		}
	    }
	    break
	}
	if {![::info exists ns($mainns)]} {
	    $doc delete
	    return -code error -errorcode {opcua Internal 0 Good} \
		"main XML namespace not found"
	}

	_ld_log "XML namespaces parsed"
	set mainnsuri $ns($mainns)
	# load aliases
	array set alias {}
	foreach node [$root selectNodes /UANodeSet/Aliases/Alias] {
	    set an [$node getAttribute Alias]
	    set alias($an) [[$node firstChild] data]
	    unset an
	}
	# provide alias for Argument typically used in InputArguments
	# and OutputArguments
	if {![::info exists alias(Argument)]} {
	    set alias(Argument) [types nodeid Argument]
	}
	_ld_log "XML aliases parsed"

	# load node information from XML
	array set nodes {}

	set kind Object
	_ld_log "Process XML $kind"
	set ncount 0
	foreach node [$root selectNodes /UANodeSet/UA$kind] {
	    dict set n kind $kind
	    set ni [$node getAttribute NodeId]
	    dict set n nodeid $ni
	    _ld_nodeatts $node n
	    _ld_noderefs $node n
	    _ld_fixparent $handle n nodes
	    set nodes($ni) $n
	    unset n ni
	    incr ncount
	}
	_ld_log "$ncount items processed"

	set kind Method
	_ld_log "Process XML $kind"
	set ncount 0
	foreach node [$root selectNodes /UANodeSet/UA$kind] {
	    dict set n kind $kind
	    set ni [$node getAttribute NodeId]
	    dict set n nodeid $ni
	    _ld_nodeatts $node n
	    _ld_noderefs $node n
	    _ld_fixparent $handle n nodes
	    set nodes($ni) $n
	    unset n ni
	    incr ncount
	}
	_ld_log "$ncount items processed"

	set kind ReferenceType
	_ld_log "Process XML $kind"
	set ncount 0
	foreach node [$root selectNodes /UANodeSet/UA$kind] {
	    dict set n kind $kind
	    set ni [$node getAttribute NodeId]
	    dict set n nodeid $ni
	    _ld_nodeatts $node n
	    _ld_noderefs $node n
	    dict set n symmetric [_ld_getboolatt $node Symmetric 0]
	    dict set n abstract [_ld_getboolatt $node IsAbstract 0]
	    _ld_fixparent $handle n nodes
	    set nodes($ni) $n
	    unset n ni
	    incr ncount
	}
	_ld_log "$ncount items processed"

	set kind Variable
	_ld_log "Process XML $kind"
	set ncount 0
	foreach node [$root selectNodes /UANodeSet/UA$kind] {
	    dict set n kind $kind
	    set ni [$node getAttribute NodeId]
	    dict set n nodeid $ni
	    _ld_nodeatts $node n
	    _ld_noderefs $node n
	    set vr [const VALUERANK_SCALAR]
	    if {[$node hasAttribute ValueRank]} {
		set vr [$node getAttribute ValueRank]
		set dim 0
		if {($vr >= [const VALUERANK_ONE_OR_MORE_DIMENSIONS]) &&
		    [$node hasAttribute ArrayDimensions]} {
		    set dim [$node getAttribute ArrayDimensions]
		    dict set n dim [split $dim ","]
		}
	    }
	    dict set n rank $vr
	    if {[$node hasAttribute DataType]} {
		set type [$node getAttribute DataType]
		set uty $type
		if {[::info exists alias($type)]} {
		    set uty $alias($type)
		}
		dict set n type $uty
		dict set n islist 0
		dict set n isext 0
		if {![catch {$node selectNodes Value} v] &&
		    ![catch {$v firstChild} v] && ($v ne {})} {
		    switch -glob -- [$v nodeName] {
			*ListOfExtensionObject {
			    set xml [$v asXML]
			    # mangle XML to have no namespace prefixes
			    set nn [split [$v nodeName] ":"]
			    set nnn ""
			    lassign $nn nns nnn
			    if {$nnn ne ""} {
				regsub -all "<$nns:" $xml "<" xml
				regsub -all "</$nns:" $xml "</" xml
			    }
			    set vd {}
			    set nnn 0
			    foreach el [_ld_x2d_list $xml] {
				lappend vd $el
				if {[dict exists $el DataType]} {
				    incr nnn
				}
			    }
			    dict set n value $vd
			    if {$nnn > 0} {
				dict set n fixnsv $nnn
			    }
			    dict set n islist 1
			    dict set n isext 1
			}
			*ExtensionObject {
			    set xml [$v asXML]
			    # mangle XML to have no namespace prefixes
			    set nn [split [$v nodeName] ":"]
			    set nnn ""
			    lassign $nn nns nnn
			    if {$nnn ne ""} {
				regsub -all "<$nns:" $xml "<" xml
				regsub -all "</$nns:" $xml "</" xml
			    }
			    set vd [_ld_x2d $xml]
			    if {$vr >= [const VALUERANK_ONE_OR_MORE_DIMENSIONS]} {
				dict set n value [list $vd]
				if {[dict exists $vd DataType]} {
				    dict set n fixnsv 1
				}
			    } else {
				dict set n value $vd
				if {[dict exists $vd DataType]} {
				    dict set n fixnsv -1
				}
			    }
			    dict set n isext 1
			}
			*ListOf* {
			    set lv {}
			    set fixnsq 0
			    foreach vv [$v childNodes] {
				set vtype [$vv nodeName]
				if {![catch {$vv firstChild} vvv]} {
				    if {[catch {$vvv data} data]} {
					set data {}
				    }
				    lappend lv \
					[_ld_convval $vtype $vvv $data fixnsq]
				} else {
				    lappend lv {}
				}
			    }
			    dict set n value $lv
			    dict set n islist 1
			    if {$fixnsq > 0} {
				dict set n fixnsq $fixnsq
			    }
			}
			default {
			    if {![catch {$v firstChild} vv]} {
				if {[catch {$vv data} data]} {
				    set data {}
				}
				set fixnsq 0
				set vvv [_ld_convval $type $vv $data fixnsq]
				if {$vr >= [const VALUERANK_ONE_OR_MORE_DIMENSIONS]} {
				    dict set n value [list $vvv]
				    if {$fixnsq} {
					dict set n fixnsq $fixnsq
				    }
				} else {
				    dict set n value $vvv
				    if {$fixnsq} {
					dict set n fixnsq -1
				    }
				}
			    }
			}
		    }
		}
		# Various tweaks to increase probability that
		# creation of Variable later works.
		if {[dict get $n islist]} {
		    if {$vr < [const VALUERANK_ONE_OR_MORE_DIMENSIONS]} {
			set vr [const VALUERANK_ONE_OR_MORE_DIMENSIONS]
			dict set n rank $vr
		    } elseif {($vr == [const VALUERANK_ONE_DIMENSION])} {
			dict set n dim [llength [dict get $n value]]
		    }
		}
		if {![dict exists $n value] &&
		    ($vr >= [const VALUERANK_ONE_DIMENSION])} {
		    dict set n rank [const VALUERANK_ONE_OR_MORE_DIMENSIONS]
		    dict unset n dim
		}
	    }
	    _ld_fixparent $handle n nodes
	    set nodes($ni) $n
	    unset n ni
	    incr ncount
	}
	_ld_log "$ncount items processed"

	set kind ObjectType
	_ld_log "Process XML $kind"
	set ncount 0
	foreach node [$root selectNodes /UANodeSet/UA$kind] {
	    dict set n kind $kind
	    set ni [$node getAttribute NodeId]
	    dict set n nodeid $ni
	    _ld_nodeatts $node n
	    _ld_noderefs $node n
	    dict set n abstract [_ld_getboolatt $node IsAbstract 0]
	    _ld_fixparent $handle n nodes
	    set nodes($ni) $n
	    unset n ni
	    incr ncount
	}
	_ld_log "$ncount items processed"

	set kind DataType
	_ld_log "Process XML $kind"
	set ncount 0
	foreach node [$root selectNodes /UANodeSet/UA$kind] {
	    dict set n kind $kind
	    set ni [$node getAttribute NodeId]
	    dict set n nodeid $ni
	    _ld_nodeatts $node n
	    _ld_noderefs $node n
	    dict set n abstract [_ld_getboolatt $node IsAbstract 0]
	    _ld_nodedefs $node n alias
	    _ld_fixparent $handle n nodes
	    set nodes($ni) $n
	    unset n ni
	    incr ncount
	}
	_ld_log "$ncount items processed"

	set kind VariableType
	_ld_log "Process XML $kind"
	set ncount 0
	foreach node [$root selectNodes /UANodeSet/UA$kind] {
	    dict set n kind $kind
	    set ni [$node getAttribute NodeId]
	    dict set n nodeid $ni
	    _ld_nodeatts $node n
	    _ld_noderefs $node n
	    dict set n abstract [_ld_getboolatt $node IsAbstract 0]
	    set vr [const VALUERANK_SCALAR]
	    if {[$node hasAttribute ValueRank]} {
		set vr [$node getAttribute ValueRank]
		set dim 0
		if {($vr >= [const VALUERANK_ONE_OR_MORE_DIMENSIONS]) &&
		    [$node hasAttribute ArrayDimensions]} {
		    set dim [$node getAttribute ArrayDimensions]
		    dict set n dim [split $dim ","]
		}
	    }
	    dict set n rank $vr
	    if {[$node hasAttribute DataType]} {
		set type [$node getAttribute DataType]
		set uty $type
		if {[::info exists alias($type)]} {
		    set uty $alias($type)
		}
		dict set n type $uty
	    }
	    _ld_fixparent $handle n nodes
	    set nodes($ni) $n
	    unset n ni
	    incr ncount
	}
	_ld_log "$ncount items processed"

	# done with XML
	$doc delete
	_ld_log "XML processing finished"

	# create and find out namespaces
	array set sns {}
	foreach {nscount nsuri} [namespace $handle] {
	    set sns($nsuri) $nscount
	}
	if {$mainnsuri eq $ns(0)} {
	    # namespace zero is explicit
	} elseif {[::info exists sns($mainnsuri)]} {
	    return -code error -errorcode {opcua Internal 0 Good} \
		"namespace $mainnsuri exists"
	}
	foreach nscount [array names ns] {
	    set nn $ns($nscount)
	    if {($nscount != $mainns) && ![::info exists sns($nn)]} {
		# remember missing namespace for later
		set uns($nscount) $nn
	    }
	    unset nn
	}
	# create primary namespace
	if {$mainnsuri ne $ns(0)} {
	    set mainns [add $handle Namespace $mainnsuri]
	    _ld_log "Created primary namespace $mainns \"$mainnsuri\""
	}
	# create missing secondary namespaces
	foreach nscount [lsort -integer [array names uns]] {
	    set nn [add $handle Namespace $uns($nscount)]
	    _ld_log "Created namespace $nn \"$uns($nscount)\""
	}

	# get namespace mapping
	foreach {nscount nsuri} [namespace $handle] {
	    set sns($nsuri) $nscount
	}
	array set nsmap {}
	foreach nscount [array names ns] {
	    set nn $ns($nscount)
	    if {$sns($nn) != $nscount} {
		set nsmap($nscount) $sns($nn)
	    }
	    unset nn
	}
	if {![::info exists nsmap($mainns)]} {
	    set nsmap($mainns) $sns($mainnsuri)
	}
	_ld_log "Namespaces finished"

	# fix namespace indices in alias array
	foreach al [array names alias] {
	    if {[scan $alias($al) "ns=%d;%s" nsidx name] == 2} {
		set nn $ns($nsidx)
		if {$sns($nn) != $nsidx} {
		    set alias($al) "ns=$sns($nn);$name"
		}
	    }
	}

	# fix namespace indices in nodes array
	foreach nodeid [array names nodes] {
	    _ld_fixns nodes($nodeid) nsmap alias \
		{nodeid parent refid type typeref} brname
	    set refs [dict get $nodes($nodeid) refs]
	    set nrefs {}
	    foreach ref $refs {
		_ld_fixns ref nsmap alias {type nodeid} {}
		lappend nrefs $ref
	    }
	    dict set nodes($nodeid) refs $nrefs
	    unset refs nrefs
	}

	# fix qualified names in typedefs, rewrite for
	# typedef subcommand, and make *.bsd fragments of it
	foreach nodeid [array names nodes] {
	    if {[dict exists $nodes($nodeid) typedef]} {
		set typedef [dict get $nodes($nodeid) typedef]
		set name [lindex $typedef 1]
		set nsidx 0
		scan $name "%d:%s" nsidx name
		scan $nodeid "ns=%d;" nsidx
		if {$nsidx != 0} {
		    set nsidx $nsmap($nsidx)
		    set name ${nsidx}:${name}
		    set typedef [lreplace $typedef 1 1 $name]
		    # fix types in structs
		    if {[lindex $typedef 0] in {struct optstruct union}} {
			set nfields {}
			foreach field [lrange $typedef 2 end] {
			    _ld_fixns field nsmap alias type {}
			    lappend nfields $field
			}
			set typedef [lreplace $typedef 2 end {*}$nfields]
		    }
		    set bsd [_ld_rewrite_td $handle typedef alias]
		    dict set nodes($nodeid) typedef $typedef
		    if {$bsd ne {}} {
			dict set nodes($nodeid) bsd $bsd
			dict set nodes($nodeid) bsd_nsidx $nsidx
		    }
		}
	    }
	}

	# fix array names of nodes since namespace indices have changed
	array set nnodes {}
	foreach nodeid [array names nodes] {
	    set node $nodes($nodeid)
	    unset nodes($nodeid)
	    set nodeid [dict get $node nodeid]
	    set nnodes($nodeid) $node
	    unset node
	}
	array set nodes [array get nnodes]
	unset nnodes

	# try again to fix missing refids on nodes
	foreach nodeid [array names nodes] {
	    set node $nodes($nodeid)
	    if {[dict get $node refid] eq {}} {
		dict unset node refid
		_ld_fixparent $handle node nodes
		set nodes($nodeid) $node
	    }
	}

	# prime cache of existing nodeids
	array set exists {}
	foreach type [reftype] {
	    # must verify here since reduced nodeset builds don't
	    # necessarily have all reference types built in
	    if {![catch {read $handle [reftype $type] NodeClass}]} {
		set exists([reftype $type]) 1
	    }
	}
	foreach type [types basic] {
	    set exists([types nodeid $type]) 1
	}
	_ld_log "Namespace indexes fixed etc."

	# create nodes
	set max [array size nodes]
	set count 0
	array set refs {}
	array set errnodes {}
	array set meths {}
	array set vars {}
	array set typenodes {}
	while {$count < $max} {
	    # assumption: dictionary order especially for integer
	    # node identifiers makes least number of turn arounds
	    foreach nodeid [lsort -dictionary [array names nodes]] {
		set node $nodes($nodeid)
		_dict_assign $node parent refid type kind typeref
		if {($parent ne {}) && ![::info exists exists($parent)]} {
		    # query server for parent nodeid
		    if {[catch {read $handle $parent NodeClass}]} {
			continue
		    } else {
			set exists($parent) 1
		    }
		}
		if {($refid ne {}) && ![::info exists exists($refid)]} {
		    # query server for reference nodeid
		    if {[catch {read $handle $refid NodeClass}]} {
			continue
		    } else {
			set exists($refid) 1
		    }
		}
		if {($type ne {}) && ![::info exists exists($type)]} {
		    # query server for type nodeid
		    if {[catch {read $handle $type NodeClass}]} {
			continue
		    } else {
			set exists($type) 1
		    }
		}
		if {($typeref ne {}) && ![::info exists exists($typeref)]} {
		    # query server for typeref nodeid
		    if {[catch {read $handle $typeref NodeClass}]} {
			continue
		    } else {
			set exists($typeref) 1
		    }
		}
		# create node
		set mkrefs 0
		if {[catch {_ld_mknode $handle node meths} err]} {
		    set errnodes($nodeid) $node
		    dict set errnodes($nodeid) error $err
		    dict set errnodes($nodeid) ecode $::errorCode
		    # check and record node's existence
		    set nex [catch {read $handle $nodeid NodeClass} kind]
		    dict set errnodes($nodeid) exists [expr {!$nex}]
		    if {!$nex} {
			set exists($nodeid) 1
			set mkrefs 1
			if {[string match "*already used by another node" \
				$err] && ([dict get $node kind] eq $kind)} {
			    # assume double definition, no error report
			    unset errnodes($nodeid)
			}
		    } elseif {![dict exists $errnodes($nodeid) emax]} {
			dict set errnodes($nodeid) emax $max
		    }
		} else {
		    set exists($nodeid) 1
		    if {$kind in {Variable VariableType}} {
			set vars($nodeid) 1
		    }
		    set mkrefs 1
		}
		unset nodes($nodeid)
		if {[dict exists $node typedef]} {
		    # remember for later
		    set typenodes($nodeid) $node
		}
		if {$mkrefs} {
		    # create list of references to be made later
		    set nrefs {}
		    foreach ref [dict get $node refs] {
			if {[dict get $ref nodeid] eq $parent} {
			    continue
			}
			lappend nrefs $ref
		    }
		    if {[llength $nrefs]} {
			set refs($nodeid) $nrefs
		    }
		}
	    }
	    set nmax [array size nodes]
	    if {$nmax >= $max} {
		# no new nodes created, give up
		break
	    }
	    # next round with non-existing nodes with error
	    incr count [expr {$nmax - $max}]
	    set max $nmax
	    foreach nodeid [array names errnodes] {
		if {![dict get $errnodes($nodeid) exists]} {
		    set emax [dict get $errnodes($nodeid) emax]
		    if {$emax > $max} {
			set emax $max
		    } else {
			incr emax -1
		    }
		    dict set errnodes($nodeid) emax $emax
		    if {$emax >= 0} {
			set nodes($nodeid) $errnodes($nodeid)
			unset errnodes($nodeid)
			incr max
			incr count -1
		    }
		}
	    }
	    if {$max <= 0} {
		break
	    }
	    incr count
	}

	_ld_log "Nodes created"
	# remaining nodes known to have missing requirements
	foreach nodeid [array names nodes] {
	    set errnodes($nodeid) $nodes($nodeid)
	    if {![dict exists $errnodes($nodeid) error]} {
		dict set errnodes($nodeid) error {required node does not exist}
		dict set errnodes($nodeid) ecode \
		    [list opcua Loader 2147483648 [sc2str -short 2147483648]]
	    }
	    unset nodes($nodeid)
	}

	# create references
	array set unrefs {}
	foreach nodeid [array names refs] {
	    foreach ref $refs($nodeid) {
		set refid [dict get $ref nodeid]
		if {![::info exists exists($refid)]} {
		    # query server for reference nodeid
		    if {![catch {read $handle $refid NodeClass}]} {
			set exists($refid) 1
		    }
		}
		if {[::info exists exists($refid)] &&
		    ![::info exists nodes($refid)] &&
		    [::info exists exists($nodeid)]} {
		    set reftype [dict get $ref type]
		    if {![::info exists exists($reftype)]} {
			set unrefs($nodeid) $ref
			continue
		    }
		    # create reference
		    if {[catch {add $handle Reference $nodeid $reftype \
				    $refid [dict get $ref forward]} err]} {
			if {![string match "*already defined" $err]} {
			    return -code error $err
			}
		    }
		} else {
		    set unrefs($nodeid) $ref
		}
	    }
	    unset refs($nodeid)
	}
	_ld_log "References created"

	# create types from nodes
	set defs {}
	array set bsds {}
	array set bsdn {}
	# collect enum and struct definitions into "defs" as commands
	foreach nodeid [lsort -dictionary [array names typenodes]] {
	    if {![::info exists exists($nodeid)]} {
		continue
	    }
	    if {[dict exists $typenodes($nodeid) typedef]} {
		set typedef [dict get $typenodes($nodeid) typedef]
		lassign $typedef kind name
		if {$kind eq "enum"} {
		    foreach {supertype etype bits} {
			Enumeration Int32 32
			SByte SByte 8 Byte SByte 8
			Int16 Int16 16 UInt16 Int16 16
			Int32 Int32 32 UInt32 Int32 32
		    } {
			if {![catch {translate $handle $nodeid \
			    !HasSubtype $supertype} encid]} {
			    break
			}
			set encid {}
		    }
		    if {$encid eq {}} {
			continue
		    }
		    set encid [lindex $encid 0]
		    set typedef [linsert $typedef 2 $nodeid $encid $etype]
		    if {[regsub -- {LengthInBits="32"} \
			    [dict get $typenodes($nodeid) bsd] \
			    "LengthInBits=\"$bits\"" bsd]} {
			dict set typenodes($nodeid) bsd $bsd
		    }
		    lappend defs [list _ld_tdef $handle bsds bsdn \
				      $typenodes($nodeid) $typedef]
		    unset typenodes($nodeid)
		} elseif {$kind eq "subtype"} {
		    set typedef [linsert $typedef 2 $nodeid]
		    lappend defs [list _ld_tdef $handle bsds bsdn \
				      $typenodes($nodeid) $typedef]
		    unset typenodes($nodeid)
		} elseif {$kind in {struct optstruct union}} {
		    if {[catch {translate $handle $nodeid \
			HasEncoding "Default Binary"} encid]} {
			continue
		    }
		    set encid [lindex $encid 0]
		    set typedef [linsert $typedef 2 $nodeid $encid]
		    lappend defs [list _ld_tdef $handle bsds bsdn \
				      $typenodes($nodeid) $typedef]
		    unset typenodes($nodeid)
		}
	    }
	}

	# issue typedef commands from "defs", successful definitions
	# add their fragments to the "bsds" array
	set max [llength $defs]
	if {$max} {
	    typedef $handle begin
	    # make enum definitions first
	    set ndefs {}
	    foreach cmd $defs {
		if {[lindex [lindex $cmd 4] 0] eq "enum"} {
		    if {![{*}$cmd]} {
			lappend ndefs $cmd
		    }
		} else {
		    lappend ndefs $cmd
		}
	    }
	    set defs $ndefs
	    set max [llength $defs]
	    # try remaining definition commands multiple times
	    set count 0
	    while {$count < $max} {
		set ndefs {}
		foreach cmd $defs {
		    if {![{*}$cmd]} {
			lappend ndefs $cmd
		    }
		}
		set defs [lreverse $ndefs]
		incr count
	    }
	    typedef $handle commit
	    set max [llength $defs]
	}

	# add the *.bsd information left from commands above
	# to the address space
	foreach nsidx [array names bsds] {
	    set nsname ""
	    foreach nsname [array names sns] {
		if {$sns($nsname) == $nsidx} {
		    break
		}
	    }
	    set nsshort [lindex [split [string trimright $nsname "/"] "/"] end]
	    set hdr "<opc:TypeDictionary\n"
	    append hdr " xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\"\n"
	    append hdr " xmlns:tns=\"$nsname\"\n"
	    append hdr " DefaultByteOrder=\"LittleEndian\"\n"
	    append hdr " xmlns:opc=\"http://opcfoundation.org/BinarySchema/\"\n"
	    append hdr " xmlns:ua=\"http://opcfoundation.org/UA/\"\n"
	    append hdr " TargetNamespace=\"$nsname\">\n"
	    append hdr " <opc:Import "
	    append hdr "Namespace=\"http://opcfoundation.org/UA/\" />\n"
	    set xml ${hdr}$bsds($nsidx)
	    append xml "</opc:TypeDictionary>\n"

	    # get /Root/Types/DataTypes/OPC Binary
	    set OB [_hxlate $handle "Types/DataTypes/OPC Binary"]
	    # get /Root/Types/VariableTypes/BaseVariableType/BaseDataVariableType/DataTypeDictionaryType
	    set DT [_hxlate $handle \
			Types/VariableTypes/BaseVariableType/BaseDataVariableType/DataTypeDictionaryType]
	    # get /Root/Types/VariableTypes/BaseVariableType/PropertyType
	    set PT [_hxlate $handle \
			Types/VariableTypes/BaseVariableType/PropertyType]
	    # get /Root/Types/VariableTypes/BaseVariableType/BaseDataVariableType/DataTypeDescriptionType
	    set DD [_hxlate $handle \
			Types/VariableTypes/BaseVariableType/BaseDataVariableType/DataTypeDescriptionType]

	    # make node for *.bsd type defs
	    set att [attrs default VariableAttributes]
	    dict set att DataType [types nodeid ByteString]
	    dict set att Value [list ByteString $xml]
	    dict set att ValueRank [const VALUERANK_SCALAR]
	    set tt [add $handle Variable [_nodeid $handle $nsidx] $OB \
			HasComponent "${nsidx}:${nsshort}" $DT $att]
	    set exists($tt) 1
	    # and store the namespace URI as property
	    set att [attrs default VariableAttributes]
	    dict set att DataType [types nodeid String]
	    dict set att Value [list String $nsname]
	    dict set att ValueRank [const VALUERANK_SCALAR]
	    set nt [add $handle Variable [_nodeid $handle $nsidx] $tt \
			HasProperty "NamespaceUri" $PT $att]
	    set exists($nt) 1
	    # and make variables for DataTypeDefinitions
	    if {![::info exists bsdn($nsidx)]} {
		continue
	    }
	    foreach {name encid} $bsdn($nsidx) {
		set att [attrs default VariableAttributes]
		dict set att DataType [types nodeid String]
		dict set att Value [list String $name]
		dict set att ValueRank [const VALUERANK_SCALAR]
		set dd [add $handle Variable [_nodeid $handle $nsidx] $tt \
			    HasComponent $name $DD $att]
		set exists($dd) 1
		# add reference to encoding object
		set en [add $handle Reference $encid HasDescription $dd 1]
		set exists($en) 1
	    }
	}
	_ld_log "Type information created"

	# retry failed "opcua write" operations with the generated types
	# TBD: the gentypes call is possibly redundant
	set msaved [methods $handle]
	_gentypes1 $handle
	foreach {m t cb} $msaved {
	    # try to restore method output types
	    catch {methods $handle $m $t}
	}
	foreach nodeid [array names errnodes] {
	    if {[::info exists exists($nodeid)]} {
		set node $errnodes($nodeid)
		if {[dict exists $node type] && [dict exists $node value]} {
		    if {![catch {_ld_wrvalue $handle $node} err]} {
			unset errnodes($nodeid)
		    }
		}
	    } else {
		dict set errnodes($nodeid) exists 0
	    }
	}
	_ld_log "Retried failed nodes"
	_ld_log "[array size errnodes] failed nodes"
	_ld_log "[array size unrefs] failed references"

	# leave error reports for caller, if requested
	if {$evar ne {}} {
	    upvar $evar ev
	    set ev [array get errnodes]
	}
	if {$rvar ne {}} {
	    upvar $rvar rv
	    set rv [array get unrefs]
	}
	if {$tvar ne {}} {
	    upvar $tvar tv
	    set tv [array get typenodes]
	}

	# try to add method output types to newly created methods
	foreach mn [array names meths] {
	    foreach nodeid $meths($mn) {
		catch {
		    set oa [translate $handle $nodeid \
				HasProperty OutputArguments]
		    set oid [lindex $oa 0]
		    methods $handle $nodeid [read $handle $oid]
		}
	    }
	}
	# drop extra references on Input/OutputArguments, i.e.
	# if a HasProperty reference is seen, remove the reference
	# to BaseDataVariableType
	set pt [_hxlate $handle \
	    Types/VariableTypes/BaseVariableType/PropertyType]
	foreach mn [array names meths] {
	    foreach nodeid $meths($mn) {
		catch {
		    set ia [translate $handle $nodeid \
				HasProperty InputArguments]
		    set iid [lindex $ia 0]
		    set tds [browse $handle $iid Forward HasTypeDefinition]
		    set haspt 0
		    foreach {typeid _ _ _ _ _} $tds {
			if {$typeid eq $pt} {
			    set haspt 1
			    break
			}
		    }
		    if {$haspt} {
			foreach {typeid _ _ _ _ _} $tds {
			    if {$typeid ne $pt} {
				delete $handle Reference $iid \
				    HasTypeDefinition $typeid 1
			    }
			}
		    }
		}
		catch {
		    set oa [translate $handle $nodeid \
				HasProperty OutputArguments]
		    set oid [lindex $oa 0]
		    set tds [browse $handle $oid Forward HasTypeDefinition]
		    set haspt 0
		    foreach {typeid _ _ _ _ _} $tds {
			if {$typeid eq $pt} {
			    set haspt 1
			    break
			}
		    }
		    if {$haspt} {
			foreach {typeid _ _ _ _ _} $tds {
			    if {$typeid ne $pt} {
				delete $handle Reference $oid \
				    HasTypeDefinition $typeid 1
			    }
			}
		    }
		}
	    }
	}
	_ld_log "Methods fixed"
	# same procedure on all Variable and VariableType nodes
	foreach nodeid [array names vars] {
	    catch {
		set tds [browse $handle $nodeid Forward HasTypeDefinition]
		set haspt 0
		foreach {typeid _ _ _ _ _} $tds {
		    if {$typeid eq $pt} {
			set haspt 1
			break
		    }
		}
		if {$haspt} {
		    foreach {typeid _ _ _ _ _} $tds {
			if {$typeid ne $pt} {
			    delete $handle Reference $nodeid \
				HasTypeDefinition $typeid 1
			}
		    }
		}
	    }
	}
	_ld_log "Variables/VariableTypes fixed"

	# shrink _lnodes array for things which should be kept,
	# final cleanup is done by caller
	foreach nodeid [array names exists] {
	    unset -nocomplain _lnodes($nodeid)
	}

	# return method names and nodeids
	return [array get meths]
    }

}

#############################################################################
# Coroutine support.

namespace eval ::opcua {

    # Coroutine aware version of read command.

    proc xread {args} {
	if {[info [lindex $args 0]] ne "server"} {
	    set coro [::info coroutine]
	    if {$coro ne {}} {
		if {[catch {read {*}$args $coro} req]} {
		    return -code error $req
		}
		set ret [::yield]
		set sc [dict get $ret ResponseHeader ServiceResult]
		if {$sc == 0} {
		    set val [lindex [dict get $ret Results] 0]
		    if {[dict exists $val value]} {
			return [dict get $val value]
		    }
		    if {[dict exists $val status]} {
			set sc [dict get $val status]
		    }
		}
		return -code error -errorcode \
		    [list opcua ReadAttributeAsync $sc \
		    [sc2str -short $sc]] [sc2str $sc]
	    }
	}
	tailcall read {*}$args
    }

    # Coroutine aware version of mread command.

    proc xmread {args} {
	if {[info [lindex $args 0]] ne "server"} {
	    set coro [::info coroutine]
	    if {$coro ne {}} {
		if {[catch {mread $coro {*}$args} req]} {
		    return -code error $req
		}
		set ret [::yield]
		set sc [dict get $ret ResponseHeader ServiceResult]
		if {$sc == 0} {
		    set vals {}
		    foreach val [dict get $ret Results] {
			if {[dict exists $val value]} {
			    lappend vals [dict get $val value]
			} else {
			    if {[dict exists $val status]} {
				set sc [dict get $val status]
				if {$sc != 0} {
				    break
				}
			    }
			    lappend vals {}
			}
		    }
		    if {$sc == 0} {
			return $vals
		    }
		}
		return -code error -errorcode \
		    [list opcua MReadAttributeAsync $sc \
		    [sc2str -short $sc]] [sc2str $sc]
	    }
	}
	tailcall mread {} {*}$args
    }

    # Coroutine aware version of mreadx command.

    proc xmreadx {args} {
	if {[info [lindex $args 0]] ne "server"} {
	    set coro [::info coroutine]
	    if {$coro ne {}} {
		if {[catch {mreadx $coro {*}$args} req]} {
		    return -code error $req
		}
		set ret [::yield]
		set sc [dict get $ret ResponseHeader ServiceResult]
		if {$sc == 0} {
		    set vals {}
		    foreach val [dict get $ret Results] {
			set sc 2147549184
			if {[dict exists $val status]} {
			    set sc [dict get $val status]
			}
			lappend vals $sc
			if {[dict exists $val value]} {
			    lappend vals [dict get $val value]
			} else {
			    lappend vals {}
			}
		    }
		    return $vals
		}
		return -code error -errorcode \
		    [list opcua MReadxAttributeAsync $sc \
		    [sc2str -short $sc]] [sc2str $sc]
	    }
	}
	tailcall mreadx {} {*}$args
    }

    # Coroutine aware version of write command.

    proc xwrite {args} {
	if {[info [lindex $args 0]] ne "server"} {
	    set coro [::info coroutine]
	    if {$coro ne {}} {
		if {[catch {write {*}$args $coro} req]} {
		    return -code error $req
		}
		set ret [::yield]
		set sc [dict get $ret ResponseHeader ServiceResult]
		if {$sc == 0} {
		    set sc [lindex [dict get $ret Results] 0]
		    if {$sc == 0} {
			return {}
		    }
		}
		return -code error -errorcode \
		    [list opcua WriteAttributeAsync $sc \
		    [sc2str -short $sc]] [sc2str $sc]
	    }
	}
	tailcall write {*}$args
    }

    # Coroutine aware version of mwrite command.

    proc xmwrite {args} {
	if {[info [lindex $args 0]] ne "server"} {
	    set coro [::info coroutine]
	    if {$coro ne {}} {
		if {[catch {mwrite $coro {*}$args} req]} {
		    return -code error $req
		}
		set ret [::yield]
		set sc [dict get $ret ResponseHeader ServiceResult]
		if {$sc == 0} {
		    foreach sc [dict get $ret Results] {
			if {$sc != 0} {
			    break
			}
		    }
		    if {$sc == 0} {
			return {}
		    }
		}
		return -code error -errorcode \
		    [list opcua MWriteAttributeAsync $sc \
		    [sc2str -short $sc]] [sc2str $sc]
	    }
	}
	tailcall mwrite {} {*}$args
    }

    # Coroutine aware version of mwritex command.

    proc xmwritex {args} {
	if {[info [lindex $args 0]] ne "server"} {
	    set coro [::info coroutine]
	    if {$coro ne {}} {
		if {[catch {mwritex $coro {*}$args} req]} {
		    return -code error $req
		}
		set ret [::yield]
		set sc [dict get $ret ResponseHeader ServiceResult]
		if {$sc == 0} {
		    return [dict get $ret Results]
		}
		return -code error -errorcode \
		    [list opcua MWritexAttributeAsync $sc \
		    [sc2str -short $sc]] [sc2str $sc]
	    }
	}
	tailcall mwritex {} {*}$args
    }

    # Coroutine aware version of call command.

    proc xcall {args} {
	if {[info [lindex $args 0]] ne "server"} {
	    set coro [::info coroutine]
	    if {$coro ne {}} {
		if {[catch {call {*}$args -async $coro} req]} {
		    return -code error $req
		}
		set ret [::yield]
		set sc [dict get $ret ResponseHeader ServiceResult]
		if {$sc == 0} {
		    set val [lindex [dict get $ret Results] 0]
		    set sc [dict get $val StatusCode]
		    if {$sc == 0} {
			return [dict get $val OutputArguments]
		    }
		}
		return -code error -errorcode \
		    [list opcua CallAsync $sc \
		    [sc2str -short $sc]] [sc2str $sc]
	    }
	}
	tailcall call {*}$args
    }

    # Coroutine aware sleep in milliseconds, in all cases
    # use after command with callback to service events.

    proc xsleep {ms} {
	variable sleepvar
	set coro [::info coroutine]
	if {$coro ne {}} {
	    ::after $ms [list $coro]
	    tailcall ::yield
	}
	set nsvar [::namespace current]::sleepvar
	::after $ms [list set $nsvar 1]
	tailcall ::vwait $nsvar
    }

    # Like genstubs, but make coroutine aware stubs
    # using xcall instead of call.

    proc xgenstubs {handle {strip {}} {substs {}} args} {
	# no tailcall, since genstubs wants to see the caller
	return [genstubs $handle $strip $substs {*}$args]
    }

    # Internal helper to cleanup coroutine context, e.g. to be
    # called before a coroutine is programmatically deleted.

    proc _coro_cleanup {name} {
	foreach {type handle} [info] {
	    if {$type ne "client"} {
		continue
	    }
	    foreach id [request $handle] {
		lassign [request $handle $id] type cmd
		if {$cmd eq $name} {
		    cancel $handle $id
		}
	    }
	}
    }

}

#############################################################################
# PubSub support.

namespace eval ::opcua {

    # PubSub support using method calls from information model.

    proc pubsub {handle op args} {
	set subcmd [::tcl::prefix match -message "subcommand" \
	    -error [list -level 1 -errorcode \
		[list TCL LOOKUP INDEX subcommand $op]]  {
	    AddConnection RemoveConnection
	    AddDataSetFolder RemoveDataSetFolder
	    AddPublishedDataItems RemovePublishedDataSet
	    AddReaderGroup AddWriterGroup RemoveGroup
	    AddDataSetReader RemoveDataSetReader
	    AddDataSetWriter RemoveDataSetWriter
	    AddVariables RemoveVariables
	    DeletePubSubConfiguration LoadPubSubConfigurationFile
	} $op]
	set nargs [llength $args]
	set me [dict get [::info frame 0]]
	if {[dict exists $me proc]} {
	    set me [dict get $me proc]
	} else {
	    set me "<cmd>"
	}
	set wna "wrong # args: should be \"$me handle $op"
	switch -- $op {
	    AddConnection {
		if {$nargs != 1} {
		    append wna " config\""
		    return -code error -errorcode {TCL WRONGARGS} $wna
		}
		set obj [_hxlate $handle Objects/Server/PublishSubscribe]
		set meth [translate $handle $obj / AddConnection]
		# keep only the nodeid
		set meth [lindex $meth 0]
		# returns nodeid
		tailcall call $handle $obj $meth \
		    !PubSubConnectionDataType [lindex $args 0]
	    }
	    RemoveConnection {
		if {$nargs != 1} {
		    append wna " connId\""
		    return -code error -errorcode {TCL WRONGARGS} $wna
		}
		set obj [_hxlate $handle Objects/Server/PublishSubscribe]
		set meth [translate $handle $obj / RemoveConnection]
		# keep only the nodeid
		set meth [lindex $meth 0]
		# returns nothing
		tailcall call $handle $obj $meth \
		    !NodeId [lindex $args 0]
	    }
	    AddDataSetFolder {
		if {$nargs != 1} {
		    append wna " name\""
		    return -code error -errorcode {TCL WRONGARGS} $wna
		}
		set obj [_hxlate $handle \
			     Objects/Server/PublishSubscribe/PublishedDataSets]
		set meth [translate $handle $obj / AddDataSetFolder]
		# keep only the nodeid
		set meth [lindex $meth 0]
		# returns nodeid
		tailcall call $handle $obj $meth \
		    !String [lindex $args 0]
	    }
	    RemoveDataSetFolder {
		if {$nargs != 1} {
		    append wna " folderId\""
		    return -code error -errorcode {TCL WRONGARGS} $wna
		}
		set obj [_hxlate $handle \
			     Objects/Server/PublishSubscribe/PublishedDataSets]
		set meth [translate $handle $obj / RemoveDataSetFolder]
		# keep only the nodeid
		set meth [lindex $meth 0]
		# returns nothing
		tailcall call $handle $obj $meth \
		    !NodeId [lindex $args 0]
	    }
	    AddPublishedDataItems {
		if {$nargs != 4} {
		    append wna " name aliases flags vars\""
		    return -code error -errorcode {TCL WRONGARGS} $wna
		}
		set obj [_hxlate $handle \
			     Objects/Server/PublishSubscribe/PublishedDataSets]
		set meth [translate $handle $obj / AddPublishedDataItems]
		# keep only the nodeid
		set meth [lindex $meth 0]
		# returns list with nodeids
		tailcall call $handle $obj $meth \
		    !String [lindex $args 0] \
		    *String [lindex $args 1] \
		    *UInt16 [lindex $args 2] \
		    *PublishedVariableDataType [lindex $args 3]
	    }
	    RemovePublishedDataSet {
		if {$nargs != 1} {
		    append wna " pdsId\""
		    return -code error -errorcode {TCL WRONGARGS} $wna
		}
		set obj [_hxlate $handle \
			     Objects/Server/PublishSubscribe/PublishedDataSets]
		set meth [translate $handle $obj / RemovePublishedDataSet]
		# keep only the nodeid
		set meth [lindex $meth 0]
		# returns nothing
		tailcall call $handle $obj $meth \
		    !NodeId [lindex $args 0]
	    }
	    AddReaderGroup {
		if {$nargs != 2} {
		    append wna " connId readerGroupData\""
		    return -code error -errorcode {TCL WRONGARGS} $wna
		}
		set obj [lindex $args 0]
		set meth [translate $handle $obj / AddReaderGroup]
		# keep only the nodeid
		set meth [lindex $meth 0]
		# returns nodeid
		tailcall call $handle $obj $meth \
		    !ReaderGroupDataType [lindex $args 1]
	    }
	    AddWriterGroup {
		if {$nargs != 2} {
		    append wna " connId writerGroupData\""
		    return -code error -errorcode {TCL WRONGARGS} $wna
		}
		set obj [lindex $args 0]
		set meth [translate $handle $obj / AddWriterGroup]
		# keep only the nodeid
		set meth [lindex $meth 0]
		# returns nodeid
		tailcall call $handle $obj $meth \
		    !WriterGroupDataType [lindex $args 1]
	    }
	    RemoveGroup {
		if {$nargs != 2} {
		    append wna " connId groupId\""
		    return -code error -errorcode {TCL WRONGARGS} $wna
		}
		set obj [lindex $args 0]
		set meth [translate $handle $obj / RemoveGroup]
		# keep only the nodeid
		set meth [lindex $meth 0]
		# returns nothing
		tailcall call $handle $obj $meth \
		    !NodeId [lindex $args 1]
	    }
	    AddDataSetReader {
		if {$nargs != 2} {
		    append wna " groupId readerData\""
		    return -code error -errorcode {TCL WRONGARGS} $wna
		}
		set obj [lindex $args 0]
		set meth [translate $handle $obj / AddDataSetReader]
		# keep only the nodeid
		set meth [lindex $meth 0]
		# returns nodeid
		tailcall call $handle $obj $meth \
		    !DataSetReaderDataType [lindex $args 1]
	    }
	    RemoveDataSetReader {
		if {$nargs != 2} {
		    append wna " groupId readerId\""
		    return -code error -errorcode {TCL WRONGARGS} $wna
		}
		set obj [lindex $args 0]
		set meth [translate $handle $obj / RemoveDataSetReader]
		# keep only the nodeid
		set meth [lindex $meth 0]
		# returns nothing
		tailcall call $handle $obj $meth \
		    !NodeId [lindex $args 1]
	    }
	    AddDataSetWriter {
		if {$nargs != 2} {
		    append wna " groupId writerData\""
		    return -code error -errorcode {TCL WRONGARGS} $wna
		}
		set obj [lindex $args 0]
		set meth [translate $handle $obj / AddDataSetWriter]
		# keep only the nodeid
		set meth [lindex $meth 0]
		# returns nodeid
		tailcall call $handle $obj $meth \
		    !DataSetWriterDataType [lindex $args 1]
	    }
	    RemoveDataSetWriter {
		if {$nargs != 2} {
		    append wna " groupId writerId\""
		    return -code error -errorcode {TCL WRONGARGS} $wna
		}
		set obj [lindex $args 0]
		set meth [translate $handle $obj / RemoveDataSetWriter]
		# keep only the nodeid
		set meth [lindex $meth 0]
		# returns nothing
		tailcall call $handle $obj $meth \
		    !NodeId [lindex $args 1]
	    }
	    AddVariables {
		# TBD: not implemented in open62541 1.3
		return -code error -errorcode [list opcua AddVariables \
		    2151677952 [sc2str -short 2151677952]]
	    }
	    RemoveVariables {
		# TBD: not implemented in open62541 1.3
		return -code error -errorcode [list opcua RemoveVariables \
		    2151677952 [sc2str -short 2151677952]]
	    }
	    DeletePubSubConfiguration {
		if {$nargs != 0} {
		    append wna "\""
		    return -code error -errorcode {TCL WRONGARGS} $wna
		}
		set obj [_hxlate $handle Objects/Server/PublishSubscribe]
		set meth [translate $handle $obj \
			      / 1:DeletePubSubConfiguration]
		# keep only the nodeid
		set meth [lindex $meth 0]
		# returns nothing
		tailcall call $handle $obj $meth
	    }
	    LoadPubSubConfigurationFile {
		if {$nargs != 1} {
		    append wna " bytes\""
		    return -code error -errorcode {TCL WRONGARGS} $wna
		}
		set obj [_hxlate $handle Objects/Server/PublishSubscribe]
		set meth [translate $handle $obj \
			      / 1:LoadPubSubConfigurationFile]
		# keep only the nodeid
		set meth [lindex $meth 0]
		# returns nothing
		tailcall call $handle $obj $meth \
		    !ByteString [lindex $args 0]
	    }
	}
    }

}

#############################################################################
# Dump address space to XML. Support procedures are prefixed with "_xd".

namespace eval ::opcua {

    # Internal helper: fixup QName in XML tags.

    proc _xd_qname {n} {
	if {[scan $n "%d:%s" _ rest] == 2} {
	    # omit numeric namespace prefix
	    set n $rest
	}
	set n0 [string index $n 0]
	if {[string match {[0-9]} $n0]} {
	    # e.g. make "3DVector" into "ThreeDVector"
	    set n [lindex {
		Zero One Two Three Four Five Six Seven Eight Nine
	    } $n0][string range $n 1 end]
	}
	regsub -all {[^[:alnum:]]} $n _ n
	return $n
    }

    # Internal helper: fix namespace index in NodeId.

    proc _xd_fix_ni {n} {
	if {[scan $n "ns=%d;%s" ns rest] == 2} {
	    incr ns -1
	    if {$ns} {
		return "ns=${ns};$rest"
	    }
	}
	return $n
    }

    # Internal helper: fix namespace index in QualifiedName.

    proc _xd_fix_qn {b} {
	if {[scan $b "%d:%s" bs rest] == 2} {
	    if {$bs > 1} {
		incr bs -1
		return ${bs}:$rest
	    }
	}
	return $b
    }

    # Internal helper: fix namespace index in ExpandedNodeId.

    proc _xd_fix_en {en} {
	set n [lindex $en 0]
	if {[scan $n "ns=%d;%s" ns rest] == 2} {
	    incr ns -1
	    if {$ns} {
		return [list "ns=${ns};$rest" {*}[lrange $en 1 end]]
	    }
	}
	return $en
    }

    # Internal helper: add LocalizedText to XML document element.

    proc _xd_val_lt {doc el val {elname LocalizedText}} {
	if {$val eq {}} {
	    return
	}
	set v [$doc createElement uax:[_xd_qname $elname]]
	if {[dict get $val locale] ne {}} {
	    set vv [$doc createElement uax:Locale]
	    $vv appendChild [$doc createText [dict get $val locale]]
	    $v appendChild $vv
	}
	if {[dict get $val text] ne {}} {
	    set vv [$doc createElement uax:Text]
	    $vv appendChild [$doc createText [dict get $val text]]
	    $v appendChild $vv
	}
	$el appendChild $v
    }

    # Internal helper: add QualifiedName to XML document element.

    proc _xd_val_qn {doc el val {elname QualifiedName}} {
	if {$val eq {}} {
	    return
	}
	set v [$doc createElement uax:[_xd_qname $elname]]
	set ns 0
	set rest $val
	scan $val "%d:%s" ns rest
	set vv [$doc createElement uax:NamespaceIndex]
	$vv appendChild [$doc createText $ns]
	$v appendChild $vv
	set vv [$doc createElement uax:Name]
	$vv appendChild [$doc createText $rest]
	$v appendChild $vv
	$el appendChild $v
    }

    # Internal helper: add boolean value to XML document element.

    proc _xd_val_bool {doc el val {elname Boolean}} {
	if {$val eq {}} {
	    return
	}
	set v [$doc createElement uax:[_xd_qname $elname]]
	$v appendChild [$doc createText [expr {$val ? "true" : "false"}]]
	$el appendChild $v
    }

    # Internal helper: add ByteString to XML document element.

    proc _xd_val_bytes {doc el val {elname ByteString}} {
	if {$val eq {}} {
	    return
	}
	set v [$doc createElement uax:[_xd_qname $elname]]
	$v appendChild [$doc createText \
			    \n[binary encode base64 -maxlen 76 $val]\n]
	$el appendChild $v
    }

    # Internal helper: add NodeId to XML document element.

    proc _xd_val_ni {doc el val {elname NodeId}} {
	if {$val eq {}} {
	    return
	}
	set v [$doc createElement uax:[_xd_qname $elname]]
	set vv [$doc createElement uax:Identifier]
	$vv appendChild [$doc createText $val]
	$v appendChild $vv
	$el appendChild $v
    }

    # Internal helper: add ExpandedNodeId to XML document element.

    proc _xd_val_en {doc el val {elname ExpandedNodeId}} {
	if {$val eq {}} {
	    return
	}
	set v [$doc createElement uax:[_xd_qname $elname]]
	if {[lindex $val 2] != 0} {
	    set val "srv=[lindex $val 2];[lindex $val 0]"
	} else {
	    set val [lindex $val 0]
	}
	set vv [$doc createElement uax:Identifier]
	$vv appendChild [$doc createText $val]
	$v appendChild $vv
	$el appendChild $v
    }

    # Internal helper: add ExtensionObject to XML document element,
    # if "encid" is empty, add structure instead.

    proc _xd_val_eo {handle doc el xvar type encid val} {
	upvar $xvar xmap
	if {$val eq {}} {
	    return
	}
	if {$encid ne {}} {
	    set v [$doc createElement uax:ExtensionObject]
	    set vv [$doc createElement uax:TypeId]
	    set vvv [$doc createElement uax:Identifier]
	    $vvv appendChild [$doc createText $encid]
	    $vv appendChild $vvv
	    $v appendChild $vv
	    set vv [$doc createElement uax:Body]
	} else {
	    set v $el
	    set vv $el
	}
	set ti [types info $handle $type]
	if {[dict get $ti kind] eq "OptStructure"} {
	    set members [list]
	    foreach {opt t mname} [dict get $ti members] {
		lappend members $t $mname
	    }
	} else {
	    set members [dict get $ti members]
	}
	if {$encid ne {}} {
	    set vvv [$doc createElement uax:[_xd_qname [dict get $ti name]]]
	} else {
	    set vvv $el
	}
	foreach {t mname} $members {
	    # strip off "!" or "*"
	    set name [string range $mname 1 end]
	    if {![dict exists $val $name]} {
		continue
	    }
	    set ilist [dict get $val $name]
	    if {[string match "!*" $mname]} {
		set ilist [list $ilist]
		set elv $vvv
	    } elseif {[llength $ilist] > 0} {
		set tt $t
		if {![catch {types info $handle $t} ti]} {
		    set tt [dict get $ti kind]
		}
		if {($name eq "ArrayDimensions") && ($t eq "UInt32")} {
		    set elv [$doc createElement uax:ArrayDimensions]
		    set name UInt32
		} elseif {[catch {
		    set elv [$doc createElement uax:ListOf[_xd_qname $tt]]
		}]} {
		    continue
		}
		$vvv appendChild $elv
	    }
	    foreach i $ilist {
		switch -glob -- $t {
		    LocalizedText {
			_xd_val_lt $doc $elv $i $name
		    }
		    QualifiedName {
			set i [_xd_fix_qn $val]
			_xd_val_qn $doc $elv $i $name
		    }
		    Boolean {
			_xd_val_bool $doc $elv $i $name
		    }
		    ByteString - ImageBMP - ImageGIF - ImageJPG -
		    ImagePNG - AudioDataType {
			_xd_val_bytes $doc $elv $i $name
		    }
		    NodeId {
			set i [_xd_fix_ni $i]
			_xd_val_ni $doc $elv $i $name
		    }
		    ExpandedNodeId {
			set i [_xd_fix_en $i]
			_xd_val_en $doc $elv $i $name
		    }
		    ExtensionObject {
			# TODO
		    }
		    DateTime - Date - UtcTime {
			if {![catch {datetime utcstring $i} ii]} {
			    set i $ii
			}
			set vvvv [$doc createElement uax:[_xd_qname $name]]
			$vvvv appendChild [$doc createText $i]
			$elv appendChild $vvvv
		    }
		    default {
			set vvvv [$doc createElement uax:[_xd_qname $name]]
			if {![catch {types info $handle $t} ti]} {
			    set kind [dict get $ti kind]
			    if {($kind eq "Structure") ||
				($kind eq "OptStructure") ||
				($kind eq "Union")} {
				_xd_val_eo $handle $doc $vvvv xmap $t {} $i
			    } else {
				$vvvv appendChild [$doc createText $i]
			    }
			} else {
			    $vvvv appendChild [$doc createText $i]
			}
			$elv appendChild $vvvv
		    }
		}
	    }
	}
	if {$vvv eq $el} {
	    return
	}
	$vv appendChild $vvv
	$v appendChild $vv
	$el appendChild $v
    }

    # Internal helper: add simple type to XML document element.

    proc _xd_val_simple {doc el type val {elname {}}} {
	if {$val eq {}} {
	    return
	}
	if {$elname eq {}} {
	    set elname $type
	}
	set v [$doc createElement uax:[_xd_qname $elname]]
	$v appendChild [$doc createText $val]
	$el appendChild $v
    }

    # Internal helper: recursively add a Value tag to XML document element.

    proc _xd_value {handle doc el xvar type val rank} {
	upvar $xvar xmap
	# no empty Value element, please
	if {$val eq {}} {
	    return
	}
	set xtype $type
	if {$xtype ni [types basic]} {
	    if {[::info exists xmap($xtype)]} {
		lassign $xmap($xtype) xtype encid
	    } elseif {![catch {types nodeid $xtype} typeid] &&
		      [::info exists xmap($typeid)]} {
		lassign $xmap($typeid) xtype encid
	    } elseif {($xtype eq "Number") || ($xtype eq $xmap(-Number))} {
		set test $val
		if {$rank > [const VALUERANK_SCALAR]} {
		    lassign $test test
		}
		if {[string is wideinteger $test]} {
		    set xtype Int64
		} elseif {[string is double $test]} {
		    set xtype Double
		} else {
		    # not representable in XML
		    return
		}
	    } elseif {($xtype eq "Integer") || ($xtype eq $xmap(-Integer))} {
		set test $val
		if {$rank > [const VALUERANK_SCALAR]} {
		    lassign $test test
		}
		if {[string is wideinteger $test]} {
		    set xtype Int64
		} else {
		    # not representable in XML
		    return
		}
	    } elseif {($xtype eq "UInteger") || ($xtype eq $xmap(-UInteger))} {
		set test $val
		if {$rank > [const VALUERANK_SCALAR]} {
		    lassign $test test
		}
		if {[string is wideinteger $test]} {
		    set xtype UInt64
		} else {
		    # not representable in XML
		    return
		}
	    } elseif {$xtype eq "NumericRange"} {
		set xtype String
	    } elseif {($xtype ne "ExtensionObject") &&
		      ![catch {types info $handle $xtype} ti]} {
		set kind [dict get $ti kind]
		if {($kind ne "Structure") &&
		    ($kind ne "OptStructure") &&
		    ($kind ne "Union")} {
		    # no XML mapping for $type
		    return
		}
		set xtype ExtensionObject
		set encid [dict get $ti encid]
	    } else {
		# no XML mapping for $type
		return
	    }
	}
	if {[llength $val] == 0} {
	    return
	}
	set elv [$doc createElement Value]
	$el appendChild $elv
	if {$rank > [const VALUERANK_SCALAR]} {
	    set elvv [$doc createElement uax:ListOf[_xd_qname $xtype]]
	    $elv appendChild $elvv
	    set elv $elvv
	} else {
	    set val [list $val]
	}
	foreach i $val {
	    switch -glob -- $xtype {
		LocalizedText {
		    _xd_val_lt $doc $elv $i
		}
		QualifiedName {
		    set i [_xd_fix_qn $val]
		    _xd_val_qn $doc $elv $i
		}
		Boolean {
		    _xd_val_bool $doc $elv $i
		}
		ByteString - ImageBMP - ImageGIF - ImageJPG -
		ImagePNG - AudioDataType {
		    _xd_val_bytes $doc $elv $i
		}
		NodeId {
		    set i [_xd_fix_ni $i]
		    _xd_val_ni $doc $elv $i
		}
		ExpandedNodeId {
		    set i [_xd_fix_en $i]
		    _xd_val_en $doc $elv $i
		}
		ExtensionObject {
		    _xd_val_eo $handle $doc $elv xmap $type \
			[_xd_fix_ni $encid] $i
		}
		DateTime - Date - UtcTime {
		    if {![catch {datetime utcstring $i} ii]} {
			set i $ii
		    }
		    _xd_val_simple $doc $elv $xtype $i
		}
		default {
		    _xd_val_simple $doc $elv $xtype $i
		}
	    }
	}
    }

    # Internal helper: write <Definition> for struct,
    # "dt" is node's DataTypeDefinition.

    proc _xd_def_struct {doc el brname dt} {
	variable _xd_alias
	if {![dict exists $dt StructureType]} {
	    return
	}
	set de [$doc createElement Definition]
	$de setAttribute Name [_xd_fix_qn $brname]
	if {[dict get $dt StructureType] == [const STRUCTURETYPE_UNION]} {
	    $de setAttribute IsUnion true
	}
	foreach f [dict get $dt Fields] {
	    set fi [$doc createElement Field]
	    $fi setAttribute Name [dict get $f Name]
	    set td [dict get $f DataType]
	    if {[::info exists _xd_alias($td)]} {
		set td $_xd_alias($td)
	    } else {
		set td [_xd_fix_ni $td]
	    }
	    $fi setAttribute DataType $td
	    if {[dict get $f IsOptional]} {
		$fi setAttribute IsOptional true
	    }
	    if {[dict get $f ValueRank] > [const VALUERANK_SCALAR]} {
		$fi setAttribute ValueRank 1
	    }
	    set desc [dict get $f Description]
	    if {($desc ne {}) &&
		([dict get $desc text] ne {})} {
		set dl [$doc createElement Description]
		$dl appendChild [$doc createText [dict get $desc text]]
		$fi appendChild $dl
	    }
	    $de appendChild $fi
	}
	$el appendChild $de
    }

    # Internal helper: write <Definition> for enum.

    proc _xd_def_enum {handle node doc el brname} {
	if {![catch {translate $handle $node / EnumStrings} en]} {
	    lassign $en en
	    set de [$doc createElement Definition]
	    $de setAttribute Name [_xd_fix_qn $brname]
	    set i 0
	    foreach es [read $handle $en] {
		set fi [$doc createElement Field]
		$fi setAttribute Name [dict get $es text]
		$fi setAttribute Value $i
		incr i
		$de appendChild $fi
	    }
	    $el appendChild $de
	} elseif {![catch {translate $handle $node / OptionSetValues} en]} {
	    lassign $en en
	    set de [$doc createElement Definition]
	    $de setAttribute Name [_xd_fix_qn $brname]
	    $de setAttribute IsOptionSet true
	    set i 0
	    foreach es [read $handle $en] {
		set fi [$doc createElement Field]
		$fi setAttribute Name [dict get $es text]
		$fi setAttribute Value $i
		incr i
		$de appendChild $fi
	    }
	    $el appendChild $de
	} elseif {![catch {translate $handle $node / EnumValues} en]} {
	    lassign $en en
	    set de [$doc createElement Definition]
	    $de setAttribute Name [_xd_fix_qn $brname]
	    foreach ev [read $handle $en] {
		set fi [$doc createElement Field]
		$fi setAttribute Name [dict get $ev DisplayName text]
		$fi setAttribute Value [dict get $ev Value]
		set dd [$doc createElement Description]
		set ddd [dict get $ev Description text]
		if {$ddd ne {}} {
		    $dd appendChild [$doc createText $ddd]
		}
		$fi appendChild $dd
		$de appendChild $fi
	    }
	    $el appendChild $de
	}
    }

    # Internal helper: check if given node is a Structure
    # TBD: currently unused

    proc _xd_check_struct {handle n} {
	set s [translate $handle [root] / Types / DataTypes / BaseDataType \
		/ Structure]
	set s [lindex $s 0]
	foreach {_ nn _ _ _ _ _ _} [tree $handle $s DataType] {
	    if {$nn eq $n} {
		return 1
	    }
	}
	return 0
    }

    # Internal helper: write <Definition> for OptionSet if any

    proc _xd_try_optset {handle node doc el brname} {
	if {![catch {translate $handle $node / OptionSetValues} en]} {
	    lassign $en en
	    set de [$doc createElement Definition]
	    $de setAttribute Name [_xd_fix_qn $brname]
	    $de setAttribute IsOptionSet true
	    set i 0
	    foreach es [read $handle $en] {
		set fi [$doc createElement Field]
		$fi setAttribute Name [dict get $es text]
		$fi setAttribute Value $i
		incr i
		$de appendChild $fi
	    }
	    $el appendChild $de
	}
    }

    # Internal helper: process OPC/UA children generating XML.

    proc _xd_proc_children {handle doc set xvar dvar top} {
	variable _xd_alias
	upvar $xvar xmap
	upvar $dvar done
	foreach {n b d c r t} [browse $handle $top Forward] {
	    if {[::info exists done($n)]} {
		# no XML added and already processed,
		# no children to be processed.
		continue
	    }
	    set done($n) 1
	    if {![string match "ns=*;*" $n]} {
		# no XML added, but visit child nodes
		_xd_proc_children $handle $doc $set xmap done $n
		continue
	    }
	    if {[string match "ns=1;*" $n]} {
		# no XML added, but visit child nodes
		_xd_proc_children $handle $doc $set xmap done $n
		continue
	    }

	    set el [$doc createElement UA$c]

	    # <References> element added later
	    set invreftop {}
	    set rl [$doc createElement References]
	    foreach {nn _ _ _ r _} [browse $handle $n Forward] {
		if {[::info exists done($nn)] && $done($nn) < 0} {
		    continue
		}
		# omit HasSubtype forward references
		if {$r eq [reftype HasSubtype]} {
		    continue
		}
		set rrl [$doc createElement Reference]
		if {[::info exists _xd_alias($r)]} {
		    set r $_xd_alias($r)
		}
		$rrl setAttribute ReferenceType [_xd_fix_ni $r]
		$rrl appendChild [$doc createText [_xd_fix_ni $nn]]
		$rl appendChild $rrl
	    }
	    foreach {nn _ _ _ r _} [browse $handle $n Inverse /] {
		# use the first Inverse Hierarchical reference as parent
		set invreftop $nn
		break
	    }
	    foreach {nn _ _ _ r _} [browse $handle $n Inverse] {
		if {[::info exists done($nn)] && $done($nn) < 0} {
		    continue
		}
		# omit inverse HasEncoding references
		if {$r eq [reftype HasEncoding]} {
		    continue
		}
		# omit inverse HasTypeDefinition references
		if {$r eq [reftype HasTypeDefinition]} {
		    continue
		}
		set rrl [$doc createElement Reference]
		if {[::info exists _xd_alias($r)]} {
		    set r $_xd_alias($r)
		}
		$rrl setAttribute ReferenceType [_xd_fix_ni $r]
		$rrl setAttribute IsForward false
		$rrl appendChild [$doc createText [_xd_fix_ni $nn]]
		$rl appendChild $rrl
	    }

	    $el setAttribute NodeId [_xd_fix_ni $n]
	    $el setAttribute BrowseName [_xd_fix_qn $b]
	    if {($invreftop ne {}) && ($c in {Method Object Variable})} {
		$el setAttribute ParentNodeId [_xd_fix_ni $invreftop]
	    }
	    if {$d ne {}} {
		set dl [$doc createElement DisplayName]
		$dl appendChild [$doc createText $d]
		$el appendChild $dl
	    }

	    # prepare multi read of node specific attributes
	    set rdlist [list]
	    lappend rdlist $n Description {}
	    switch -exact -- $c {
		Method {
		    lappend rdlist $n Executable {}
		}
		Object {
		    lappend rdlist $n EventNotifier {}
		}
		ObjectType {
		    lappend rdlist $n IsAbstract {}
		}
		ReferenceType {
		    lappend rdlist $n IsAbstract {}
		    lappend rdlist $n Symmetric {}
		    lappend rdlist $n InverseName {}
		}
		DataType {
		    lappend rdlist $n IsAbstract {}
		    lappend rdlist $n DataTypeDefinition {}
		}
		VariableType {
		    lappend rdlist $n IsAbstract {}
		    lappend rdlist $n DataType {}
		    lappend rdlist $n ValueRank {}
		    lappend rdlist $n ArrayDimensions {}
		    lappend rdlist $n Value {}
		}
		Variable {
		    lappend rdlist $n AccessLevel {}
		    lappend rdlist $n DataType {}
		    lappend rdlist $n ValueRank {}
		    lappend rdlist $n ArrayDimensions {}
		    lappend rdlist $n Value {}
		}
	    }

	    # read node specific attributes
	    array unset data
	    foreach {sc v} [mreadx $handle {} {*}$rdlist] {_ attr _} $rdlist {
		if {$sc == 0} {
		    set data($attr) $v
		}
	    }

	    if {[::info exists data(Description)]} {
		set desc $data(Description)
	    } else {
		set desc {}
	    }
	    if {($desc ne {}) && ([dict get $desc text] ne {})} {
		set dl [$doc createElement Description]
		$dl appendChild [$doc createText [dict get $desc text]]
		$el appendChild $dl
	    }

	    # now add <References>
	    $el appendChild $rl

	    switch -exact -- $c {
		Method {
		    if {[::info exists data(Executable)] &&
			!$data(Executable)} {
			$el setAttribute Executable false
			$el setAttribute UserExecutable false
		    }
		}
		Object {
		    if {[::info exists data(EventNotifier)] &&
			$data(EventNotifier)} {
			$el setAttribute EventNotifier $data(EventNotifier)
		    }
		}
		ObjectType {
		    if {[::info exists data(IsAbstract)] &&
			$data(IsAbstract)} {
			$el setAttribute IsAbstract true
		    }
		}
		ReferenceType {
		    if {[::info exists data(InverseName)]} {
			set d $data(InverseName)
			if {($d ne {}) && ([dict get $d text] ne {})} {
			    set dl [$doc createElement InverseName]
			    $dl appendChild [$doc createText [dict get $d text]]
			    $el appendChild $dl
			}
		    }
		    if {[::info exists data(Symmetric)] &&
			$data(Symmetric)} {
			$el setAttribute Symmetric true
		    }
		    if {[::info exists data(IsAbstract)] &&
			$data(IsAbstract)} {
			$el setAttribute IsAbstract true
		    }
		}
		DataType {
		    if {[::info exists data(IsAbstract)] &&
			$data(IsAbstract)} {
			$el setAttribute IsAbstract true
		    }
		    set ok 0
		    if {[::info exists data(DataTypeDefinition)]} {
			_xd_def_struct $doc $el $b $data(DataTypeDefinition)
			incr ok
		    } elseif {![catch {types info $handle $n} ti]} {
			switch -exact -- [dict get $ti kind] {
			    Enumeration {
				_xd_def_enum $handle $n $doc $el $b
				incr ok
			    }
			}
		    }
		    if {!$ok} {
			_xd_try_optset $handle $n $doc $el $b
		    }
		}
		VariableType - Variable {
		    if {$c eq "VariableType"} {
			if {[::info exists data(IsAbstract)] &&
			    $data(IsAbstract)} {
			    $el setAttribute IsAbstract true
			}
		    } else {
			if {[::info exists data(AccessLevel)]} {
			    set lv $data(AccessLevel)
			    if {$lv != 1} {
				$el setAttribute AccessLevel $lv
				$el setAttribute UserAccessLevel $lv
			    }
			}
		    }
		    set ty i=0
		    if {[::info exists data(DataType)]} {
			set ty $data(DataType)
		    }
		    if {[::info exists _xd_alias($ty)]} {
			set ty $_xd_alias($ty)
		    }
		    $el setAttribute DataType [_xd_fix_ni $ty]
		    set v [const VALUERANK_SCALAR]
		    if {[::info exists data(ValueRank)]} {
			set v $data(ValueRank)
		    }
		    if {$v != [const VALUERANK_SCALAR]} {
			$el setAttribute ValueRank $v
		    }
		    if {[::info exists data(ArrayDimensions)] &&
			($data(ArrayDimensions) ne {})} {
			$el setAttribute ArrayDimensions \
			    [join $data(ArrayDimensions) ,]
		    }
		    if {[::info exists data(Value)]} {
			set val $data(Value)
			_xd_value $handle $doc $el xmap $ty $val $v
		    }
		}
	    }

	    $set appendChild $el

	    _xd_proc_children $handle $doc $set xmap done $n
	}
    }

    # Fetch type map for representing ExtensionObjects and other
    # types in XML.

    proc _xd_typemap {handle xvar} {
	upvar $xvar xmap
	# pass 0: subtyped and enumeration types from "opcua type info..."
	foreach t [types list $handle] {
	    set ti [types info $handle $t]
	    set kind [dict get $ti kind]
	    set typeid [dict get $ti typeid]
	    if {$kind eq "Enumeration"} {
		set xmap($typeid) Int32
	    } elseif {$kind ni {Structure Union OptStruct}} {
		set xmap($typeid) $kind
	    }
	}
	# special entries for abstract types Number, Integer, and UInteger
	set xn [translate $handle [root] / Types / DataTypes / BaseDataType \
		    / Number]
	set xmap(-Number) [lindex $xn 0]
	set xn [translate $handle $xmap(-Number) / Integer]
	set xmap(-Integer) [lindex $xn 0]
	set xn [translate $handle $xmap(-Number) / UInteger]
	set xmap(-UInteger) [lindex $xn 0]
	# pass 1: mark data types
	set xs [translate $handle [root] / Types / DataTypes / "XML Schema"]
	set xs [lindex $xs 0]
	set tree [tree $handle [root] {Object Variable DataType} 1]
	set dtnames [list]
	array set tmap {}
	foreach {l n b d c r t pn} $tree {
	    if {($c eq "DataType") && ![read $handle $n IsAbstract]} {
		lappend dtnames $b
		set tmap($b) $n
	    }
	}
	# pass 2: find "XML Schema" variable nodes
	set nblist [list]
	foreach {l n b d c r t pn} $tree {
	    if {($c eq "Variable") && ($xs in $pn) && ($b in $dtnames)} {
		lappend nblist $n $b
	    }
	}
	# pass 3: find XML encodings (inverse HasDescription references)
	foreach {n b} $nblist {
	    if {[catch {
		lassign [browse $handle $n Inverse HasDescription] nn bn
	    }]} {
		continue
	    }
	    if {$bn eq "Default XML"} {
		set xmap($tmap($b)) [list ExtensionObject $nn]
	    }
	}
    }

    # Build exclude information by seeding the "done" array
    # with node identifiers to be omitted later.

    proc _xd_build_exclude {handle dvar} {
	upvar $dvar done
	set dtd [translate $handle [root] / Types / VariableTypes / \
		     BaseVariableType / BaseDataVariableType / \
		     DataTypeDescriptionType]
	set dtd [lindex $dtd 0]
	set opc [translate $handle [root] / Types / DataTypes / {OPC Binary}]
	set opc [lindex $opc 0]
	set xml [translate $handle [root] / Types / DataTypes / {XML Schema}]
	set xml [lindex $xml 0]
	set tree [tree $handle [root] {Object Variable DataType} 1]
	foreach {l n b d c r t pn} $tree {
	    if {$opc in $pn} {
		set done($n) -1			;# to be skipped later
		continue
	    }
	    if {($xml in $pn) && ![string match "ns=*;*" $n]} {
		if {$n ne $xml} {
		    # but must visit "XML Schema" anyway
		    set done($n) -1		;# to be skipped later
		    continue
		}
	    }
	    if {$c eq "Variable"} {
		if {![catch {
		    browse $handle $n Forward HasTypeDefinition VariableType
		} br]} {
		    # omit "OPC Binary" items but keep "XML Schema"
		    if {($opc in $pn) && ([lindex $br 0] eq $dtd)} {
			set done($n) -1		;# to be skipped later
		    }
		}
	    }
	}
    }

    # Track lookups to alias array.

    proc _xd_alias_used {aname ename op} {
	variable _xd_alias_used
	incr _xd_alias_used($ename)
    }

    # Produce XML dump given OPC/UA handle, returns XML document
    # describing the OPC/UA address space.

    proc xmldump {handle} {
	variable _xd_alias
	variable _xd_alias_used
	foreach {i n} [namespace $handle] {
	    # empty loop body
	}
	if {$i <= 1} {
	    return -code error -errorcode {opcua Internal 0 Good} \
		"no namespace(s) to process"
	}
	# fetch type map for XML extension objects
	array set xmap {}
	_xd_typemap $handle xmap
	set doc [dom createDocumentNode]
	set set [$doc createElement UANodeSet]
	$set setAttribute xmlns \
	    "http://opcfoundation.org/UA/2011/03/UANodeSet.xsd"
	$set setAttribute xmlns:uax \
	    "http://opcfoundation.org/UA/2008/02/Types.xsd"
	# write <NamespaceUris> list
	set ns [$doc createElement NamespaceUris]
	$set appendChild $ns
	foreach {i n} [namespace $handle] {
	    if {$i > 1} {
		set nn [$doc createElement Uri]
		$nn appendChild [$doc createText $n]
		$ns appendChild $nn
	    }
	}
	# build <Models> list
	set ml [$doc createElement Models]
	$set appendChild $ml
	foreach {i n} [namespace $handle] {
	    if {$i > 1} {
		set nn [$doc createElement Model]
		if {[catch {
			translate $handle [root] / Objects / Server \
			     / Namespaces / ${i}:$n / NamespaceUri
		} nnn]} {
		    $nn setAttribute ModelUri $n
		} else {
		    set nnn [lindex $nnn 0]
		    $nn setAttribute ModelUri [read $handle $nnn]
		    set nnn [translate $handle [root] / Objects / Server \
			     / Namespaces / ${i}:$n / NamespacePublicationDate]
		    set nnn [lindex $nnn 0]
		    $nn setAttribute PublicationDate \
			[datetime utcstring [read $handle $nnn]]
		    set nnn [translate $handle [root] / Objects / Server \
			     / Namespaces / ${i}:$n / NamespaceVersion]
		    set nnn [lindex $nnn 0]
		    $nn setAttribute Version [read $handle $nnn]
		}
		$ml appendChild $nn
	    }
	}
	# prepare <Aliases> list
	set al [$doc createElement Aliases]
	$set appendChild $al
	set atypes [types basic]
	set t [translate $handle [root] / Types / DataTypes / BaseDataType]
	set t [lindex $t 0]
	foreach {_ tn b _ _ _ _ _} [tree $handle $t DataType] {
	    if {![string match "ns=*;*" $tn]} {
		set _xd_alias($tn) $b
	    }
	}
	set t [translate $handle [root] / Types / ReferenceTypes / References]
	set t [lindex $t 0]
	foreach {_ tn b _ _ _ _ _} [tree $handle $t ReferenceType] {
	    if {![string match "ns=*;*" $tn]} {
		set _xd_alias($tn) $b
	    }
	}
	# prepare for tracking used aliases
	array set _xdalias_used {}
	trace add variable _xd_alias read \
	    [::namespace current]::_xd_alias_used
	# finally dump the tree, the "done" array tracks visited nodes
	array set done {}
	_xd_build_exclude $handle done
	_xd_proc_children $handle $doc $set xmap done [root]
	$doc appendChild $set
	# fixup <Alias> list, only dump used items
	trace remove variable _xd_alias read \
	    [::namespace current]::_xd_alias_used
	set alias_list {}
	foreach {k v} [array get _xd_alias] {
	    if {[::info exists _xd_alias_used($k)]} {
		lappend alias_list $k $v
	    }
	}
	foreach {k v} [lsort -dictionary -stride 2 -index 1 $alias_list] {
	    set an [$doc createElement Alias]
	    $an setAttribute Alias $v
	    $an appendChild [$doc createText $k]
	    $al appendChild $an
	}
	# finish up
	array unset _xd_alias
	array unset _xd_alias_used
	set ret [$doc asXML -indent 2 -xmlDeclaration 1 -encString utf-8]
	$doc delete
	return $ret
    }

}

#############################################################################
# Make public procs visible in opcua ensemble.

namespace eval ::opcua {

    apply [list ns {
	set cmds [::namespace ensemble configure $ns -subcommands]
	if {("prdict" ni $cmds) &&
	    ([::info proc "::opcua::prdict"] ne {})} {
	    lappend cmds prdict
	}
	if {("filesystem" ni $cmds) &&
	    ([::info proc "::opcua::filesystem"] ne {})} {
	    lappend cmds filesystem
	}
	lappend cmds tree ptree children parent genstubs dumpbsds gentypes
	lappend cmds deftypes attr_init dict_init loader xmldump
	lappend cmds xread xmread xmreadx xwrite xmwrite xmwritex
	lappend cmds xcall xsleep xgenstubs
	lappend cmds pubsub
	::namespace ensemble configure $ns -subcommands $cmds
	::namespace export {[a-z]*}
    } [::namespace current]] [::namespace current]

}
