# taskman.tcl
#
# Simple coroutine based task management.
#
# Copyright (c) 2023-25 Christian Werner <chw at ch minus werner dot de>
#
# See the file "license.terms" for information on usage and redistribution of
# this file, and for a DISCLAIMER OF ALL WARRANTIES.
#
###############################################################

package provide taskman 0.1

namespace eval ::taskman {

    # Globals

    variable mq		;# array holding message queues
    variable cq		;# array holding conditions
    variable eq		;# array holding events
    variable fq		;# array holding flags
    variable vw		;# used for timed vwait

    # <mq>, <cq>, <eq>, and <fq> array element names are the names
    # for message queues, conditions, events, and flag groups,
    # respectively.
    #
    # A message is any non-empty string.
    #
    # A condition is similar to a POSIX condition variable, but
    # since we use cooperative multitasking (coroutines) an
    # enclosing mutex is not needed.
    #
    # An event is a boolean (1 = on, event set, 0 = off, event reset)
    # which can be waited for it's on (set) state.
    #
    # A flag group is an integer which can be waited on using
    # a bit mask.
    #
    # Array entries in <mq> with name pattern m,<name> are lists
    # with zero or more pending messages in queue <name>.
    #
    # Array entries in <mq> with name pattern w,<name> are lists
    # with zero or more names of waiting coroutines in
    # queue <name>.
    #
    # Array entries in <cq> are lists with zero or more
    # names of waiting coroutines on condition <name>.
    #
    # Array entries in <eq> with name pattern v,<name> represent
    # the state of event <name> and name pattern w,<name> the
    # names of waiting coroutines.
    #
    # Array entries in <fq> with name pattern v,<name> represent
    # the state of flag group <name>, name pattern w,<name> the
    # waiting coroutines, each represented by coroutine name and
    # wait mask.
    #
    # Overview of procedures:
    #
    #   Message queues:		qwait qput qpeek qflush
    #   Conditions:		cwait csignal cbroadcast
    #   Events:			ewait eset ereset
    #   Flag groups:		fwait fset freset
    #   Time:			tsleep
    #   Task management:	task periodic_task taskvar tkill
    #   Miscellaneous:		tvars tinfo teval texists

    array set mq {}
    array set cq {}
    array set eq {}
    array set fq {}

    # Sleep for some milliseconds, coroutine aware,
    # an operational event loop is always required.

    proc tsleep {ms} {
	variable vw
	set coro [info coroutine]
	if {$coro ne {}} {
	    after $ms [list $coro]
	    tailcall yield
	}
	after $ms [list set [namespace current]::vw 1]
	tailcall vwait [namespace current]::vw
    }

    # Wait for message on queue with optional timeout. Empty
    # result indicates empty queue or timeout.
    # Caller must be a coroutine context.

    proc qwait {name {ms -1}} {
	variable mq
	set coro [info coroutine]
	if {$coro eq {}} {
	    return -code error "not called in a coroutine"
	}
	if {![info exists mq(m,$name)] || ([llength $mq(m,$name)] == 0)} {
	    if {$ms > 0} {
		set id [after $ms [list $coro {}]]
	    }
	    if {$ms != 0} {
		lappend mq(w,$name) $coro
		set msg [yield]
		set index [lsearch $mq(w,$name) $coro]
		if {$index >= 0} {
		    set mq(w,$name) [lreplace $mq(w,$name) $index $index]
		}
		if {[info exists id]} {
		    after cancel $id
		}
	    } else {
		set msg {}
	    }
	} else {
	    set mq(m,$name) [lassign $mq(m,$name) msg]
	}
	return $msg
    }

    # Put message into queue waking up one consumer.
    # Caller can be a coroutine context.

    proc qput {name msg} {
	variable mq
	if {$msg eq {}} {
	    return -code error "no empty message allowed"
	}
	if {![info exists mq(w,$name)]} {
	    set waiter {}
	} else {
	    set mq(w,$name) [lassign $mq(w,$name) waiter]
	}
	if {$waiter eq {}} {
	    lappend mq(m,$name) $msg
	    return
	}
	tailcall $waiter $msg
    }

    # Peek queue with optional pattern and return number
    # of matching messages in queue.
    # Caller can be a coroutine context.

    proc qpeek {name {pat {}}} {
	variable mq
	if {![info exists mq(m,$name)]} {
	    return 0
	}
	if {$pat eq {}} {
	    return [llength $mq(m,$name)]
	}
	return [llength [lsearch -all $mq(m,$name) $pat]]
    }

    # Remove all messages from queue and wake up all
    # consumers. Return number of removed messages.
    # Caller can be a coroutine context.

    proc qflush {name} {
	variable mq
	set ret 0
	if {[info exists mq(m,$name)]} {
	    set ret [llength $mq(m,$name)]
	    set mq(m,$name) {}
	}
	if {[info exists mq(w,$name)]} {
	    while {1} {
		set waiter {}
		set mq(w,$name) [lassign $mq(w,$name) waiter]
		if {$waiter eq {}} {
		    break
		}
		$waiter {}
	    }
	}
	return $ret
    }

    # Wait for condition with optional timeout. Returns 1 when
    # condition signaled, or -1 on timeout.
    # Caller must be a coroutine context.

    proc cwait {name {ms -1}} {
	variable cq
	set coro [info coroutine]
	if {$coro eq {}} {
	    return -code error "not called in a coroutine"
	}
	if {$ms >= 0} {
	    set id [after $ms [list $coro -1]]
	}
	lappend cq($name) $coro
	set ret [yield]
	set index [lsearch $cq($name) $coro]
	if {$index >= 0} {
	    set cq($name) [lreplace $cq($name) $index $index]
	}
	if {[info exists id]} {
	    after cancel $id
	}
	return $ret
    }

    # Signal condition waking up at most one waiting coroutine.
    # Caller can be a coroutine context.

    proc csignal {name} {
	variable cq
	set waiter {}
	if {[info exists cq($name)]} {
	    set $cq($name) [lassign $cq($name) waiter]
	}
	if {$waiter ne {}} {
	    $waiter 1
	}
    }

    # Signal condition waking up all waiting coroutines.
    # Caller can be a coroutine context.

    proc cbroadcast {name} {
	variable cq
	if {![info exists cq($name)]} {
	    return
	}
	while {1} {
	    set waiter {}
	    set cq($name) [lassign $cq($name) waiter]
	    if {$waiter eq {}} {
		break
	    }
	    $waiter 1
	}
    }

    # Wait for an event with optional event reset and timeout.
    # Return event state (usually 1, -1 on timeout). If timeout
    # is 0, returns current event state (0 or 1).
    # Caller must be a coroutine context.

    proc ewait {name {reset 0} {ms -1}} {
	variable eq
	set coro [info coroutine]
	if {$coro eq {}} {
	    return -code error "not called in a coroutine"
	}
	set state 0
	if {![info exists eq(v,$name)]} {
	    set eq(v,$name) $state
	} else {
	    set state [expr {$eq(v,$name) != 0}]
	}
	if {$state} {
	    if {$reset} {
		set eq(v,$name) 0
	    }
	    return $state
	}
	if {$ms > 0} {
	    set id [after $ms [list $coro -1]]
	}
	if {$ms != 0} {
	    lappend eq(w,$name) $coro
	    set state [yield]
	    set index [lsearch $eq(w,$name) $coro]
	    if {$index >= 0} {
		set eq(w,$name) [lreplace $eq(w,$name) $index $index]
	    }
	    if {[info exists id]} {
		after cancel $id
	    }
	    if {$reset && ($state >= 0)} {
		# don't change event in timeout case
		set eq(v,$name) 0
	    }
	}
	return $state
    }

    # Set an event waking up waiting coroutines and
    # returning number of wakeups.
    # Caller can be a coroutine context.

    proc eset {name} {
	variable eq
	set count 0
	set eq(v,$name) 1
	if {[info exists eq(w,$name)]} {
	    while {$eq(v,$name)} {
		set $eq(w,$name) [lassign $eq(w,$name) waiter]
		if {$waiter eq {}} {
		    break
		}
		incr count
		$waiter $eq(v,$name)
	    }
	}
	return $count
    }

    # Reset an event and return its previous state.
    # Caller can be a coroutine context.

    proc ereset {name} {
	variable eq
	set state 0
	if {[info exists eq(v,$name)]} {
	    set state [expr {$eq(v,$name) != 0}]
	}
	set eq(v,$name) 0
	return $state
    }

    # Wait for flag(s) in a group with optional reset and timeout.
    # Return flag state (empty on timeout). If timeout is 0,
    # return current flag state immediately.
    # Caller must be a coroutine context.

    proc fwait {name mask {reset 0} {ms -1}} {
	variable fq
	set coro [info coroutine]
	if {$coro eq {}} {
	    return -code error "not called in a coroutine"
	}
	set state 0
	if {![info exists fq(v,$name)]} {
	    set fq(v,$name) $state
	} else {
	    set state $fq(v,$name)
	}
	if {$state & $mask} {
	    if {$reset} {
		set fq(v,$name) [expr {$state & ~$mask}]
	    }
	    return $state
	}
	if {$ms > 0} {
	    set id [after $ms [list $coro {}]]
	}
	if {$ms != 0} {
	    set waiter [list $coro $mask]
	    lappend fq(w,$name) $waiter
	    set state [yield]
	    set index [lsearch $fq(w,$name) $waiter]
	    if {$index >= 0} {
		set fq(w,$name) [lreplace $fq(w,$name) $index $index]
	    }
	    if {[info exists id]} {
		after cancel $id
	    }
	    if {$state eq {}} {
		# timeout
		return $state
	    }
	    if {$reset} {
		set fq(v,$name) [expr {$state & ~$mask}]
	    }
	}
	return $state
    }

    # Set flags in a group waking up waiting coroutines and
    # returning number of wakeups.
    # Caller can be a coroutine context.

    proc fset {name mask} {
	variable fq
	set count 0
	if {![info exists fq(v,$name)]} {
	    set fq(v,$name) 0
	}
	set fq(v,$name) [expr {$fq(v,$name) | $mask}]
	if {[info exists fq(w,$name)] && $fq(v,$name)} {
	    set clist [list]
	    set state $fq(v,$name)
	    foreach waiter $fq(w,$name) {
		lassign $waiter coro mask
		if {$state & $mask} {
		    lappend clist $coro
		}
	    }
	    set count [llength $clist]
	    if {$count} {
		foreach coro $clist {
		    $coro $state
		}
	    }
	}
	return $count
    }

    # Reset flags in group and return previous group state.
    # Caller can be a coroutine context.

    proc freset {name mask} {
	variable fq
	set state 0
	if {[info exists fq(v,$name)]} {
	    set state $fq(v,$name)
	}
	set fq(v,$name) [expr {$state & ~$mask}]
	return $state
    }

    # Kill task (coroutine) given name possibly removing it
    # from queues or killing its current sleep operation.
    # The outcome of this operation is the deletion of the coroutine.
    # Caller can be a coroutine context.

    proc tkill {name} {
	variable mq
	variable cq
	variable eq
	variable fq
	if {![string match "::*" $name]} {
	    set name ::${name}
	}
	if {[info command $name] ne $name} {
	    return
	}
	# cleanup timer
	set qname [list $name {}]	;# qwait/fwait
	set ename [list $name -1]	;# ewait/cwait
	foreach id [after info] {
	    lassign [after info $id] cmd type
	    if {($cmd eq $name) || ($cmd eq $qname) || ($cmd eq $ename)} {
		after cancel $id
	    }
	}
	# cleanup waiting on message queue
	foreach id [array names mq w,*] {
	    set index [lsearch $mq($id) $name]
	    if {$index >= 0} {
		set mq($id) [lreplace $mq($id) $index $index]
	    }
	}
	# cleanup waiting on condition
	foreach id [array names cq] {
	    set index [lsearch $cq($id) $name]
	    if {$index >= 0} {
		set cq($id) [lreplace $cq($id) $index $index]
	    }
	}
	# cleanup waiting on event
	foreach id [array names eq w,*] {
	    set index [lsearch $eq($id) $name]
	    if {$index >= 0} {
		set eq($id) [lreplace $eq($id) $index $index]
	    }
	}
	# cleanup waiting on flag group
	foreach id [array names fq w,*] {
	    set index 0
	    foreach we $fq($id) {
		lassign $we coro mask
		if {$coro eq $name} {
		    set fq($id) [lreplace $fq($id) $index $index]
		    break
		}
		incr index
	    }
	}
	# cleanup waiting for opcua async operation completion
	set cmd ::opcua::_coro_cleanup
	if {[info command $cmd] eq $cmd} {
	    $cmd $name
	}
	# cleanup channel event handlers, if any
	foreach chan [chan names] {
	    if {![catch {chan event $chan readable} cmd]} {
		if {$cmd eq $name} {
		    chan event $chan readable {}
		}
	    }
	    if {![catch {chan event $chan writable} cmd]} {
		if {$cmd eq $name} {
		    chan event $chan writable {}
		}
	    }
	}
	# this destroys the coroutine, finally
	rename $name {}
    }

    # Create a task (coroutine) similar to "proc".
    # The real start is performed by invoking <name> and allows
    # for zero or one argument to be passed.

    proc task {name argname body} {
	if {![string match "::*" $name]} {
	    set name ::${name}
	}
	if {[info command $name] eq $name} {
	    return -code error "$name exists"
	}
	if {[llength $argname] > 1} {
	    return -code error "only zero or one arguments allowed"
	}
	if {$argname eq {}} {
	    coroutine $name apply \
		[format {{} {::yield%s%s}} "\n" $body "\n"]
	} else {
	    coroutine $name apply \
		[format {{} {set %s [::yield]%s%s%s}} $argname "\n" $body "\n"]
	}
    }

    # Create a task (coroutine) similar to "proc" which
    # executes <body> each <period> milliseconds.
    # The real start is performed by invoking <name>.

    proc periodic_task {name period body} {
	if {![string match "::*" $name]} {
	    set name ::${name}
	}
	if {[info command $name] eq $name} {
	    return -code error "$name exists"
	}
	if {![string is integer $period] || ($period <= 0)} {
	    return -code error "need positive integer period"
	}
	coroutine $name apply \
	    [format {{} {::yield ; while 1 { %s %s%s%s%s}}} \
		[namespace current]::tsleep $period "\n" $body "\n"]
    }

    # Task local variables living in toplevel proc body of task,
    # stolen from ::coroutine::util package.

    proc taskvar {args} {
	if {[info level] < 2} {
	    return
	}
	if {[info coroutine] eq {}} {
	    return -code error "not called in a coroutine"
	}
	set cmd [list upvar "#1"]
	foreach var $args {
	    lappend cmd $var $var
	}
	tailcall {*}$cmd
    }

    # Evaluate code in a task. <cmd> with <args> is run in
    # global namespace in task's body.

    proc teval {name cmd args} {
	::tcl::unsupported::inject $name \
	    [namespace current]::_eval $cmd {*}$args
	return [$name]
    }

    proc _eval {cmd args} {
	yield [uplevel 1 namespace eval :: $cmd {*}$args]
    }

    # Similar to the "info" command but carried out
    # in task's body.

    proc tinfo {name args} {
	::tcl::unsupported::inject $name \
	    [namespace current]::_eval info {*}$args
	return [$name]
    }

    # Fetch task variables given pattern(s), returns list
    # (name value ...) of variables in task's body.

    proc tvars {name args} {
	::tcl::unsupported::inject $name \
	    [namespace current]::_eval [namespace current]::_vars {*}$args
	return [$name]
    }

    proc _vars {args} {
	set result {}
	if {[llength $args] == 0} {
	    set args *
	}
	foreach i $args {
	    foreach k [uplevel "#1" info vars $i] {
		set m($k) {}
	    }
	}
	foreach i [lsort -dictionary [array names m]] {
	    upvar "#1" $i var
	    if {[array exists var]} {
		foreach k [lsort -dictionary [array names var]] {
		    lappend result [list [list $i]($k) $var($k)]
		}
	    } else {
		catch {lappend result [list $i $var]}
	    }
	}
	return [join $result "\n"]
    }

    # Does coroutine <name> exist?

    proc texists {name} {
	if {[info command $name] ne $name} {
	    return 0
	}
	if {[catch {::tcl::unsupported::inject $name return}]} {
	    return 0
	}
	return 1
    }

    # Expose public interface procs and a taskman ensemble
    # with this mapping (* means exported):
    #
    #	*taskman::tsleep	- taskman sleep
    #	*taskman::qwait		- taskman qwait
    #	*taskman::qput		- taskman qput
    #	*taskman::qpeek		- taskman qpeek
    #	*taskman::qflush	- taskman qflush
    #	*taskman::cwait		- taskman cwait
    #	*taskman::csignal	- taskman csignal
    #	*taskman::cbroadcast	- taskman cbroadcast
    #	*taskman::ewait		- taskman ewait
    #	*taskman::eset		- taskman eset
    #	*taskman::ereset	- taskman ereset
    #	*taskman::fwait		- taskman fwait
    #	*taskman::fset		- taskman fset
    #	*taskman::freset	- taskman freset
    #	*taskman::tkill		- taskman kill
    #	*taskman::task		- taskman task
    #	*taskman::periodic_task	- taskman periodic
    #	*taskman::taskvar	- taskman variable
    #	 taskman::tinfo		- taskman info
    #	 taskman::teval		- taskman eval
    #	 taskman::tvars		- taskman tvars
    #	*taskman::texists	- taskman exists

    namespace export tsleep qwait qput qpeek qflush \
	cwait csignal cbroadcast ewait eset ereset \
	fwait fset freset tkill task periodic_task taskvar texists

    namespace ensemble create -subcommands {
	sleep qwait qput qpeek qflush
	cwait csignal cbroadcast ewait eset ereset
	fwait fset freset kill task periodic variable
	info eval tvars exists
    } -map {
	sleep tsleep
	kill tkill
	periodic periodic_task
	variable taskvar
	info tinfo
	eval teval
	exists texists
    }

}

###############################################################
# If invoked directly, provide a demo of some of the features.

if {[info exists argv0] && ($argv0 eq [info script])} {

    namespace import taskman::*

    # Task local variable getters

    proc geti {} {
	taskvar i max
	return "$i/$max"
    }

    proc mkmsg {} {
	return "msg[geti]"
    }

    # Producer 0 task

    task prod0 {max} {
	puts "*** Start prod0 performing $max productions"
	set i 0
	while {$i < $max} {
	    tsleep 200
	    set msg [mkmsg]
	    puts "prod0@[geti]:\tput $msg"
	    qput Q $msg
	    incr i
	}
	puts "*** End prod0 performing $max productions"
    }

    # Consumer 1 task

    task cons1 {max} {
	puts "*** Start cons1 performing $max consumptions"
	set i 0
	while {$i < $max} {
	    set msg [qwait Q 200]
	    if {$msg ne {}} {
		puts "cons1@[geti]:\tgot $msg"
	    } else {
		puts "cons1@[geti]:\ttimeout"
	    }
	    if {$i % 3 == 0} {
		tsleep 2000
	    }
	    incr i
	}
	puts "*** End cons1 performing $max consumptions"
    }

    # Consumer 2 task

    task cons2 {max} {
	puts "*** Start cons2 performing $max consumptions"
	set i 0
	while {$i < $max} {
	    set msg [qwait Q 250]
	    if {$msg ne {}} {
		puts "cons2@[geti]:\tgot $msg"
	    } else {
		puts "cons2@[geti]:\ttimeout"
	    }
	    if {$i % 2 == 0} {
		tsleep 1000
	    }
	    incr i
	}
	puts "*** End cons2 performing $max consumptions"
    }

    # Event wait task

    task ev1 {max} {
	puts "*** Start ev1 performing $max event waits"
	set i 0
	set t0 [clock milliseconds]
	while {$i < $max} {
	    set state [ewait E 1 250]
	    set t1 [clock milliseconds]
	    puts "ev1@[geti]:\tstate $state @[expr {$t1 - $t0}]ms"
	    if {$i % 2 == 0} {
		tsleep 333
	    }
	    incr i
	}
	puts "*** End ev1 performing $max event waits"
    }

    # Flag wait task

    task fg1 {max} {
	puts "*** Start fg1 performing $max flag group waits"
	set i 0
	set t0 [clock milliseconds]
	while {$i < $max} {
	    set state [fwait F 0x100 1 250]
	    set t1 [clock milliseconds]
	    if {$state eq {}} {
		set state timeout
	    } else {
		set state "state $state"
	    }
	    puts "fg1@[geti]:\t$state @[expr {$t1 - $t0}]ms"
	    if {$i % 2 == 0} {
		tsleep 333
	    }
	    incr i
	}
	puts "*** End fg1 performing $max event waits"
    }

    # Periodic 3 task

    periodic_task peri3 4000 {
	puts "*** Ping [clock format [clock seconds]]"
    }

    # Periodic 4 task

    periodic_task peri4 200 {
	puts "*** Set event E"
	eset E
	incr val	;# this is a taskvar
	set val [expr {($val | 0x110) & 0xffff}]
	puts "*** Set flag group F/$val"
	fset F $val
    }

    # The main logic

    peri3		;# starts the Periodic 3 task
    ev1 50		;# starts the Event 1 task
    fg1 50		;# starts the Flag Group 1 task
    cons1 50		;# starts the Consumer 1 task
    prod0 100		;# starts the Producer 0 task
    cons2 50		;# starts the Consumer 2 task
    peri4		;# starts the Periodic 4 task

    tsleep 10000

    # Peek into tasks
    #
    #puts "*** INFO(prod0):\n[taskman::tinfo prod0 frame -3]"
    #puts "*** INFO(cons1):\n[taskman::tinfo cons1 frame -3]"
    #puts "*** INFO(cons2):\n[taskman::tinfo cons2 frame -3]"
    #puts "*** INFO(peri3):\n[taskman::tinfo peri3 frame -3]"
    #puts "*** INFO(ev1):\n[taskman::tinfo ev1 frame -3]"
    #puts "*** INFO(fg1):\n[taskman::tinfo fg1 frame -3]"
    #puts "*** VARS(prod0):\n[taskman::tvars prod0]"
    #puts "*** ENDINFO"

    puts "*** Kill prod0"
    tkill prod0

    tsleep 3000

    puts "*** Kill cons1"
    tkill cons1

    tsleep 5000

    puts "*** Kill cons2"
    tkill cons2

    puts "*** Kill peri3"
    tkill peri3

    puts "*** Kill peri4"
    tkill peri4

    puts "*** Flushed [qflush Q] msgs"

}
