# prdict.tcl --
#
# Pretty print a dict. Borrowed and adapted from
#
#   https://wiki.tcl-lang.org/page/pdict%3A+Pretty+print+a+dict
#
# Usage: opcua::prdict ?depth? dict ?pattern ...?
#
# Depth if given must be a positive integer indicating the
# maximum nesting of the dict. Default depth is number of
# arguments. A pretty printed result is returned which in
# most cases should still be a valid dict.
#
# If ::tcl::unsupported::representation is available, depth
# is ignored and determined automatically. Additionally,
# lists are pretty printed as well.

package provide topcua::prdict 0.1

namespace eval ::opcua {
    variable _t_u_r 0
    if {[::info command ::tcl::unsupported::representation] eq
	"::tcl::unsupported::representation"} {
	incr _t_u_r
    }
}

proc opcua::_prdictelem {args} {
    set args [lassign $args dvar pat]
    if {$dvar eq {}} {
	return
    }
    if {$pat eq {}} {
	set pat "*"
    }
    set indent [uplevel 1 {set indent}]
    append indent "    "
    set out ""
    set repr [::tcl::unsupported::representation $dvar]
    if {[string first "value is a list" $repr] == 0} {
	append out $indent "\{\n"
	foreach elem $dvar {
	    append out [_prdictelem $elem {*}$args] "\n"
	}
	append out $indent "\}\n"
    } elseif {[string first "value is a dict" $repr] == 0} {
	append out $indent "\{\n"
	append out [prdict 0 $dvar {*}$args] "\n"
	append out $indent "\}\n"
    } elseif {[string first "value is a bytearray" $repr] == 0} {
	append out $indent [binary encode hex $dvar] "\n"
    } else {
	append out $indent $dvar "\n"
    }
    return [string trimright $out "\n"]
}

proc opcua::prdict {args} {
    variable _t_u_r
    lassign [::info level 0] cmd
    if {[string is integer [lindex $args 0]]} {
	set args [lassign $args depth]
    } else {
	set depth [llength $args]
    }
    set args [lassign $args dvar pat]
    if {$dvar eq {}} {
	return -code error \
	    "wrong args: should be \"$cmd ?depth? dict ?pattern ...?\""
    }
    if {$depth <= 0} {
	set depth 1
    }
    if {$_t_u_r} {
	unset depth
    }
    if {$pat eq {}} {
	set pat "*"
    }
    if {([::info level] > 1) && [uplevel 1 {::info exists indent}]} {
	set indent [uplevel 1 {set indent}]
	append indent "    "
    } else {
	set indent ""
    }
    set out ""
    if {$_t_u_r} {
	set repr [::tcl::unsupported::representation $dvar]
	if {[string first "value is a list" $repr] == 0} {
	    if {[llength $dvar] % 2} {
		return -code error "not a dict value: $dvar"
	    }
	} elseif {[string first "value is a dict" $repr] != 0} {
	    return -code error "not a dict value: $dvar"
	}
    }
    if {[catch {dict keys $dvar $pat} keys]} {
	return -code error "not a dict value: $keys"
    }
    if {[llength $keys]} {
	set size [::tcl::mathfunc::max {*}[lmap k $keys {string length $k}]]
	foreach key $keys {
	    set dsubvar [dict get $dvar $key]
	    if {$_t_u_r || ($depth > 1)} {
		set isList 0
		set isBin 0
		if {$_t_u_r} {
		    set isVal 1
		    set repr [::tcl::unsupported::representation $dsubvar]
		    if {[string first "value is a list" $repr] == 0} {
			incr isList
		    } elseif {[string first "value is a dict" $repr] == 0} {
			set isVal [catch {dict keys $dsubvar} keys]
		    } elseif {[string first "value is a bytearray" $repr] == 0} {
			incr isBin
		    }
		} else {
		    set isVal [catch {dict keys $dsubvar} keys]
		}
		if {$isList} {
		    append out $indent $key " \{\n"
		    foreach elem $dsubvar {
			append out [_prdictelem $elem {*}$args] "\n"
		    }
		    append out $indent "\}\n"
		} elseif {!$isVal && [llength $keys] == 0} {
		    append out [format {%s%-*s} $indent $size $key]
		    append out " \{\}\n"
		} elseif {$isBin} {
		    append out [format {%s%-*s} $indent $size $key]
		    append out " " [binary encode hex $dsubvar] "\n"
		} elseif {$isVal} {
		    append out [format {%s%-*s} $indent $size $key]
		    append out " " [list $dsubvar] "\n"
		} else {
		    append out $indent $key " \{\n"
		    incr depth -1
		    append out [$cmd $depth $dsubvar {*}$args] "\n"
		    incr depth 1
		    append out $indent "\}\n"
		}
	    } else {
		append out [format {%s%-*s} $indent $size $key]
		append out " " [list $dsubvar] "\n"
	    }
	}
    }
    return [string trimright $out "\n"]
}

# Make opcua::prdict visible in opcua ensemble.

namespace eval ::opcua {
    apply [list ns {
	if {[::info command $ns] eq {}} return
	set cmds [::namespace ensemble configure $ns -subcommands]
	lappend cmds prdict
	::namespace ensemble configure $ns -subcommands $cmds
	::namespace export {[a-z]*}
    } [::namespace current]] [::namespace current]
}
