#
# tfirmata.tcl
#
# Arduino Firmata implementation
#
# chw 2017: reviewed, updated to protocol version 2.5,
#           added stepper, encoder, onewire, serial, and task functions

package require Tcl 8.5
package provide tfirmata 2.5

namespace eval tfirmata {
    variable pinModes \
        [list in out analog pwm servo shift twi onewire \
             stepper encoder serial input_pullup]
    variable idCount 0
    variable wakeup 0
}

# open serial port and return reference command
proc tfirmata::open {port {noreset 0}} {
    set chan [::open $port r+]
    ::chan configure $chan -mode 57600,n,8,1
    return [chan $chan $noreset]
}

# use given channel and return reference command
proc tfirmata::chan {chan {noreset 0} {parent {}}} {
    variable idCount

    set ns [namespace current]
    set cmd ${ns}::tfirmata#[incr idCount]

    if {$chan ne {}} {
        ::chan configure $chan -translation binary -blocking 0
        ::chan event $chan readable \
            [list ${ns}::parseSerialReadData $cmd $chan]
    }

    # now with cmd's namespace
    set ns $cmd

    # create access command
    proc $cmd {args} [format {subCmd %s {*}$args} [list $ns]]

    # set up command variables
    namespace eval $ns {
        # file descriptor for Arduino, and buffer for reading serial
        variable chan ""
        variable rxBuf ""

        # Arduino configuration. Device config is a dictionary, where keys
        # are Arduino pin numbers, values are lists of supported modes and
        # their resolutions. Analog mapping is a dictionary, where keys are
        # analog channel numbers, values are Arduino pins.
        variable deviceConfig ""
        variable analogMapping ""

        # firmata version
        variable firmwareVersion ""
        variable firmwareName ""

        # digital ports and analog inputs
        variable digitalIns 0
        variable digitalOuts ""
        variable analogIns ""

        # for pin state command
        variable pin ""
        variable mode ""
        variable state ""

        # for counting errors in received messages
        variable errorCount 0

        # for onewire transaction ids
        variable owTid 0

        # message callbacks
        variable digitalCallback ""
        variable analogCallback ""
        variable encoderCallback ""
        variable serialCallback ""
        variable owSearchCallback ""

        # twi callbacks, stored in a dict, where keys are TWI device address,
        # values are a list of dicts, where each dict specifies the callback
        # code, whether the callback is repeating, TWI control byte sequence,
        # and the number of TWI bytes to retrieve.
        variable twiCallbacks ""

        # onewire callbacks, like twi callbacks for onewire transactions
        variable owCallbacks ""

        # holds bytes from TWI reply message when using blocking read
        variable twiReplyBytes ""

        # bitmask of done steppers
        variable wakeup 0

        # timeout flag for TWI synchronous reads
        variable twiTimeout 0

        # callback on channel error
        variable errCallback {}

        # channel error flag
        variable errFlag 0

        # this is 1 only while in open
        variable inOpen 1
    }

    # remember channel
    set ${ns}::chan $chan

    # if parent channel given, clone some information
    if {$parent ne {}} {
        set ${ns}::firmwareVersion [set ${parent}::firmwareVersion]
        set ${ns}::firmwareName [set ${parent}::firmwareName]
        set ${ns}::digitalIns [set ${parent}::digitalIns]
        set ${ns}::digitalOuts [set ${parent}::digitalOuts]
        set ${ns}::analogIns [set ${parent}::analogIns]
        set ${ns}::deviceConfig [set ${parent}::deviceConfig]
        namespace eval $ns {
            variable txBuf ""
        }
    }

    # done if not a real I/O channel provided
    if {$chan eq {}} {
        set ${ns}::inOpen 0
        return $cmd
    }

    # send reset so digital reporting works correctly without a board reset
    if {!$noreset} {
        write $ns [binary format c 0xff]
    }

    # get firmware name and version
    if {[write $ns [binary format ccc 0xf0 0x79 0xf7]]} {
        close $ns
        error "failed to query firmware info"
    }
    set timer [after 5000 [list set ${ns}::firmwareName ""]]
    vwait ${ns}::firmwareName
    after cancel $timer
    if {[set ${ns}::firmwareName] eq ""} {
        close $ns
        error "failed to read firmware info"
    }

    # get board capabilities
    if {[write $ns [binary format ccc 0xf0 0x6b 0xf7]]} {
        close $ns
        error "failed to query device configuration"
    }
    set timer [after 1000 [list set ${ns}::deviceConfig ""]]
    vwait ${ns}::deviceConfig
    after cancel $timer
    if {[set ${ns}::deviceConfig] eq ""} {
        close $ns
        error "failed to read device configuration"
    }

    # get board analog mapping
    if {[write $ns [binary format ccc 0xf0 0x69 0xf7]]} {
        close $ns
        error "failed to query analog mapping"
    }
    set timer [after 1000 [list set ${ns}::analogMapping ""]]
    vwait ${ns}::analogMapping
    after cancel $timer
    if {[set ${ns}::analogMapping] eq ""} {
        close $ns
        error "failed to read analog mapping"
    }

    # create lists for I/O values
    set ${ns}::analogIns [lrepeat [dict size [set ${ns}::analogMapping]] 0]
    set numPorts [getNumPorts $ns]
    set ${ns}::digitalIns [lrepeat $numPorts 0]
    set ${ns}::digitalOuts [lrepeat $numPorts 0]

    # disable streaming if no reset was performed
    if {$noreset} {
        set maxport [getNumPorts $ns]
        for {set port 0} {$port < $maxport} {incr port} {
            write $ns [binary format cc [expr {0xd0 + $port}] 0]
        }
    }

    # clear flag
    set ${ns}::inOpen 0

    return $cmd
}

# return all open tfirmata handles
proc tfirmata::handles {} {
    set result {}
    foreach ns [namespace children [namespace current]] {
        if {[info exists ${ns}::chan]} {
            lappend result $ns
        }
    }
    return $result
}

# write data on tfirmata handle
proc tfirmata::write {ns data} {
    if {[set ${ns}::errFlag]} {
        error "broken I/O channel on \"$ns\""
        return -1
    }
    set chan [set ${ns}::chan]

    if {$chan eq {}} {
        # virtual channel used for task data
        append ${ns}::txBuf $data
        return 0
    }

    if {[catch {
        ::chan puts -nonewline $chan $data
        ::chan flush $chan
    } err]} {
        # don't fully track errors while in open
        if {[set ${ns}::inOpen]} {
            return -1
        }
        ::chan event $chan readable {}
        set ${ns}::errFlag 1
        set cmd [set ${ns}::errCallback]
        if {$cmd ne {}} {
            after cancel [list $cmd $ns]
            after idle [list $cmd $ns]
        } else {
            error "I/O error on channel: $err"
        }
        return -1
    }
    return 0
}

# parse sub command
proc tfirmata::subCmd {ns args} {
    set args [lassign $args subCmd]
    if {$subCmd eq ""} {
        error "no tfirmata sub command provided"
    }
    if {$subCmd ni {
        mode dstream astream dget dset dset25 aget aset
        firmware state amapping errors dcommand acommand period
        servolimits twiconfig twiget twiset reset close errcommand
        step stepoff stepconfig stepwait ecommand eattach equery ereset
        ereport edetach eget serconfig sercommand serwrite serread
        serclose serflush serlisten owsearch owtrans owconfig owcallback
        taskdata createtask deletetask delaytask scheduletask
        resettasks
    }} {
        error "unknown tfirmata subcommand \"$subCmd\""
    }
    if {![info exists ${ns}::chan]} {
        error "not an open tfirmata channel \"$subCmd\""
    }
    if {[llength $args] == 0} {
        $subCmd $ns
    } else {
        $subCmd $ns {*}$args
    }
}

# configure pin mode(s) to in, out, analog, pwm, servo, etc.
proc tfirmata::mode {ns args} {
    variable pinModes
    if {[llength $args] < 2} {
        error "expect pin numbers followed by pin mode"
    }
    while {1} {
        if {$args eq ""} {
            break
        }
        set pins [list]
        while {1} {
            if {$args eq ""} {
                error "expect pin mode to follow pin numbers"
            }
            set args [lassign $args i]
            if {$i in [lrange $pinModes 0 4]} {
                break
            }
            lappend pins $i
        }
        if {$pins eq ""} {
            error "expect pin numbers before \"$i\""
        }
        set mode $i
        foreach pin $pins {
            if {![dict exists [set ${ns}::deviceConfig] $pin]} {
                error "invalid pin \"$pin\""
            }
            set modeId [lsearch $pinModes $mode]
            set modeIds [dict get [set ${ns}::deviceConfig] $pin]
            if {![dict exists $modeIds $modeId]} {
                error "invalid mode \"$mode\" for pin \"$pin\""
            }
            write $ns [binary format ccc 0xf4 $pin $modeId]
        }
    }
}

# enable/disable streaming of digital port(s)
proc tfirmata::dstream {ns args} {
    if {[llength $args] < 2} {
        error "expect port numbers followed by \"on\" or \"off\""
    }
    while {1} {
        if {$args eq ""} {
            break
        }
        set ports [list]
        while {1} {
            if {$args eq ""} {
                error "expect \"on\" or \"off\" to follow pin numbers"
            }
            set args [lassign $args i]
            if {$i in {on off}} {
                break
            }
            lappend ports $i
        }
        if {$ports eq ""} {
            error "expect port numbers before \"$i\""
        }
        set en 0
        if {$i eq "on"} {
            set en 1
        }
        foreach port $ports {
            if {![string is integer $port] || $port < 0 || \
                    $port > [getNumPorts $ns]} {
                error "invalid port \"$port\""
            }
            write $ns [binary format cc [expr {0xd0 + $port}] $en]
        }
    }
}

# enable/disable streaming of analog input(s).
proc tfirmata::astream {ns args} {
    if {[llength $args] < 2} {
        error "expect pin numbers followed by \"on\" or \"off\""
    }
    while {1} {
        if {$args eq ""} {
            break
        }
        set pins [list]
        while {1} {
            if {$args eq ""} {
                error "expect \"on\" or \"off\" to follow pin numbers"
            }
            set args [lassign $args i]
            if {$i in {on off}} {
                break
            }
            lappend pins $i
        }
        if {$pins eq ""} {
            error "expect pin numbers before \"$i\""
        }
        set en 0
        if {$i eq "on"} {
            set en 1
        }
        foreach pin $pins {
            if {![dict exists [set ${ns}::analogMapping] $pin]} {
                error "invalid analog pin \"$pin\""
            }
            write $ns [binary format cc [expr {0xc0 + $pin}] $en]
        }
    }
}

# get digital pin value(s)
proc tfirmata::dget {ns args} {
    set bitVals {}
    foreach pin $args {
        if {![dict exists [set ${ns}::deviceConfig] $pin]} {
            error "invalid pin \"$pin\""
        }
        set port [expr {$pin >> 3}]
        set bit [expr {$pin & 0x7}]
        set portVal [lindex [set ${ns}::digitalIns] $port]
        lappend bitVals [expr {($portVal & (1 << $bit)) != 0}]
    }
    return $bitVals
}

# set digital pin value(s)
proc tfirmata::dset {ns args} {
    set ports {}
    foreach {pin val} $args {
        if {![dict exists [set ${ns}::deviceConfig] $pin]} {
            error "invalid pin \"$pin\""
        }
        if {$val ni {0 1}} {
            error "value should be 0 or 1"
        }
        set port [expr {$pin >> 3}]
        set pin [expr {$pin & 0x7}]
        set v [lindex [set ${ns}::digitalOuts] $port]
        if {$val} {
            set v [expr {$v | (1 << $pin)}]
        } else {
            set v [expr {$v & ~(1 << $pin)}]
        }
        lset ${ns}::digitalOuts $port $v
        if {$port ni $ports} {
            lappend ports $port
        }
    }
    foreach port $ports {
        set v [lindex [set ${ns}::digitalOuts] $port]
        set ls [expr {$v & 0x7f}]
        set ms [expr {($v >> 7) & 0x7f}]
        write $ns [binary format ccc [expr {0x90 + $port}] $ls $ms]
    }
}

# set digital pin value(s), 2.5 command format
proc tfirmata::dset25 {ns args} {
    set cmds ""
    set n 0
    if {[set ${ns}::firmwareVersion] < 2.5} {
        error "unsupported by firmware"
    }
    foreach {pin val} $args {
        if {![dict exists [set ${ns}::deviceConfig] $pin]} {
            error "invalid pin \"$pin\""
        }
        if {$val ni {0 1}} {
            error "value should be 0 or 1"
        }
        append cmds [binary format ccc 0xf5 $pin $val]
        incr n
    }
    if {$n} {
        write $ns $cmds
    }
}

# get analog in value(s)
proc tfirmata::aget {ns args} {
    set analogVals {}
    foreach ain $args {
        if {![dict exists [set ${ns}::analogMapping] $ain]} {
            error "invalid analog in \"$ain\""
        }
        lappend analogVals [lindex [set ${ns}::analogIns] $ain]
    }
    return $analogVals
}

# set analog output(s)
proc tfirmata::aset {ns args} {
    foreach {pin val} $args {
        set lsb [expr {$val & 0x7f}]
        set msb [expr {($val >> 7) & 0x7f}]
        write $ns [binary format cccccc 0xf0 0x6f $pin $lsb $msb 0xf7]
    }
}

# get Arduino firmware information
proc tfirmata::firmware {ns} {
    return [list [set ${ns}::firmwareName] [set ${ns}::firmwareVersion]]
}

# get pin state(s). State includes the mode a pin is set to.
proc tfirmata::state {ns args} {
    variable pinModes
    if {$args eq ""} {
        error "expect a pin number, list of pin numbers, or \"all\""
    } elseif {$args eq "all"} {
        set pins [dict keys [set ${ns}::deviceConfig]]
    } else {
        set pins $args
    }
    set states [list]
    foreach pin $pins {
        if {![dict exists [set ${ns}::deviceConfig] $pin]} {
            error "invalid pin \"$pin\""
        }
        if {[write $ns [binary format cccc 0xf0 0x6d $pin 0xf7]]} {
            error "error writing pin \"$pin\" query"
            break
        }
        vwait ${ns}::state
        if {[set ${ns}::pin] != $pin} {
            error "error reading back pin \"$pin\" state"
        }
        set mode [lindex $pinModes [set ${ns}::mode]]
        lappend states $pin [list $mode {*}[set ${ns}::state]]
    }
    return $states
}

# get analog channel mapping(s)
proc tfirmata::amapping {ns args} {
    set mappings [list]
    foreach arg $args {
        if {![dict exists [set ${ns}::analogMapping] $arg]} {
            error "error getting mapping for analog input \"$arg\""
        }
        lappend mappings [dict get [set ${ns}::analogMapping] $arg]
    }
    return $mappings
}

# set digital message event command
proc tfirmata::dcommand {ns cmd} {
    set ${ns}::digitalCallback $cmd
}

# set analog message event command
proc tfirmata::acommand {ns cmd} {
    set ${ns}::analogCallback $cmd
}

# set encoder message event command
proc tfirmata::ecommand {ns cmd} {
    set ${ns}::encoderCallback $cmd
}

# set sampling period
proc tfirmata::period {ns ms} {
    if {[string is integer $ms] && $ms >= 0 && $ms <= 0x3fff} {
        write $ns [binary format ccccc 0xf0 0x7a {*}[decompose $ms] 0xf7]
    } else {
        error "analog sampling period should be between 0 and 0x3fff"
    }
}

# set servo limits for pin(s)
proc tfirmata::servolimits {ns args} {
    if {$args == {}} {
        error "expect pin numbers followed by limits"
    }
    while {1} {
        set pins {}
        set limits {}
        while {1} {
            set args [lassign $args arg]
            if {$arg == {}} {
                break
            }
            if {[llength $arg] == 1} {
                lappend pins $arg
            } elseif {[llength $arg] == 2} {
                set limits $arg
                break
            } else {
                error "argument list length error"
            }
        }
        if {$pins == {} && $limits == {}} {
            return
        }
        if {$pins == {}} {
            error "expect pins before limits"
        }
        if {$limits == {}} {
            error "expect limits after pins"
        }
        foreach pin $pins {
            if {![dict exists [set ${ns}::deviceConfig] $pin]} {
                error "invalid pin \"$pin\""
            }
            set modeIds [dict get [set ${ns}::deviceConfig] $pin]
            if {![dict exists $modeIds 4]} {
                error "invalid servo mode setting servo limits for pin \"$pin\""
            }
            set min [lindex $limits 0]
            set max [lindex $limits 1]
            if {![string is integer $min] || $min < 0 || $min > 0x3fff || \
                ![string is integer $max] || $max < 0 || $max > 0x3fff || \
                    $min >= $max} {
                error "invalid servolimits limits"
            }
            write $ns [binary format cccccccc 0xf0 0x70 $pin \
                {*}[decompose $min] {*}[decompose $max] 0xf7]
        }
    }
}

# configure twi interface pins with optional delay
proc tfirmata::twiconfig {ns {delay 0}} {
    if {![string is integer $delay] || $delay < 0 || $delay > 0x3fff} {
        error "delay error"
    }
    write $ns [binary format ccccc 0xf0 0x78 {*}[decompose $delay] 0xf7]
}

# perform read messages on TWI. Returns list of bytes read if blocking read,
# otherwise returns {} and later runs. For slave addresses, see comments
# in twiset below.
proc tfirmata::twiget {ns args} {
    if {$args == {}} {
        error "too few arguments"
    }
    set repeat 0
    set bytes {}
    set codeSpecified 0
    set code {}
    set i 0
    set len [llength $args]
    while {1} {
        if {$i == $len} {
            break
        }
        set arg [lindex $args $i]
        if {$arg eq "-repeat"} {
            if {$i != 0} {
                error "argument error in -repeat"
            }
            set repeat 1
        } elseif {$arg eq "-stop"} {
            if {$i != 0 || $len != 2} {
                error "argument error in -stop"
            }
            set twiAddr [lindex $args 1]
            if {$twiAddr < 0} {
                error "twiget address error"
            }
            # auto restart and 10 bit mode
            set addrHi [expr {($twiAddr & 0x600) >> 8}]
            set twiAddr [expr {$twiAddr & (~0x600)}]
            if {![dict exists [set ${ns}::twiCallbacks] $twiAddr]} {
                error "address error in -stop"
            }
            if {$twiAddr > 0x3ff} {
                error "address out of range"
            }
            dict unset ${ns}::twiCallbacks $twiAddr
            # send message to Arduino to clear read
            set addrHi [expr {$addrHi | 0x18 | ($twiAddr >> 7)}]
            set addrLo [expr {$twiAddr & 0x7f}]
            if {$twiAddr > 0x7f} {
                set addrHi [expr {$addrHi | 0x20}]
            }
            write $ns [binary format ccccc 0xf0 0x76 $addrLo $addrHi 0xf7]
            return
        } elseif {$arg eq "-command"} {
            if {$bytes == {}} {
                error "argument error in -command"
            }
            if {$i + 2 != $len} {
                error "argument error in -command"
            }
            set codeSpecified 1
            set code [lindex $args $i+1]
            break
        } else {
            # otherwise parse bytes for TWI device
            if {![string is integer $arg] || $arg < 0 || $arg > 0xff} {
                error "error in data bytes"
            }
            lappend bytes [expr {$arg}]
        }
        incr i
    }

    if {$bytes == {}} {
        error "no data bytes"
    }

    set bytes [lassign $bytes twiAddr]
    set addrHi [expr {($twiAddr & 0x6000) >> 8}]
    set twiAddr [expr {$twiAddr & (~0x6000)}]
    set twiControl [lrange $bytes 0 end-1]
    set numTwiReadBytes [lindex $bytes end]

    if {$twiAddr < 0 || $twiAddr > 0x3ff} {
        error "address out of range"
    }
    set addrLo [expr {$twiAddr & 0x7f}]
    if {$repeat} {
        set addrHi [expr {$addrHi | 0x10}]
    } else {
        set addrHi [expr {$addrHi | 0x08}]
    }
    if {$twiAddr > 0x7f} {
        set addrHi [expr {$addrHi | 0x20 | ($twiAddr >> 7)}]
    }

    # update TWI callbacks if TWI callback provided
    if {$codeSpecified} {
        set entriesIndex [searchTwiCallbacks $ns $twiAddr $twiControl]
        if {$entriesIndex != -1} {
            deleteTwiCallback $ns $twiAddr $entriesIndex
        }
        set d [dict create twiControl $twiControl \
                numBytes $numTwiReadBytes repeat $repeat code $code]
        addTwiCallback $ns $twiAddr $d
    }

    # send message to Arduino to set up read
    set bytes [decompose {*}$bytes]
    set wrErr [write $ns \
        [binary format cccc[string repeat c [llength $bytes]]c \
             0xf0 0x76 $addrLo $addrHi {*}$bytes 0xf7]]

    # block for response from Arduino if no TWI callback provided
    if {!$wrErr && !$codeSpecified} {
        set clen [expr {[llength $twiControl] - 1}]
        set ${ns}::twiTimeout 0
        while {1} {
            set timer [after 250 [subst -nocommands {
                set ${ns}::twiReplyBytes [set ${ns}::twiReplyBytes]
                set ${ns}::twiTimeout 1
            }]]
            vwait ${ns}::twiReplyBytes
            after cancel $timer
            if {[set ${ns}::twiTimeout]} {
                break
            }
            set replyBytes [lassign [set ${ns}::twiReplyBytes] replyAddr]
            set replyControl [lrange $replyBytes 0 $clen]
            if {$replyAddr == $twiAddr && $replyControl == $twiControl} {
                set offs [expr {$numTwiReadBytes - 1}]
                return [lrange $replyBytes end-$offs end]
            }
        }
    }

    return {}
}

# perform write message on TWI, 0x4000 or'ed to address enables
# "auto restart transmission" mode which is required for MMA8452Q
# and MPL3115As chips according to firmata protocol description,
# 0x2000 or'ed to address forces 10 bit addressing. However, the
# ConfigurableFirmata implementation on Arduino does not support
# 10 bit addressing mode.
proc tfirmata::twiset {ns args} {
    if {[llength $args] < 2} {
        error "too few arguments"
    }
    set args [lassign $args twiAddr]
    if {![string is integer $twiAddr] || $twiAddr < 0} {
        error "address out of range"
    }
    set addrHi [expr {($twiAddr & 0x6000) >> 8}]
    set twiAddr [expr {$twiAddr & (~0x6000)}]
    if {$twiAddr > 0x3ff} {
        error "address out of range"
    } elseif {$twiAddr > 0x7f} {
        set addrHi [expr {$addrHi | 0x20 | ($twiAddr >> 7)}]
    }
    set addrLo [expr {$twiAddr & 0x7f}]
    foreach arg $args {
        if {![string is integer $arg] || $arg < 0 || $arg > 0xff} {
            error "error in data bytes"
        }
    }
    set data [decompose {*}$args]
    write $ns [binary format cccc[string repeat c [llength $data]]c \
        0xf0 0x76 $addrLo $addrHi {*}$data 0xf7]
}

# set channel error handler
proc tfirmata::errcommand {ns cmd} {
    set ${ns}::errCallback $cmd
}

# get number of receive errors
proc tfirmata::errors {ns} {
    return [set ${ns}::errorCount]
}

# reset Arduino firmware
proc tfirmata::reset {ns} {
    # send reset
    write $ns [binary format c 0xff]
}

# close channel associated with command and delete command
proc tfirmata::close {ns} {
    set chan [set ${ns}::chan]
    if {$chan ne {}} {
        ::close $chan
    }
    namespace delete $ns
    rename $ns {}
}

# sleep for ms milliseconds by waiting on the Tcl event loop
proc tfirmata::sleep {ms} {
    set timer [after $ms [list set [namespace current]::wakeup 1]]
    vwait [namespace current]::wakeup
    after cancel $timer
}

# -----------------------------------------------------------------------------

# parse serial data received from Arduino
proc tfirmata::parseSerialReadData {ns chan} {
    namespace upvar $ns rxBuf rxBuf

    # disable channel handler on EOF
    if {[::chan eof $chan]} {
        ::chan event $chan readable {}
        set ${ns}::errFlag 1
        set cmd [set ${ns}::errCallback]
        if {$cmd ne {}} {
            after cancel [list $cmd $ns]
            after idle [list $cmd $ns]
        } else {
            error "EOF on channel"
        }
        return
    }

    # read all received serial port bytes and append to rxBuf
    if {[catch {set a [::chan read $chan]} err]} {
        ::chan event $chan readable {}
        set ${ns}::errFlag 1
        set cmd [set ${ns}::errCallback]
        if {$cmd ne {}} {
            after cancel [list $cmd $ns]
            after idle [list $cmd $ns]
        } else {
            error "I/O error on channel: $err"
        }
        return
    }

    binary scan $a H[expr {[string length $a] * 2}] bytes
    append rxBuf $bytes

    while {1} {
        # find start of message
        set skippedBytes 0
        while {1} {
            if {$rxBuf eq ""} {
                return
            }
            scan [string range $rxBuf 0 1] %x b
            if {$b == 0xf9 || ($b >=0x90 && $b <= 0x9f) || \
                    ($b >= 0xe0 && $b <= 0xef) || $b == 0xf0} {
                break
            } else {
                set rxBuf [string range $rxBuf 2 end]
                set skippedBytes 1
            }
        }

        if {$skippedBytes} {
            incr ${ns}::errorCount
        }

        # parse message
        if {$b == 0xf9} {
            set rv [parseVersionMsg $ns $rxBuf]
        } elseif {$b >= 0x90 && $b <= 0x9f} {
            set rv [parseDigitalMsg $ns $rxBuf]
        } elseif {$b >= 0xe0 && $b <= 0xef} {
            set rv [parseAnalogMsg $ns $rxBuf]
        } elseif {$b == 0xf0} {
            if {[string length $rxBuf] < 6} {
                return
            }
            scan [string range $rxBuf 2 3] %x b
            if {$b == 0x79} {
                set rv [parseFirmwareMsg $ns $rxBuf]
            } elseif {$b == 0x6c} {
                set rv [parseCapabilitiesMsg $ns $rxBuf]
            } elseif {$b == 0x6a} {
                set rv [parseAnalogMappingMsg $ns $rxBuf]
            } elseif {$b == 0x6e} {
                set rv [parsePinStateMsg $ns $rxBuf]
            } elseif {$b == 0x77} {
                set rv [parseTwiMsg $ns $rxBuf]
            } elseif {$b == 0x60} {
                set rv [parseSerialMsg $ns $rxBuf]
            } elseif {$b == 0x61} {
                set rv [parseEncMsg $ns $rxBuf]
            } elseif {$b == 0x72} {
                set rv [parseStepperMsg $ns $rxBuf]
            } elseif {$b == 0x73} {
                set rv [parseOwMsg $ns $rxBuf]
            } elseif {$b == 0x71} {
                set rv [parseStringMsg $ns $rxBuf]
            } elseif {$b == 0x7b} {
                set rv [parseTaskMsg $ns $rxBuf]
            } else {
                set rv 0
            }
        }

        # if no characters consumed, try again later
        if {$rv == 0} {
            return
        }

        # if error parsing message, trim buffer for searching for next message
        if {$rv == -1} {
            set rxBuf [string range $rxBuf 2 end]
            incr ${ns}::errorCount
            continue
        }

        # otherwise trim buffer to account for consumed bytes
        set rxBuf [string range $rxBuf $rv end]
    }
}

# parses buf for version message. Returns -1 if format error, 0 if not enough
# bytes in buf, or number of bytes consumed if successfully parsed
proc tfirmata::parseVersionMsg {ns buf} {
    if {[string length $buf] < 6} {
        return 0
    }
    scan [string range $buf 2 3] %x major
    scan [string range $buf 4 5] %x minor
    if {($major & 0x80) || ($minor & 0x80)} {
        return -1
    }
    set ${ns}::firmwareVersion $major.$minor
    return 6
}

# parses buf for digital message. Returns -1 if format error, 0 if not enough
# bytes in buf, or number of bytes consumed if successfully parsed
proc tfirmata::parseDigitalMsg {ns buf} {
    if {[string length $buf] < 6} {
        return 0
    }
    scan [string range $buf 0 1] %x port
    set port [expr {$port - 0x90}]
    scan [string range $buf 2 3] %x ls
    scan [string range $buf 4 5] %x ms
    if {($ls & 0x80) || ($ms & 0x80)} {
        return -1
    }
    set value [expr {$ms << 7 | $ls}]
    if {[set ${ns}::deviceConfig] ne ""} {
        lset ${ns}::digitalIns $port $value
    }
    if {[set ${ns}::digitalCallback] ne ""} {
        set code [set ${ns}::digitalCallback]
        regsub -all %P $code $port code
        regsub -all %V $code $value code
        namespace eval :: $code
    }
    return 6
}

# parses buf for analog message. Returns -1 if format error, 0 if not enough
# bytes in buf, or number of bytes consumed if successfully parsed
proc tfirmata::parseAnalogMsg {ns buf} {
    if {[string length $buf] < 6} {
        return 0
    }
    scan [string range $buf 0 1] %x channel
    set channel [expr {$channel - 0xe0}]
    scan [string range $buf 2 3] %x ls
    scan [string range $buf 4 5] %x ms
    if {($ls & 0x80) || ($ms & 0x80)} {
        return -1
    }
    set value [expr {($ms << 7) | $ls}]
    if {[set ${ns}::deviceConfig] ne ""} {
        lset ${ns}::analogIns $channel $value
    }
    if {[set ${ns}::analogCallback] ne ""} {
        set code [set ${ns}::analogCallback]
        regsub -all %C $code $channel code
        regsub -all %V $code $value code
        namespace eval :: $code
    }
    return 6
}

# parses buf for SysEx firmware message. Returns -1 if format error, 0 if not
# enough bytes in buf, or number of bytes consumed if successfully parsed
proc tfirmata::parseFirmwareMsg {ns buf} {
    set i 4
    set len [string length $buf]
    while {$i < $len} {
        scan [string range $buf $i $i+1] %x b
        if {$b == 0xf7} {
            if {$i < 10} {
                return -1
            }
            set ${ns}::firmwareVersion $major.$minor
            set decname ""
            while {[llength $name]} {
                set c [lindex $name 0]
                set lo [expr {$c & 0x7f}]
                set hi [expr {$c >> 8}]
                set name [lrange $name 1 end]
                set c [expr {($hi << 7) | $lo}]
                append decname [format %c $c]
            }
            set ${ns}::firmwareName $decname
            return $i
        } elseif {$b & 0x80} {
            return -1
        } else {
            if {$i == 4} {
                set major $b
            } elseif {$i == 6} {
                set minor $b
            } else {
                lappend name $b
            }
        }
        incr i 2
    }
    return 0
}

# parses buf for SysEx capabilites response message. Returns -1 if format
# error, 0 if not enough bytes in buf, or number of bytes consumed if
# successfully parsed
proc tfirmata::parseCapabilitiesMsg {ns buf} {
    set config [dict create]
    set pinNum 0
    set i 4
    set len [string length $buf]
    while {1} {
        set modes [list]
        while {1} {
            if {$i + 4 > $len} {
                return 0
            }
            scan [string range $buf $i $i+1] %x b
            if {$b & 0x80} {
                return -1
            }
            if {$b == 0x7f} {
                dict set config $pinNum $modes
                incr i 2
                break
            }
            set mode $b
            scan [string range $buf $i+2 $i+3] %x resolution
            if {$mode ni {0 1 2 3 4 5 6 7 8 9 10 11}} {
                return -1
            }
            if {$mode < 8 && $resolution > 16} {
                return -1
            }
            lappend modes $mode $resolution
            incr i 4
        }
        scan [string range $buf $i $i+1] %x b
        if {$b == 0xf7} {
            if {$modes == {}} {
                return -1
            }
            incr i 2
            break
        }
        incr pinNum
    }
    set ${ns}::deviceConfig $config
    return $i
}

# parses buf for SysEx analog mapping message. Returns -1 if format error, 0
# if not enough bytes in buf, or number of bytes consumed if successfully
# parsed
proc tfirmata::parseAnalogMappingMsg {ns buf} {
    set mapping {}
    set pin 0
    set i 4
    set len [string length $buf]
    while {1} {
        if {$i + 2 > $len} {
            return 0
        }
        scan [string range $buf $i $i+1] %x channel
        if {$channel == 0xf7} {
            if {$mapping == {}} {
                return -1
            }
            incr i 2
            break
        }
        if {$channel & 0x80} {
            return -1
        }
        if {$channel != 0x7f} {
            dict set mapping $channel $pin
        }
        incr pin
        incr i 2
    }
    set ${ns}::analogMapping $mapping
    return $i
}

# parses buf for SysEx pin state message. Returns -1 if format error, 0 if not
# enough bytes in buf, or number of bytes consumed if successfully parsed
proc tfirmata::parsePinStateMsg {ns buf} {
    set len [string length $buf]
    if {$len < 12} {
        return 0
    }
    scan [string range $buf 4 5] %x pin
    scan [string range $buf 6 7] %x mode
    if {($pin & 0x80) || ($mode & 0x80)} {
        return -1
    }
    set state 0
    set count 0
    set i 8
    while {1} {
        if {$i + 2 > $len} {
            return 0
        }
        scan [string range $buf $i $i+1] %x b
        if {$b == 0xf7} {
            if {$i == 8} {
                return -1
            }
            set ${ns}::pin $pin
            set ${ns}::mode $mode
            set ${ns}::state $state
            incr i 2
            break
        }
        if {$b & 0x80} {
            return -1
        }
        set state [expr {$state + ($b << (8 * $count))}]
        incr count
        incr i 2
    }
    return $i
}

# parses buf for SysEx TWI reply message. Returns -1 if format error, 0 if not
# enough bytes in buf, or number of bytes consumed if successfully parsed
proc tfirmata::parseTwiMsg {ns buf} {
    set len [string length $buf]
    if {$len < 18} {
        return 0
    }
    set bytes [list]
    set i 4
    while {1} {
        if {$i + 2 > $len} {
            return 0
        }
        scan [string range $buf $i $i+1] %x ls
        if {$ls == 0xf7} {
            incr i 2
            break
        }
        if {$i + 4 > [string length $buf]} {
            return 0
        }
        scan [string range $buf $i+2 $i+3] %x ms
        if {($ls & 0x80) || ($ms & 0x80)} {
            return -1
        }
        lappend bytes [expr {($ms << 7) | $ls}]
        incr i 4
    }
    set twiAddr [lindex $bytes 0]
    set len [getTwiControlLen $ns $twiAddr]
    if {$len != -1} {
        set twiControl [lrange $bytes 1 $len]
        set index [searchTwiCallbacks $ns $twiAddr $twiControl]
        if {$index != -1} {
            set code [getTwiCallbackCode $ns $twiAddr $index]
            if {![getTwiCallbackRepeat $ns $twiAddr $index]} {
                deleteTwiCallback $ns $twiAddr $index
            }
            regsub -all %C $code [lrange $bytes 1 $len] code
            regsub -all %D $code [lrange $bytes $len+1 end] code
            namespace eval :: $code
        }
    }
    set ${ns}::twiReplyBytes $bytes
    return $i
}

# parses buf for SysEx onewire reply message. Returns -1 if format error,
# 0 if not enough bytes in buf, or number of bytes consumed if
# successfully parsed
proc tfirmata::parseOwMsg {ns buf} {
    set len [string length $buf]
    if {$len < 10} {
        return 0
    }
    set i 4
    scan [string range $buf $i $i+1] %x type
    incr i 2
    scan [string range $buf $i $i+1] %x pin
    incr i 2
    set done 0
    set bytes {}
    while {$i < $len} {
        scan [string range $buf $i $i+1] %x b0
        incr i 2
        if {$b0 == 0xf7} {
            set done 1
            break
        }
        lassign {0 0 0 0 0 0 0} b1 b2 b3 b4 b5 b6 b7
        scan [string range $buf $i $i+1] %x b1
        incr i 2
        if {$b1 != 0xf7} {
            scan [string range $buf $i $i+1] %x b2
            incr i 2
            if {$b2 != 0xf7} {
                scan [string range $buf $i $i+1] %x b3
                incr i 2
                if {$b3 != 0xf7} {
                    scan [string range $buf $i $i+1] %x b4
                    incr i 2
                    if {$b4 != 0xf7} {
                        scan [string range $buf $i $i+1] %x b5
                        incr i 2
                        if {$b5 != 0xf7} {
                            scan [string range $buf $i $i+1] %x b6
                            incr i 2
                            if {$b6 != 0xf7} {
                                scan [string range $buf $i $i+1] %x b7
                                incr i 2
                            } else {
                                set done 1
                            }
                        } else {
                            set done 1
                        }
                    } else {
                        set done 1
                    }
                } else {
                    set done 1
                }
            } else {
                set done 1
            }
        } else {
            set done 1
        }
        lappend bytes [expr {(($b0 >> 0) & 0x7f) | (($b1 << 7) & 0x80)}]
        lappend bytes [expr {(($b1 >> 1) & 0x7f) | (($b2 << 6) & 0xc0)}]
        lappend bytes [expr {(($b2 >> 2) & 0x3f) | (($b3 << 5) & 0xe0)}]
        lappend bytes [expr {(($b3 >> 3) & 0x1f) | (($b4 << 4) & 0xf0)}]
        lappend bytes [expr {(($b4 >> 4) & 0x0f) | (($b5 << 3) & 0xf8)}]
        lappend bytes [expr {(($b5 >> 5) & 0x07) | (($b6 << 2) & 0xfc)}]
        lappend bytes [expr {(($b6 >> 6) & 0x03) | (($b7 << 1) & 0xfe)}]
    }
    if {$done} {
        # strip off fill bytes from above
        set bytes [lrange $bytes 0 [expr {($len - 10) / 2 * 7 / 8 - 1}]]
        if {$type == 0x43} {
            set bytes [lassign $bytes lo hi]
            if {[crc8 $bytes]} {
                # CRC error, don't pass to callback command
                return $i
            }
            set tid [expr {$lo | ($hi << 8)}]
            set code [getOwCallback $ns $pin $tid]
            if {$code ne ""} {
                regsub -all %C $code $pin code
                regsub -all %T $code $tid code
                regsub -all %V $code [list $bytes] code
                namespace eval :: $code
            }
        } else {
            set addrs {}
            foreach {b0 b1 b2 b3 b4 b5 b6 b7} $bytes {
                if {[crc8 [list $b0 $b1 $b2 $b3 $b4 $b5 $b6 $b7]]} {
                    # CRC error, don't add to address list
                    continue
                }
                # onewire address in Linux format
                # "aa.bbccddeeffgghh" where "aa" is chip id
                lappend addrs [format %02X.%02X%02X%02X%02X%02X%02X%02X \
                    $b0 $b1 $b2 $b3 $b4 $b5 $b6 $b7]
            }
            set code0 [set ${ns}::owSearchCallback]
            if {$code0 ne ""} {
                set alarms [expr {$type == 0x45}]
                regsub -all %C $code0 $pin code0
                regsub -all %S $code0 $alarms code0
                foreach a $addrs {
                    set code $code0
                    regsub -all %A $code $a code
                    namespace eval :: $code
                }
            }
        }
        return $i
    }
    return 0
}

# parses buf for SysEx encoder reply message. Returns -1 if format error,
# 0 if not enough bytes in buf, or number of bytes consumed if successfully
# parsed
proc tfirmata::parseEncMsg {ns buf} {
    set len [string length $buf]
    if {$len < 16} {
        return 0
    }
    set done 0
    set i 4
    set data {}
    while {$i < $len} {
        scan [string range $buf $i $i+1] %x b
        incr i 2
        if {$b == 0xf7} {
            set done 1
            break
        }
        if {$len - $i < 8} {
            break
        }
        set dir [expr {($b & 0x40) ? -1 : 1}]
        set enc [expr {$b & 0x3f}]
        scan [string range $buf $i $i+1] %x v0
        incr i 2
        scan [string range $buf $i $i+1] %x v1
        incr i 2
        scan [string range $buf $i $i+1] %x v2
        incr i 2
        scan [string range $buf $i $i+1] %x v3
        incr i 2
        set v [expr {$dir * ($v0 | ($v1 << 7) | ($v2 << 14) | ($v3 << 21))}]
        if {![info exists ${ns}::encoder$enc] ||
            [set ${ns}::encoder$enc] != $v} {
            lappend data $enc $v
        }
        set ${ns}::encoder$enc $v
    }
    if {$done} {
        if {[set ${ns}::encoderCallback] ne "" && [llength $data]} {
            foreach {enc value} $data {
                set code [set ${ns}::encoderCallback]
                regsub -all %C $code $enc code
                regsub -all %V $code $value code
                namespace eval :: $code
            }
        }
        return $i
    }
    return 0
}

# parses buf for SysEx stepper message. Returns -1 if format error, 0 if
# not enough bytes in buf, or number of bytes consumed if successfully
# parsed
proc tfirmata::parseStepperMsg {ns buf} {
    set len [string length $buf]
    if {$len < 6} {
        return 0
    }
    set i 4
    while {$i < $len} {
        scan [string range $buf $i $i+1] %x b
        incr i 2
        if {$b == 0xf7} {
            break
        }
        if {$b >= 0 && $b <= 5} {
            set m [set ${ns}::wakeup]
            set m [expr {$m | (1 << $b)}]
            set ${ns}::wakeup $m
        }
    }
    return $i
}

# parses buf for SysEx serial reply message. Returns -1 if format error,
# 0 if not enough bytes in buf, or number of bytes consumed if successfully
# parsed
proc tfirmata::parseSerialMsg {ns buf} {
    set len [string length $buf]
    if {$len < 8} {
        return 0
    }
    set i 4
    scan [string range $buf $i $i+1] %x chan
    incr i 2
    if {($chan & 0xf0) != 0x40} {
        return -1
    }
    set chan [expr {$chan & 0x0f}]
    set data {}
    while {$i < $len} {
        scan [string range $buf $i $i+1] %x b0
        incr i 2
        if {$b0 == 0xf7} {
            break
        }
        scan [string range $buf $i $i+1] %x b1
        incr i 2
        if {$b1 == 0xf7} {
            break
        }
        if {($b0 & 0x80) || ($b1 & 0x80)} {
            return -1
        }
        append data [format %c [expr {($b1 << 7) | $b0}]]
    }
    if {[set ${ns}::serialCallback] ne "" && [llength $data]} {
        set code [set ${ns}::serialCallback]
        regsub -all %C $code $chan code
        regsub -all %V $code [list $data] code
        namespace eval :: $code
    }
    return $i
}

# parses buf for SysEx string message. Returns -1 if format error, 0 if not
# enough bytes in buf, or number of bytes consumed if successfully parsed
proc tfirmata::parseStringMsg {ns buf} {
    set i 4
    set len [string length $buf]
    while {$i < $len} {
        scan [string range $buf $i $i+1] %x b
        if {$b == 0xf7} {
            if {$i < 6} {
                return -1
            }
            set decstr ""
            while {[llength $str]} {
                set c [lindex $str 0]
                set str [lrange $str 1 end]
                set lo [expr {$c & 0x7f}]
                set hi [expr {$c >> 8}]
                set c [expr {($hi << 7) | $lo}]
                append decstr [format %c $c]
            }
            puts stderr "tfirmata: $decstr"
            return $i
        } elseif {$b & 0x80} {
            return -1
        } else {
            lappend str $b
        }
        incr i 2
    }
    return 0
}

# parses buf for SysEx scheduler message. Returns -1 if format error, 0 if not
# enough bytes in buf, or number of bytes consumed if successfully parsed
proc tfirmata::parseTaskMsg {ns buf} {
    set i 4
    set len [string length $buf]
    while {$i < $len} {
        scan [string range $buf $i $i+1] %x b
        if {$b == 0xf7} {
            if {$i < 4} {
                return -1
            }
            if {[lindex $str 0] == 8} {
                set tid [lindex $str 1]
                set pos unknown
                if {$i > 10} {
                    set pos [expr {[lindex $str 8] >> 6}]
                    set pos [expr {$pos | [lindex $str 9] << 1}]
                    set pos [expr {$pos | [lindex $str 10] << 8}]
                }
                puts stderr "tfirmata: error in task $tid at $pos"
            }
            return $i
        } elseif {$b & 0x80} {
            return -1
        } else {
            lappend str $b
        }
        incr i 2
    }
    return 0
}

# -----------------------------------------------------------------------------

# returns index of callback in TWI callbacks, or -1 if it doesn't exists
proc tfirmata::searchTwiCallbacks {ns twiAddr twiControl} {
    set cbs [set ${ns}::twiCallbacks]
    if {![dict exists $cbs $twiAddr]} {
        return -1
    }
    set entries [dict get $cbs $twiAddr]
    for {set i 0} {$i < [llength $entries]} {incr i} {
        set e [lindex $entries $i]
        if {$twiControl == [dict get $e twiControl]} {
            return $i
        }
    }
    return -1
}

# delete callback from TWI callbacks
proc tfirmata::deleteTwiCallback {ns twiAddr index} {
    set entries [dict get [set ${ns}::twiCallbacks] $twiAddr]
    set entries [lreplace $entries $index $index]
    if {[llength $entries] == 0} {
        dict unset ${ns}::twiCallbacks $twiAddr
    } else {
        dict set ${ns}::twiCallbacks $twiAddr $entries
    }
}

# add callback to TWI callbacks
proc tfirmata::addTwiCallback {ns twiAddr callback} {
    dict lappend ${ns}::twiCallbacks $twiAddr $callback
}

# returns length of control bytes for TWI device with address twiAddr. Returns
# -1 if TWI address isn't in TWI callbacks
proc tfirmata::getTwiControlLen {ns twiAddr} {
    set cbs [set ${ns}::twiCallbacks]
    if {![dict exists $cbs $twiAddr]} {
        return -1
    }
    set entries [dict get $cbs $twiAddr]
    set e [lindex $entries 0]
    return [llength [dict get $e twiControl]]
}

# returns code for TWI callback, or {} if callback isn't in TWI callbacks
proc tfirmata::getTwiCallbackCode {ns twiAddr index} {
    set cbs [set ${ns}::twiCallbacks]
    if {![dict exists $cbs $twiAddr]} {
        return {}
    }
    set entries [dict get $cbs $twiAddr]
    set e [lindex $entries $index]
    return [dict get $e code]
}

# returns repeat status of TWI callback, or -1 if callback isn't in TWI
# callbacks
proc tfirmata::getTwiCallbackRepeat {ns twiAddr index} {
    set cbs [set ${ns}::twiCallbacks]
    if {![dict exists $cbs $twiAddr]} {
        return -1
    }
    set entries [dict get $cbs $twiAddr]
    set e [lindex $entries $index]
    return [dict get $e repeat]
}

# -----------------------------------------------------------------------------

# search onewire
proc tfirmata::owsearch {ns args} {
    if {$args == {}} {
        error "too few arguments"
    }
    set code {}
    set alarms 0
    set i 0
    set len [llength $args]
    while {1} {
        if {$i == $len} {
            break
        }
        set arg [lindex $args $i]
        if {$arg eq "-alarms"} {
            set alarms 1
        } elseif {$arg eq "-command"} {
            if {$i + 1 >= $len} {
                error "argument error in -command"
            }
            incr i
            set code [lindex $args $i]
        } else {
            set pin [lindex $args $i]
        }
        incr i
    }
    if {![info exists pin]} {
        error "pin missing"
    }
    set modeIds [dict get [set ${ns}::deviceConfig] $pin]
    if {![dict exists $modeIds 7]} {
        error "not supported for pin \"$pin\""
    }
    set ${ns}::owSearchCallback $code
    set cmd [expr {$alarms ? 0x44 : 0x40}]
    write $ns [binary format ccccc 0xf0 0x73 $cmd $pin 0xf7]
}

# config onewire
proc tfirmata::owconfig {ns pin {power 0}} {
    set modeIds [dict get [set ${ns}::deviceConfig] $pin]
    if {![dict exists $modeIds 7]} {
        error "not supported for pin \"$pin\""
    }
    set power [expr {$power ? 0x01 : 0x00}]
    write $ns [binary format cccccc 0xf0 0x73 0x41 $pin $power 0xf7]
}

# perform onewire transaction
proc tfirmata::owtrans {ns args} {
    if {$args == {}} {
        error "too few arguments"
    }
    set cmd 0
    set i 0
    set needtid 0
    set len [llength $args]
    while {1} {
        if {$i == $len} {
            break
        }
        set arg [lindex $args $i]
        if {$arg eq "-reset"} {
            set cmd [expr {$cmd | 0x01}]
        } elseif {$arg eq "-skip"} {
            set cmd [expr {$cmd | 0x02}]
        } elseif {$arg eq "-address"} {
            set cmd [expr {$cmd | 0x04}]
            if {$i + 1 >= $len} {
                error "argument error in -address"
            }
            incr i
            set addr [lindex $args $i]
        } elseif {$arg eq "-read"} {
            set cmd [expr {$cmd | 0x08}]
            if {$i + 1 >= $len} {
                error "argument error in -read"
            }
            incr i
            set rln [lindex $args $i]
            if {![info exists tid]} {
                set needtid 1
            }
        } elseif {$arg eq "-write"} {
            set cmd [expr {$cmd | 0x20}]
            if {$i + 1 >= $len} {
                error "argument error in -write"
            }
            incr i
            set wrdata [lindex $args $i]
        } elseif {$arg eq "-tid"} {
            if {$i + 1 >= $len} {
                error "argument error in -tid"
            }
            incr i
            set tid [lindex $args $i]
            set needtid 0
        } elseif {$arg eq "-delay"} {
            set cmd [expr {$cmd | 0x10}]
            if {$i + 1 >= $len} {
                error "argument error in -delay"
            }
            incr i
            set delay [lindex $args $i]
        } elseif {$arg eq "-command"} {
            if {$i + 1 >= $len} {
                error "argument error in -command"
            }
            incr i
            set code [lindex $args $i]
        } else {
            set pin [lindex $args $i]
        }
        incr i
    }
    if {![info exists pin]} {
        error "pin missing"
    }
    set modeIds [dict get [set ${ns}::deviceConfig] $pin]
    if {![dict exists $modeIds 7]} {
        error "not supported for pin \"$pin\""
    }
    if {$needtid} {
        set ${ns}::owTid [expr {([set ${ns}::owTid] + 1) & 0xffff}]
        set tid [set ${ns}::owTid]
    }
    set msg [binary format cccc 0xf0 0x73 $cmd $pin]
    set data {}
    if {$cmd & 0x04} {
        # onewire address in Linux format
        # "aa.bbccddeeffgghh" where "aa" is chip id
        if {[scan $addr %x.%s a0 rest] != 2} {
            error "bad address \"$addr\""
        }
        if {[binary scan [binary format H* $rest] H2H2H2H2H2H2H2 \
            a1 a2 a3 a4 a5 a6 a7] != 7} {
            error "bad address \"$addr\""
        }
        scan $a1 %x a1
        scan $a2 %x a2
        scan $a3 %x a3
        scan $a4 %x a4
        scan $a5 %x a5
        scan $a6 %x a6
        scan $a7 %x a7
        lappend data $a0 $a1 $a2 $a3 $a4 $a5 $a6 $a7
    }
    if {$cmd & 0x08} {
        lappend data [expr {$rln & 0xff}] [expr {($rln >> 8) & 0xff}]
        lappend data [expr {$tid & 0xff}] [expr {($tid >> 8) & 0xff}]
    }
    if {$cmd & 0x10} {
        lappend data [expr {$delay & 0xff}] [expr {($delay >> 8) & 0xff}] \
            [expr {($delay >> 16) & 0xff}] [expr {($delay >> 24) & 0xff}]
    }
    if {$cmd & 0x20} {
        foreach k $wrdata {
            lappend data [expr {$k & 0xff}]
        }
    }
    set len [llength $data]
    set len7 $len
    set stuff 0
    while {$len7 % 7} {
        lappend data 0
        incr len7
        incr stuff
    }
    set msg8 {}
    while {[llength $data]} {
        set data [lassign $data k0 k1 k2 k3 k4 k5 k6]
        lappend msg8 [expr {$k0 & 0x7f}]
        lappend msg8 [expr {(($k0 >> 7) & 0x01) | (($k1 << 1) & 0x7e)}]
        lappend msg8 [expr {(($k1 >> 6) & 0x03) | (($k2 << 2) & 0x7c)}]
        lappend msg8 [expr {(($k2 >> 5) & 0x07) | (($k3 << 3) & 0x78)}]
        lappend msg8 [expr {(($k3 >> 4) & 0x0f) | (($k4 << 4) & 0x70)}]
        lappend msg8 [expr {(($k4 >> 3) & 0x1f) | (($k5 << 5) & 0x60)}]
        lappend msg8 [expr {(($k5 >> 2) & 0x3f) | (($k6 << 6) & 0x40)}]
        lappend msg8 [expr {($k6 >> 1) & 0x7f}]
    }
    set len [expr {$len * 8 / 7}]
    if {$stuff == 0} {
        incr len -1
    }
    foreach b [lrange $msg8 0 $len] {
        append msg [binary format c $b]
    }
    append msg [binary format c 0xf7]
    if {[write $ns $msg]} {
        # don't remember callback, write didn't happen anyway
        return
    }
    if {[info exists tid] && [info exists code]} {
        dict set ${ns}::owCallbacks ${pin},${tid} $code
    }
}

# returns code of onewire callback for a read transaction
proc tfirmata::getOwCallback {ns pin tid} {
    set cbs [set ${ns}::owCallbacks]
    if {![dict exists $cbs ${pin},${tid}]} {
        if {[dict exists $cbs any]} {
            return [dict get $cbs any]
        }
        return {}
    }
    set code [dict get $cbs ${pin},${tid}]
    dict unset ${ns}::owCallbacks ${pin},${tid}
    return $code
}

# set one wire callback for any unmatched transaction
proc tfirmata::owcallback {ns cmd} {
    dict set ${ns}::owCallbacks any $cmd
}

# -----------------------------------------------------------------------------

# configure stepper
proc tfirmata::stepconfig {ns num dly intf spr pin1 pin2 {pin3 {}} {pin4 {}}} {
    if {$num < 0 || $num > 5} {
        error "invalid stepper number, must be 0..5"
    }
    if {$dly < 1 || $dly > 2} {
        error "unsupported delay, must be 1 or 2"
    }
    if {!($intf == 1 || $intf == 2 || $intf == 4)} {
        error "unsupported interface, must be 1, 2, or 4"
    }
    if {$spr <= 0} {
        error "invalid steps-per-revolution"
    }
    set modeIds [dict get [set ${ns}::deviceConfig] $pin1]
    if {![dict exists $modeIds 8]} {
        error "not supported for pin \"$pin1\""
    }
    set modeIds [dict get [set ${ns}::deviceConfig] $pin2]
    if {![dict exists $modeIds 8]} {
        error "not supported for pin \"$pin2\""
    }
    lappend pins $pin1 $pin2
    set cfg [expr {(($dly - 1) << 3) | $intf}]
    if {$intf != 4} {
        write $ns [binary format cccccccccc 0xf0 0x72 0x00 \
            $num $cfg {*}[decompose $spr] $pin1 $pin2 0xf7]
    } else {
        set modeIds [dict get [set ${ns}::deviceConfig] $pin3]
        if {![dict exists $modeIds 8]} {
            error "not supported for pin \"$pin3\""
        }
        set modeIds [dict get [set ${ns}::deviceConfig] $pin4]
        if {![dict exists $modeIds 8]} {
            error "not supported for pin \"$pin4\""
        }
        lappend pins $pin3 $pin4
        write $ns [binary format cccccccccccc 0xf0 0x72 0x00 \
                 $num $cfg {*}[decompose $spr] $pin1 $pin2 $pin3 $pin4 0xf7]
    }
    set ${ns}::stepperPins_$num $pins
    set m [set ${ns}::wakeup]
    set m [expr {$m & ~(1 << $num)}]
    set ${ns}::wakeup $m
}

# step stepper
proc tfirmata::step {ns num steps speed {accel {}} {decel {}}} {
    if {$num < 0 || $num > 5} {
        error "invalid stepper number, must be 0..5"
    }
    if {abs($steps) > 0x1fffff} {
        error "number of steps out of range"
    }
    set dir 0
    if {$steps < 0} {
        set dir 1
        set steps [expr {-$steps}]
    }
    if {$speed < 0 || $speed > 0x3fff} {
        error "speed out of range"
    }
    set cmd [binary format cccccccccc \
                 0xf0 0x72 0x01 $num $dir {*}[decompose3 $steps] \
                 {*}[decompose $speed]]
    if {$accel ne ""} {
        if {![string is integer $accel] || $accel <= 0 || $accel > 0x3fff} {
            error "acceleration out of range"
        }
        append cmd [binary format cc {*}[decompose $accel]]
        if {$decel ne ""} {
            incr n
            if {![string is integer $decel] || $decel <= 0 || $decel > 0x3fff} {
                error "deceleration out of range"
            }
            append cmd [binary format cc {*}[decompose $decel]]
        }
    }
    append cmd [binary format c 0xf7]
    write $ns $cmd
    set m [set ${ns}::wakeup]
    set m [expr {$m & ~(1 << $num)}]
    set ${ns}::wakeup $m
}

# power down stepper pins
proc tfirmata::stepoff {ns num} {
    if {$num < 0 || $num > 5} {
        error "invalid stepper number, must be 0..5"
    }
    if {[info exists ${ns}::stepperPins_$num]} {
        set cmd ""
        foreach pin [set ${ns}::stepperPins_$num] {
            # temporary output and zero, then back to stepper
            append cmd [binary format ccc 0xf4 $pin 1]
            append cmd [binary format ccc 0xf5 $pin 0]
            append cmd [binary format ccc 0xf4 $pin 8]
        }
        if {$cmd ne ""} {
            write $ns $cmd
        }
    }
    set m [set ${ns}::wakeup]
    set m [expr {$m & ~(1 << $num)}]
    set ${ns}::wakeup $m
}

# wait for stepper done
proc tfirmata::stepwait {ns {mask 0} {ms 2000}} {
    set m [expr {[set ${ns}::wakeup] & $mask}]
    if {$m} {
        return $m
    }
    set timer [after $ms [subst -nocommands {
        set ${ns}::wakeup [set ${ns}::wakeup]
    }]]
    vwait ${ns}::wakeup
    after cancel $timer
    return [expr {[set ${ns}::wakeup] & $mask}]
}

# -----------------------------------------------------------------------------

# attach encoder
proc tfirmata::eattach {ns enc pin1 pin2} {
    if {$enc < 0 || $enc > 4} {
        error "invalid encoder number, must be 0..4"
    }
    if {![dict exists [set ${ns}::deviceConfig] $pin1]} {
        error "invalid first pin \"$pin\""
    }
    if {![dict exists [set ${ns}::deviceConfig] $pin2]} {
        error "invalid second pin \"$pin\""
    }
    set modeIds [dict get [set ${ns}::deviceConfig] $pin1]
    if {![dict exists $modeIds 9]} {
        error "not supported for first pin"
    }
    set modeIds [dict get [set ${ns}::deviceConfig] $pin2]
    if {![dict exists $modeIds 9]} {
        error "not supported for second pin"
    }
    write $ns [binary format ccccccc 0xf0 0x61 0x00 $enc $pin1 $pin2 0xf7]
    if {![info exists ${ns}::encoder$enc]} {
        set ${ns}::encoder$enc 0
    }
}

# query encoder positions
proc tfirmata::equery {ns {enc {}}} {
    if {$enc eq ""} {
        # all encoders
        write $ns [binary format cccc 0xf0 0x61 0x02 0xf7]
        return
    }
    if {$enc < 0 || $enc > 4} {
        error "invalid encoder number, must be 0..4"
    }
    write $ns [binary format ccccc 0xf0 0x61 0x01 $enc 0xf7]
}

# reset (zero) encoder
proc tfirmata::ereset {ns enc} {
    if {$enc < 0 || $enc > 4} {
        error "invalid encoder number, must be 0..4"
    }
    write $ns [binary format ccccc 0xf0 0x61 0x03 $enc 0xf7]
    set ${ns}::encoder$enc 0
}

# enable/disable encoder reports
proc tfirmata::ereport {ns flag} {
    set en 0
    if {$flag eq "on" || ([string is integer $flag] && $flag)} {
        set en 1
    }
    write $ns [binary format ccccc 0xf0 0x61 0x04 $en 0xf7]
}

# detach encoder
proc tfirmata::edetach {ns enc} {
    if {$enc < 0 || $enc > 4} {
        error "invalid encoder number, must be 0..4"
    }
    write $ns [binary format ccccc 0xf0 0x61 0x05 $enc 0xf7]
}

# return last read encoder value
proc tfirmata::eget {ns enc} {
    if {$enc < 0 || $enc > 4} {
        error "invalid encoder number, must be 0..4"
    }
    if {![info exists ${ns}::encoder$enc]} {
        return [set ${ns}::encoder$enc]
    }
    return 0
}

# -----------------------------------------------------------------------------

# configure serial
proc tfirmata::serconfig {ns num baud {rxpin {}} {txpin {}}} {
    if {$num < 0 || $num > 16} {
        error "invalid serial port, must be 0..15"
    }
    if {$baud < 0 || $baud > 0x1ffff} {
        error "invalid baud rate"
    }
    if {$rxpin ne "" && $txpin ne ""} {
        # HW serial RX wants "serial", SW serial RX wants "in"
        set need [expr {($num < 8) ? 10 : 0}]
        set modeIds [dict get [set ${ns}::deviceConfig] $rxpin]
        if {![dict exists $modeIds $need]} {
            error "not supported for pin \"$rxpin\""
        }
        # HW serial TX wants "serial", SW serial TX wants "out"
        set need [expr {($num < 8) ? 10 : 1}]
        set modeIds [dict get [set ${ns}::deviceConfig] $txpin]
        if {![dict exists $modeIds $need]} {
            error "not supported for pin \"$txpin\""
        }
        write $ns [binary format ccccccccc 0xf0 0x60 [expr {0x10 | $num}] \
            {*}[decompose3 $baud] $rxpin $txpin 0xf7]
    } else {
        write $ns [binary format ccccccc 0xf0 0x60 [expr {0x10 | $num}] \
            {*}[decompose3 $baud] 0xf7]
    }
}

# set serial read callback command
proc tfirmata::sercommand {ns cmd} {
    set ${ns}::serialCallback $cmd
}

# write serial
proc tfirmata::serwrite {ns num data} {
    if {$num < 0 || $num > 16} {
        error "invalid serial port, must be 0..15"
    }
    set len [string length $data]
    if {$len > 25} {
        error "data too large, max. 25"
    }
    set cmd [binary format ccc 0xf0 0x60 [expr {0x20 | $num}]]
    foreach b [split $data {}] {
        scan $b %c b
        append cmd \
            [binary format cc [expr {$b & 0x7f}] [expr {($b >> 7) & 0x7f}]]
    }
    append cmd [binary format c 0xf7]
    write $ns $cmd
}

# request read serial
proc tfirmata::serread {ns num {stop 0} {max 0}} {
    if {$num < 0 || $num > 16} {
        error "invalid serial port, must be 0..15"
    }
    if {$max < 0 || $max > 0x3fff} {
        error "invalid number of bytes to read (0..16383)"
    }
    set stop [expr {$stop ? 1 : 0}]
    if {$max == 0} {
        write $ns \
            [binary format ccccc 0xf0 0x60 [expr {0x30 | $num}] $stop 0xf7]
    } else {
        write $ns \
            [binary format ccccccc 0xf0 0x60 [expr {0x30 | $num}] $stop \
                 {*}[decompose $max] 0xf7]
    }
}

# close serial
proc tfirmata::serclose {ns num} {
    if {$num < 0 || $num > 16} {
        error "invalid serial port, must be 0..15"
    }
    write $ns [binary format cccc 0xf0 0x60 [expr {0x50 | $num}] 0xf7]
}

# flush serial
proc tfirmata::serflush {ns num} {
    if {$num < 0 || $num > 16} {
        error "invalid serial port, must be 0..15"
    }
    write $ns [binary format cccc 0xf0 0x60 [expr {0x60 | $num}] 0xf7]
}

# listen (SW) serial
proc tfirmata::serlisten {ns num} {
    if {$num < 0 || $num > 16} {
        error "invalid serial port, must be 0..15"
    }
    write $ns [binary format cccc 0xf0 0x60 [expr {0x70 | $num}] 0xf7]
}

# -----------------------------------------------------------------------------

# make and return task data buffer
proc tfirmata::taskdata {ns} {
    if {[set ${ns}::firmwareVersion] < 2.4} {
        error "unsupported by firmware"
    }
    return [chan {} 1 $ns]
}

# create firmata task, second handle provides task data
proc tfirmata::createtask {ns id ns2} {
    if {[set ${ns}::firmwareVersion] < 2.4} {
        error "unsupported by firmware"
    }
    set id [expr {$id & 0x7f}]
    if {[catch {set ${ns2}::txBuf} data]} {
        error "$ns2 is not a task data buffer"
    }
    set len [string length $data]
    if {$len >= 16384} {
        close $ns2
        error "task data of $ns2 too large"
    }
    if {$len < 1} {
        close $ns2
        error "no task data in $ns2"
    }
    close $ns2
    write $ns [binary format ccccccc 0xf0 0x7b 0x00 $id [expr {$len & 0x7f}] \
                   [expr {($len >> 7) & 0x7f}] 0xf7]
    while {$len > 0} {
        set max $len
        if {$max > 49} {
            set max 49
        }
        set b [string range $data 0 $max-1]
        set data [string range $data $max end]
        set len [expr {$len - $max}]
        set stuff 0
        set max7 $max
        while {$max7 % 7} {
            append b [binary format c 0x00]
            incr stuff
            incr max7
        }
        set msg8 {}
        binary scan $b c* b
        while {[llength $b]} {
            set b [lassign $b k0 k1 k2 k3 k4 k5 k6]
            lappend msg8 [expr {$k0 & 0x7f}]
            lappend msg8 [expr {(($k0 >> 7) & 0x01) | (($k1 << 1) & 0x7e)}]
            lappend msg8 [expr {(($k1 >> 6) & 0x03) | (($k2 << 2) & 0x7c)}]
            lappend msg8 [expr {(($k2 >> 5) & 0x07) | (($k3 << 3) & 0x78)}]
            lappend msg8 [expr {(($k3 >> 4) & 0x0f) | (($k4 << 4) & 0x70)}]
            lappend msg8 [expr {(($k4 >> 3) & 0x1f) | (($k5 << 5) & 0x60)}]
            lappend msg8 [expr {(($k5 >> 2) & 0x3f) | (($k6 << 6) & 0x40)}]
            lappend msg8 [expr {($k6 >> 1) & 0x7f}]
        }
        # build "add to task" message
        set msg [binary format cccc 0xf0 0x7b 0x02 $id]
        set max [expr {$max * 8 / 7}]
        if {$stuff == 0} {
            incr max -1
        }
        foreach b [lrange $msg8 0 $max] {
            append msg [binary format c $b]
        }
        append msg [binary format c 0xf7]
        write $ns $msg
    }
}

# delete firmata task
proc tfirmata::deletetask {ns id} {
    if {[set ${ns}::firmwareVersion] < 2.4} {
        error "unsupported by firmware"
    }
    set id [expr {$id & 0x7f}]
    write $ns [binary format ccccc 0xf0 0x7b 0x01 $id 0xf7]
}

# delay firmata task
proc tfirmata::delaytask {ns ms} {
    if {[set ${ns}::firmwareVersion] < 2.4} {
        error "unsupported by firmware"
    }
    set t0 [expr {$ms & 0x7f}]
    set t1 [expr {($ms >> 7) & 0x7f}]
    set t2 [expr {($ms >> 14) & 0x7f}]
    set t3 [expr {($ms >> 21) & 0x7f}]
    set t4 [expr {($ms >> 28) & 0xf}]
    write $ns [binary format ccccccccc 0xf0 0x7b 0x03 $t0 $t1 $t2 $t3 $t4 0xf7]
}

# schedule firmata task
proc tfirmata::scheduletask {ns id ms} {
    if {[set ${ns}::firmwareVersion] < 2.4} {
        error "unsupported by firmware"
    }
    set id [expr {$id & 0x7f}]
    set t0 [expr {$ms & 0x7f}]
    set t1 [expr {($ms >> 7) & 0x7f}]
    set t2 [expr {($ms >> 14) & 0x7f}]
    set t3 [expr {($ms >> 21) & 0x7f}]
    set t4 [expr {($ms >> 28) & 0xf}]
    write $ns \
        [binary format cccccccccc 0xf0 0x7b 0x04 $id $t0 $t1 $t2 $t3 $t4 0xf7]
}

# reset firmata tasks
proc tfirmata::resettasks {ns} {
    if {[set ${ns}::firmwareVersion] < 2.4} {
        error "unsupported by firmware"
    }
    write $ns [binary format cccc 0xf0 0x7b 0x07 0xf7]
}

# -----------------------------------------------------------------------------

# returns number of device ports
proc tfirmata::getNumPorts {ns} {
    set config [set ${ns}::deviceConfig]
    set numPorts [expr {[dict size $config] / 8}]
    if {[dict size $config] % 8} {
        incr numPorts
    }
    return $numPorts
}

# separates values provided into two 7-bit integers
proc tfirmata::decompose {args} {
    set l [list]
    foreach arg $args {
        lappend l [expr {$arg & 0x7f}] [expr {($arg >> 7) & 0x7f}]
    }
    return $l
}

# separates values provided into three 7-bit integers
proc tfirmata::decompose3 {args} {
    set l [list]
    foreach arg $args {
        lappend l [expr {$arg & 0x7f}] [expr {($arg >> 7) & 0x7f}] \
            [expr {($arg >> 14) & 0x7f}]
    }
    return $l
}

# computes CRC8 for onewire
proc tfirmata::crc8 {list} {
    set crc 0
    foreach n $list {
        for {set i 0} {$i < 8} {incr i} {
            set crc [expr {($crc >> 1) ^ (($n ^ $crc) & 1 ? 0x8c : 0)}]
            set n [expr {$n >> 1}]
        }
    }
    return $crc
}
