#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <errno.h>
#include <time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <modbus/modbus.h>
#ifdef _WIN32
#include <windows.h>
#else
#include <unistd.h>
#include <signal.h>
#include <sys/times.h>
#include <arpa/inet.h>
#include <sys/socket.h>
#define SOCKET int
#define closesocket close
#endif

#ifndef MODBUS_FC_WRITE_SINGLE_COIL
#define MODBUS_FC_WRITE_SINGLE_COIL 0x05
#endif
#ifndef MODBUS_FC_WRITE_SINGLE_REGISTER
#define MODBUS_FC_WRITE_SINGLE_REGISTER 0x06
#endif
#ifndef MODBUS_FC_WRITE_MULTIPLE_COILS
#define MODBUS_FC_WRITE_MULTIPLE_COILS 0x0f
#endif
#ifndef MODBUS_FC_WRITE_MULTIPLE_REGISTERS
#define MODBUS_FC_WRITE_MULTIPLE_REGISTERS 0x10
#endif
#ifndef MODBUS_FC_MASK_WRITE_REGISTER
#define MODBUS_FC_MASK_WRITE_REGISTER 0x16
#endif
#ifndef MODBUS_FC_WRITE_AND_READ_REGISTERS
#define MODBUS_FC_WRITE_AND_READ_REGISTERS 0x17
#endif

/* max. MODBUS address to support */
#define MAXADDR 9999

/* bit or reg range description */
struct range {
    int first;
    int last;		/* exclusive! */
    int flags;
    int address;
    int count;
};

/* bit field for mapping direction */
#define COIL2REG 1
#define REG2COIL 2

/*
 *	Aliasing is always 1:1 between
 *	- discrete inputs and coils
 *	- input registers and holding registers
 *	i.e. the decimal addresses
 *	- 00001..09999 are identical to 10001..19999
 *	- 30001..39999 are identical to 40001..49999
 *
 *	Example aliasing between bits and registers:
 *
 *	coil/input Bits				input/holding Regs.
 *	(zero based)				(zero based)
 *
 *	0..31		(0x0000..0x001f)	0..1		(0x000..0x001)
 *	1024..1055	(0x0800..0x081f)	128..129	(0x080..0x081)
 *	2048..2079	(0x1000..0x101f)	256..257	(0x100..0x101)
 *	3072..3103	(0x1800..0x181f)	384..385	(0x180..0x181)
 *
 *	Command line options for the above aliasing:
 *
 *	    -alias 0x0/2 -alias 0x80/2
 *	    -alias 0x100/2 -alias 0x180/2
 *	or
 *	    -alias 0x0,0x01 -alias 0x80,0x801
 *	    -alias 0x100,0x101 -alias 0x180,0x181
 *
 *	or decimal which is 1-based
 *
 *	    -alias 1/2 -alias 129/2
 *	    -alias 257/2 -alias 385/2
 *	or
 *	    -alias 1,2 -alias 129,130
 *	    -alias 257,258 -alias 385,386
 *
 *	i.e. the -alias option wants register start address plus
 *	length or register end address.
 */

static int port = 15002;
static int nranges = 0;
static struct range *regranges = NULL;
static const char *nvfile = NULL;
static char nvbuf[MAXADDR * 2 * sizeof(uint16_t)];
static int mbdebug = 0;
static int done = 0;

static void
parse_cmdline(int argc, char **argv)
{
    int i;

    for (i = 0; i < argc; i++) {
	if (strcmp("-port", argv[i]) == 0) {
	    if (++i >= argc) {
missing:
		fprintf(stderr, "missing argument to \"%s\"\n", argv[i - 1]);
		exit(1);
	    }
	    if (sscanf(argv[i], "%d", &port) != 1) {
badarg:
		fprintf(stderr, "invalid argument \"%s\" for \"%s\"\n",
			argv[i], argv[i - 1]);
		exit(1);
	    }
	    continue;
	}
	if (strcmp("-alias", argv[i]) == 0) {
	    int first, last;

	    if (++i >= argc) {
		goto missing;
	    }
	    if (sscanf(argv[i], "0x%x/%d", &first, &last) == 2) {
		if (first < 0 || first + last > MAXADDR ||
		    first + last < 0) {
badalias:
		    fprintf(stderr, "invalid alias \"%s\"\n", argv[i]);
		    exit(1);
		}
		if (regranges != NULL) {
		    regranges[nranges].first = first;
		    regranges[nranges].last = first + last;
		}
		nranges++;
		continue;
	    }
	    if (sscanf(argv[i], "%d/%d", &first, &last) == 2) {
		--first;	/* input is 1-based */
		if (first < 0 || first + last > MAXADDR ||
		    first + last < 0) {
		    goto badalias;
		}
		if (regranges != NULL) {
		    regranges[nranges].first = first;
		    regranges[nranges].last = first + last;
		}
		nranges++;
		continue;
	    }
	    if (sscanf(argv[i], "0x%x,0x%x", &first, &last) == 2) {
		++last;		/* last is inclusive */
		if (first < 0 || last > MAXADDR ||
		    first + last < 0 || first + last > MAXADDR) {
		    goto badalias;
		}
		if (regranges != NULL) {
		    regranges[nranges].first = first;
		    regranges[nranges].last = last;
		}
		nranges++;
		continue;
	    }
	    if (sscanf(argv[i], "%d,%d", &first, &last) == 2) {
		--first;	/* input is 1-based */
		if (first < 0 || last > MAXADDR ||
		    first + last < 0 || first + last > MAXADDR) {
		    goto badalias;
		}
		if (regranges != NULL) {
		    regranges[nranges].first = first;
		    regranges[nranges].last = last;
		}
		nranges++;
		continue;
	    }
	    goto badarg;
	}
	if (strcmp("-nvfile", argv[i]) == 0) {
	    if (++i >= argc) {
		goto missing;
	    }
	    nvfile = argv[i];
	    continue;
	}
	if (strcmp("-mbdebug", argv[i]) == 0) {
	    mbdebug++;
	    continue;
	}
	fprintf(stderr, "unknown option \"%s\"\n", argv[i]);
	exit(1);
    }
}

static void
handle_nv(int wr, modbus_mapping_t *mb_mapping)
{
    static FILE *f = NULL;

    if (nvfile == NULL) {
	return;
    }
    if (f == NULL && wr == 0) {
	/* first read operation opens the file */
	f = fopen(nvfile, "rb+");
	if (f == NULL) {
	    f = fopen(nvfile, "wb+");
	}
	if (f != NULL) {
	    setvbuf(f, nvbuf, _IOFBF, sizeof(nvbuf));
	}
    }
    if (f != NULL) {
	if (wr) {
	    if (fwrite(mb_mapping->tab_registers, sizeof(uint16_t),
		       MAXADDR, f) != MAXADDR ||
		fwrite(mb_mapping->tab_bits, sizeof(uint8_t),
		       MAXADDR, f) != MAXADDR) {
		fclose(f);
		f = NULL;
	    } else if (fflush(f) != 0 || fseek(f, 0, SEEK_SET) != 0) {
		fclose(f);
		f = NULL;
	    }
	} else {
	    fread(mb_mapping->tab_registers, sizeof(uint16_t), MAXADDR, f);
	    fread(mb_mapping->tab_bits, sizeof(uint8_t), MAXADDR, f);
	    if (fseek(f, 0, SEEK_SET) != 0) {
		fclose(f);
		f = NULL;
	    }
	}
    }
}

static int
sectick(void)
{
#ifdef _WIN32
    int t = GetTickCount();

    return t / 1000;
#else
    static int tps = -1;
    struct tms b;
    int t = times(&b);

    if (tps == -1) {
	tps = sysconf(_SC_CLK_TCK);
	if (tps <= 0) {
	    tps = 1;
	}
    }
    return t / tps;
#endif
}

#ifdef _WIN32
static BOOL
gotsig(DWORD type)
{
    done++;
    return TRUE;
}
#else
static void
gotsig(int signo)
{
    done++;
}
#endif

int
main(int argc, char **argv)
{
    modbus_t *ctx;
    modbus_mapping_t *mb_mapping;
    struct range *bitranges = NULL;
    int i, j, k, n, rc, fdmax, t0, t1;
    SOCKET fd, serv_fd;
    fd_set fds[2];
#ifdef _WIN32
    int index;
    SOCKET fda[64];
#else
    struct sigaction sa;
    char buf[32];
#endif
    uint8_t *orig_input_bits;
    uint16_t *orig_input_registers;
    uint8_t msg[MODBUS_TCP_MAX_ADU_LENGTH];

    parse_cmdline(argc - 1, argv + 1);
    if (nranges > 0) {
	regranges = (struct range *) malloc(sizeof(struct range) * nranges);
	if (regranges == NULL) {
nomem:
	    fputs("out of memory\n", stderr);
	    exit(1);
	}
	nranges = 0;
	/* re-parse to fill regranges */
	parse_cmdline(argc - 1, argv + 1);
    }
    if (nranges > 0) {
	bitranges = (struct range *) malloc(sizeof(struct range) * nranges);
	if (bitranges == NULL) {
	    goto nomem;
	}
	for (i = 0; i < nranges; i++) {
	    bitranges[i].first = regranges[i].first * 16;
	    bitranges[i].last = regranges[i].last * 16;
	    if (bitranges[i].first < 0 || bitranges[i].last > MAXADDR ||
		regranges[i].first < 0 || regranges[i].last > MAXADDR) {
		fputs("address map impossible\n", stderr);
		exit(1);
	    }
	}
    }
#ifdef _WIN32
    for (index = 0; index < 64; index++) {
	fda[index] = INVALID_SOCKET;
    }
    ctx = modbus_new_tcp("0.0.0.0", port);
#else
    sprintf(buf, "%d", port);
    ctx = modbus_new_tcp_pi("::0", buf);
#endif
    if (ctx == NULL) {
	fprintf(stderr, "failed to create context: %s\n",
		modbus_strerror(errno));
	exit(1);
    }
    /* all address spaces with MAXADDR items */
    mb_mapping = modbus_mapping_new(/* coil bits */ MAXADDR,
				    /* input bits */ MAXADDR,
				    /* holding regs */ MAXADDR,
				    /* input regs */ MAXADDR);
    if (mb_mapping == NULL) {
	fprintf(stderr, "failed to allocate mapping: %s\n",
		modbus_strerror(errno));
	modbus_free(ctx);
	exit(1);
    }
    /* make coil bits identical to input bits */
    orig_input_bits = mb_mapping->tab_input_bits;
    mb_mapping->tab_input_bits = mb_mapping->tab_bits;
    /* make input registers identical to holding registers */
    orig_input_registers = mb_mapping->tab_input_registers;
    mb_mapping->tab_input_registers = mb_mapping->tab_registers;
    if (mbdebug) {
	modbus_set_debug(ctx, mbdebug);
    }
#ifdef _WIN32
    serv_fd = modbus_tcp_listen(ctx, 5);
#else
    serv_fd = modbus_tcp_pi_listen(ctx, 5);
#endif
    if (serv_fd == -1) {
	fprintf(stderr, "failed to create listen socket: %s\n",
		modbus_strerror(errno));
	modbus_free(ctx);
	exit(1);
    }
    FD_ZERO(&fds[0]);
    FD_SET(serv_fd, &fds[0]);
#ifdef _WIN32
    fdmax = 0;
    fda[fdmax++] = serv_fd;
#else
    fdmax = serv_fd;
#endif

    /* load from nvfile, if any */
    handle_nv(0, mb_mapping);

    /* start condition: registers aliased to bits */
    for (i = 0; i < nranges; i++) {
	int address, count;
	uint8_t *newbits;
	uint16_t *newregs;

	newregs = mb_mapping->tab_registers;
	newbits = mb_mapping->tab_bits;
	address = regranges[i].first;
	count = regranges[i].last - address;
	for (j = address; j < address + count; j++) {
	    for (k = 0; k < 16; k++) {
		if (newregs[j] & (1 << k)) {
		    newbits[j * 16 + k] = ON;
		} else {
		    newbits[j * 16 + k] = OFF;
		}
	    }
	}
    }

    t0 = sectick();

#ifdef _WIN32
    SetConsoleCtrlHandler((PHANDLER_ROUTINE) gotsig, TRUE);
#else
    memset(&sa, 0, sizeof(sa));
    sa.sa_handler = gotsig;
    sigemptyset(&sa.sa_mask);
    sigaction(SIGTERM, &sa, NULL);
    sigaction(SIGINT, &sa, NULL);
#endif

    while (!done) {
	struct timeval tv = { 1, 0 };

	fds[1] = fds[0];
	if (select(fdmax + 1, &fds[1], NULL, NULL, &tv) == -1) {
	    if (errno == EINTR) {
		continue;
	    }
	}
	/* store to nvfile about each 30 seconds */
	t1 = sectick();
	if (t1 - t0 > 30) {
	    handle_nv(1, mb_mapping);
	    t0 = t1;
	}
	for (i = 0; i <= fdmax; i++) {
#ifdef _WIN32
	    fd = fda[i];
	    if (fd == INVALID_SOCKET) {
		continue;
	    }
#else
	    fd = i;
#endif
	    if (!FD_ISSET(fd, &fds[1])) {
		continue;
	    }
	    if (fd == serv_fd) {
#ifdef _WIN32
		int addrlen;
		SOCKET newfd;
#else
		socklen_t addrlen;
		int newfd;
#endif
		struct sockaddr addr;

		addrlen = sizeof(addr);
		memset(&addr, 0, sizeof(addr));
		newfd = accept(serv_fd, &addr, &addrlen);
		if (newfd == -1) {
		    continue;
		}
#ifdef _WIN32
		for (index = 0; index < 64; index++) {
		    if (fda[index] == INVALID_SOCKET) {
			break;
		    }
		}
		if (index >= 64) {
		    closesocket(newfd);
		    continue;
		}
		fda[index] = newfd;
		if (index > fdmax) {
		    fdmax = index;
		}
#else
		if (newfd > fdmax) {
		    fdmax = newfd;
		}
#endif
		FD_SET(newfd, &fds[0]);
		continue;
	    }
	    modbus_set_socket(ctx, fd);
	    /* read request */
	    rc = modbus_receive(ctx, msg);
	    if (rc > 0) {
		int offset, function, count, special = 0;
		uint16_t address, endaddr;
		uint8_t *oldbits, *newbits;
		uint16_t *oldregs, *newregs;

		/* check for special function/address */
		offset = modbus_get_header_length(ctx);
		function = msg[offset];
		address = (msg[offset + 1] << 8) + msg[offset + 2];
		count = 1;
		switch (function) {
		case MODBUS_FC_WRITE_SINGLE_COIL:
		    for (k = 0; k < nranges; k++) {
			if (address >= bitranges[k].first &&
			    address  < bitranges[k].last) {
			    special |= COIL2REG;
			    bitranges[k].flags = 1;
			    bitranges[k].address = address;
			    bitranges[k].count = count;
			} else {
			    bitranges[k].flags = 0;
			}
		    }
		    break;
		case MODBUS_FC_WRITE_MULTIPLE_COILS:
		    count = (msg[offset + 3] << 8) + msg[offset + 4];
		    endaddr = address + count;
		    if (endaddr <= address) {
			break;
		    }
		    for (k = 0; k < nranges; k++) {
			if (endaddr > bitranges[k].first &&
			    address < bitranges[k].last) {
			    special |= COIL2REG;
			    bitranges[k].flags = 1;
			    if (address < bitranges[k].first) {
				bitranges[k].address = bitranges[k].first;
				address = bitranges[k].first;
				bitranges[k].count = endaddr - address;
			    } else {
				bitranges[k].address = address;
				bitranges[k].count = count;
			    }
			    if (endaddr > bitranges[k].last) {
				bitranges[k].count =
				    bitranges[k].last - address;
			    }
			} else {
			    bitranges[k].flags = 0;
			}
		    }
		    break;
		case MODBUS_FC_MASK_WRITE_REGISTER:
		case MODBUS_FC_WRITE_SINGLE_REGISTER:
		    for (k = 0; k < nranges; k++) {
			if (address >= regranges[k].first &&
			    address  < regranges[k].last) {
			    special |= REG2COIL;
			    regranges[k].flags = 1;
			    regranges[k].address = address;
			    regranges[k].count = count;
			} else {
			    regranges[k].flags = 0;
			}
		    }
		    break;
		case MODBUS_FC_WRITE_AND_READ_REGISTERS:
		    address = (msg[offset + 5] << 8) + msg[offset + 6];
		    count = (msg[offset + 7] << 8) + msg[offset + 8];
		    goto common_multi_wr;
		case MODBUS_FC_WRITE_MULTIPLE_REGISTERS:
		    count = (msg[offset + 3] << 8) + msg[offset + 4];
		common_multi_wr:
		    endaddr = address + count;
		    if (endaddr <= address) {
			break;
		    }
		    for (k = 0; k < nranges; k++) {
			if (endaddr > regranges[k].first &&
			    address < regranges[k].last) {
			    special |= REG2COIL;
			    regranges[k].flags = 1;
			    if (address < regranges[k].first) {
				regranges[k].address = regranges[k].first;
				address = regranges[k].first;
				regranges[k].count = endaddr - address;
			    } else {
				regranges[k].address = address;
				regranges[k].count = count;
			    }
			    if (endaddr > regranges[k].last) {
				regranges[k].count =
				    regranges[k].last - address;
			    }
			} else {
			    regranges[k].flags = 0;
			}
		    }
		    break;
		}
		/* save original data */
		if (special & COIL2REG) {
		    oldbits = mb_mapping->tab_bits;
		    newbits = orig_input_bits;
		    /* one byte per input bit/coil */
		    for (k = 0; k < nranges; k++) {
			memcpy(newbits + bitranges[k].first,
			       oldbits + bitranges[k].first,
			       bitranges[k].last - bitranges[k].first);
		    }
		} else if (special & REG2COIL) {
		    oldregs = mb_mapping->tab_registers;
		    newregs = orig_input_registers;
		    /* 2 bytes per input/holding reg */
		    for (k = 0; k < nranges; k++) {
			memcpy(newregs + regranges[k].first,
			       oldregs + regranges[k].first,
			       2 * (regranges[k].last - regranges[k].first));
		    }
		}
		/* carry out function and send reply */
		modbus_reply(ctx, msg, rc, mb_mapping);
		/* mirror data */
		if (special & COIL2REG) {
		    oldbits = orig_input_bits;
		    newbits = mb_mapping->tab_bits;
		    newregs = mb_mapping->tab_registers;
		    for (n = 0; n < nranges; n++) {
			if (!bitranges[n].flags) {
			    continue;
			}
			for (j = bitranges[n].address;
			     j < bitranges[n].address + bitranges[n].count;
			     j++) {
			    if (newbits[j] != oldbits[j]) {
				newregs[j / 16] &= ~(1 << (j % 16));
				if (newbits[j]) {
				    newregs[j / 16] |= (1 << (j % 16));
				}
			    }
			}
		    }
		} else if (special & REG2COIL) {
		    oldregs = orig_input_registers;
		    newregs = mb_mapping->tab_registers;
		    newbits = mb_mapping->tab_bits;
		    for (n = 0; n < nranges; n++) {
			if (!regranges[n].flags) {
			    continue;
			}
			for (j = regranges[n].address;
			     j < regranges[n].address + regranges[n].count;
			     j++) {
			    if (newregs[j] != oldregs[j]) {
				for (k = 0; k < 16; k++) {
				    if (newregs[j] & (1 << k)) {
					newbits[j * 16 + k] = ON;
				    } else {
					newbits[j * 16 + k] = OFF;
				    }
				}
			    }
			}
		    }
		}
	    } else if (rc == -1) {
		closesocket(fd);
		FD_CLR(fd, &fds[0]);
#ifdef _WIN32
		for (index = 0; index < 64; index++) {
		    if (fd == fda[index]) {
			break;
		    }
		}
		if (index < 64) {
		    fda[index] = INVALID_SOCKET;
		}
		for (index = 64 - 1; index >= 0; index--) {
		    if (fda[index] != INVALID_SOCKET) {
			fdmax = index;
			break;
		    }
		}
#else
		if (fd == fdmax) {
		    fdmax--;
		}
#endif
	    }
	}
    }

    /* store final state to nvfile, if any */
    handle_nv(1, mb_mapping);

    /* tear down */
    modbus_set_socket(ctx, serv_fd);
    modbus_close(ctx);
    modbus_free(ctx);
    /* restore originals */
    mb_mapping->tab_input_bits = orig_input_bits;
    mb_mapping->tab_input_registers = orig_input_registers;
    modbus_mapping_free(mb_mapping);
    return 0;
}

/*
 * Local Variables:
 * mode: c
 * c-basic-offset: 4
 * fill-column: 78
 * tab-width: 8
 * End:
 */
