/*
 * mbbitspy.c --
 *
 *	Modbus (coils) bit spy.
 */

#include <unistd.h>
#include <time.h>
#include <fcntl.h>
#include <errno.h>
#include <signal.h>
#include <sys/time.h>
#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#undef MOUSE_MOVED
#include <curses.h>
#include <modbus/modbus.h>

static int istcp = 0;
static int hex = 0;
static modbus_t *mb = NULL;
static int maxb = 0;

#define array_size(a) (sizeof(a) / sizeof(a[0]))

static int
mbspy(void)
{
    int top = 0, i, k, nw, edit = 0, epos, count, lcount = 0, err;
    unsigned char mb_data[9999];
    char mb_flag[9999], mb_err[9999];

    if (maxb > array_size(mb_data)) {
	maxb = array_size(mb_data);
    } else if (maxb <= 0) {
	maxb = array_size(mb_data);
    }
    memset(mb_data, 0, sizeof(mb_data));
    memset(mb_flag, 0, sizeof(mb_flag));
    memset(mb_err, 3, sizeof(mb_err));
    if (initscr() == (WINDOW *) ERR) {
	return 0;
    }
    if (COLS < 57 || LINES < 5) {
	endwin();
	return 0;
    }
    nw = (LINES - 1) * 32;
    if (nw > maxb) {
	nw = maxb;
    }
#ifdef NCURSES_VERSION
    ESCDELAY = 300;
#endif
    raw();
    noecho();
    idlok(stdscr, TRUE);
    scrollok(stdscr, FALSE);
    keypad(stdscr, TRUE);
    nodelay(stdscr, TRUE);
    meta(stdscr, TRUE);
    nonl();
    if (has_colors()) {
	start_color();
	init_pair(1, COLOR_WHITE, COLOR_BLUE);
	init_pair(2, COLOR_MAGENTA, COLOR_BLUE);
	init_pair(3, COLOR_BLACK, COLOR_CYAN);
	init_pair(4, COLOR_BLUE, COLOR_WHITE);
	wbkgdset(stdscr, COLOR_PAIR(1));
    }
    for (;;) {
	int ch;
#ifndef _WIN32
	fd_set rs;
	struct timeval tv;
#endif
	static const char inde[] = " *";
	static const char indp[] = "\\-/|";

	lcount++;
	k = maxb;
	i = 0;
	while (!edit && k > 0) {
	    int n = 100;

	    if (i + n > maxb) {
		n = maxb - i;
	    }
	    if (memcmp(mb_flag + i, mb_err, n) == 0) {
		goto skipBad;
	    }
	    memset(mb_flag + i, 0, n);
	    count = modbus_read_bits(mb, i, n, mb_data + i);
	    if (count > 0) {
		memset(mb_flag + i, 1, count);
	    } else {
		memset(mb_flag + i, 3, n);
		err = errno;
	    }
skipBad:
	    i += n;
	    k -= n;
#ifdef _WIN32
	    ch = getch();
	    if (ch != ERR) {
		ungetch(ch);
		break;
	    }
#else
	    FD_ZERO(&rs);
	    FD_SET(0, &rs);
	    tv.tv_sec = 0;
	    tv.tv_usec = 0;
	    if (select(1, &rs, 0, 0, &tv) == 1) {
		break;
	    }
#endif
	}
	if (!edit && memchr(mb_flag, 1, sizeof(mb_flag)) == NULL) {
	    if (memchr(mb_flag, 3, sizeof(mb_flag)) != NULL) {
		memset(mb_flag, 0, sizeof(mb_flag));
	    }
	    if (has_colors()) {
		wattrset(stdscr, A_NORMAL | COLOR_PAIR(3));
		for (i = 0; i < COLS; i++) {
		    waddch(stdscr, ' ');
		}
	    } else {
		wclrtoeol(stdscr);
	    }
	    wmove(stdscr, 0, 0);
	    wprintw(stdscr, "MBBITSPY");
	    wprintw(stdscr, " %c  EDIT=^E   EXIT=^D", indp[lcount % 4]);
	    wprintw(stdscr, hex ? "   DEC=^X" : "   HEX=^X");
	    wmove(stdscr, 1, 0);
	    wprintw(stdscr, "modbus_read_bits: %s\n", modbus_strerror(err));
	    wclrtobot(stdscr);
	    wrefresh(stdscr);
	    goto again;
	}
dorefr:
	if (has_colors()) {
	    wattrset(stdscr, A_NORMAL | COLOR_PAIR(1));
	}
	for (i = top; i < top + nw; i += 32) {
	    wmove(stdscr, 1 + (i - top) / 32, 0);
	    if (hex) {
		wprintw(stdscr, "%04X:    ", i);
	    } else {
		wprintw(stdscr, "%05d:   ", i + 1);
	    }
	    for (k = 0; k < 32; k++) {
		if (i + k >= maxb) {
		    waddch(stdscr, ' ');
		} else if (mb_flag[i + k]) {
		    if (mb_flag[i + k] > 1) {
			wbkgdset(stdscr, COLOR_PAIR(4));
		    }
		    waddch(stdscr, mb_data[i + k] ? '1' : '0');
		    if (mb_flag[i + k] > 1) {
			wbkgdset(stdscr, COLOR_PAIR(1));
		    }
		} else {
		    waddch(stdscr, '?');
		}
		waddch(stdscr, ' ');
	    }
	    wclrtoeol(stdscr);
	}
	wclrtobot(stdscr);
	wmove(stdscr, 0, 0);
	if (has_colors()) {
	    wattrset(stdscr, A_NORMAL | COLOR_PAIR(3));
	    for (i = 0; i < COLS; i++) {
		waddch(stdscr, ' ');
	    }
	} else {
	    wclrtoeol(stdscr);
	}
	wmove(stdscr, 0, 0);
	wprintw(stdscr, "MBBITSPY");
	if (edit) {
	    int x, y;

	    wprintw(stdscr, " %c  SAVE=^W   QUIT=^D", inde[edit > 1]);
	    wprintw(stdscr, hex ? "   DEC=^X" : "   HEX=^X");
	    wmove(stdscr, 0, 9 + 64 - 6);
	    if (hex) {
		wprintw(stdscr, " %04X", epos);
	    } else {
		wprintw(stdscr, "%05d", epos + 1);
	    }
	    x = epos % 32;
	    y = (epos - top) / 32;
	    wmove(stdscr, y + 1, 9 + x * 2);
	} else {
	    wprintw(stdscr, " %c  EDIT=^E   EXIT=^D", indp[lcount % 4]);
	    wprintw(stdscr, hex ? "   DEC=^X" : "   HEX=^X");
	    wmove(stdscr, 0, 9);
	}
	wrefresh(stdscr);
again:
	err = 0;
#ifdef _WIN32
	if (WaitForSingleObject(GetStdHandle(STD_INPUT_HANDLE), 200)
	    != WAIT_OBJECT_0) {
	    if (edit) {
		goto again;
	    }
	    continue;
	}
#else
	FD_ZERO(&rs);
	FD_SET(0, &rs);
	tv.tv_sec = 0;
	tv.tv_usec = 200000;
	if (select(1, &rs, 0, 0, &tv) < 0) {
	    if (errno == EINTR) {
		goto again;
	    }
	    break;
	}
	if (!FD_ISSET(0, &rs)) {
	    if (edit) {
		goto again;
	    }
	    continue;
	}
#endif
	ch = getch();
#ifdef _WIN32
	if (ch == ERR) {
	    continue;
	}
#endif
	switch (ch) {
	case 0x1B:
	    if (edit) {
		edit = 0;
		continue;
	    }
	case ERR:
	case 'q':
	case 'Q':
	case 'x':
	case 'X':
	    goto done;
	case 0x04:
	    if (edit) {
		edit = 0;
		continue;
	    }
	    goto done;
	case 0x18:
	    hex = !hex;
	    goto dorefr;
	case KEY_HOME:
	    if (top != 0) {
		top = 0;
		epos = top;
		goto dorefr;
	    }
	    goto again;
	case KEY_END:
	    if (top + nw < maxb) {
		top = maxb - nw;
		epos = maxb - 1;
		goto dorefr;
	    }
	    goto again;
	case KEY_DOWN:
	    if (edit) {
		if (epos + 32 < maxb) {
		    epos += 32;
		    if (epos >= top + nw) {
			top += 32;
		    }
		    goto dorefr;
		}
	    } else if (top + nw < maxb) {
		top += 32;
		goto dorefr;
	    }
	    goto again;
	case KEY_UP:
	    if (edit) {
		if (epos >= 32) {
		    epos -= 32;
		    if (epos < top) {
			top -= 32;
		    }
		    goto dorefr;
		}
	    } else if (top >= 32) {
		top -= 32;
		goto dorefr;
	    }
	    goto again;
	case KEY_NPAGE:
	    if (edit) {
		if (epos + nw + nw < maxb) {
		    epos += nw;
		    if (epos >= top + nw) {
			top += nw;
		    }
		    goto dorefr;
		}
		if (epos < maxb - nw) {
		    epos = maxb - 1;
		    top = ((maxb - nw + 31) / 32) * 32;
		    goto dorefr;
		}
	    } else if (top + nw + nw < maxb) {
		top += nw;
		goto dorefr;
	    } else if (top < maxb - nw) {
		top = ((maxb - nw + 31) / 32) * 32;
		goto dorefr;
	    }
	    goto again;
	case KEY_PPAGE:
	    if (edit) {
		if (epos - nw >= 0) {
		    epos -= nw;
		    if (epos < top) {
			top -= nw;
		    }
		    goto dorefr;
		}
		if (epos != 0) {
		    epos = 0;
		    top = 0;
		    goto dorefr;
		}
	    } else if (top - nw >= 0) {
		top -= nw;
		goto dorefr;
	    } else if (top != 0) {
		top = 0;
		goto dorefr;
	    }
	    goto again;
	case KEY_RIGHT:
	    if (edit && epos < maxb - 1) {
		epos++;
		if (epos >= top + nw) {
		    top += 32;
		}
		goto dorefr;
	    }
	    goto again;
	case KEY_LEFT:
	    if (edit && epos > 0) {
		epos--;
		if (epos < top) {
		    top -= 32;
		}
		goto dorefr;
	    }
	    goto again;
	case ' ':
	case '\r':
	case '\n':
	    goto dorefr;
	case '\f':
	    wclear(stdscr);
	    goto dorefr;
	case 0x05:
	    if (!edit) {
		edit = 1;
		epos = top;
		goto dorefr;
	    }
	    goto again;
	case 0x17:
	    if (edit > 1) {
		int err = 0;

		for (i = 0; i < maxb; i++) {
		    if (mb_flag[i] > 1) {
			if (modbus_write_bit(mb, i, mb_data[i]) != -1) {
			    mb_flag[i] = 1;
			} else {
			    err++;
			}
			if (!istcp) {
#ifdef _WIN32
			    Sleep(10);
#else
			    const struct timespec ms10 = { 0, 10000000 };

			    nanosleep(&ms10, NULL);
#endif
			}
		    }
		}
		if (!err) {
		    edit = 1;
		}
		continue;
	    }
	    goto again;
	case '0':
	case '1':
	    if (edit && mb_flag[epos / 4]) {
		int val = (ch == '0') ? OFF : ON;

		edit = 2;
		mb_data[epos] = val;
		mb_flag[epos] = 2;
		if (edit && epos < maxb - 1) {
		    epos++;
		    if (epos >= top + nw) {
			top += 32;
		    }
		}
		goto dorefr;
	    }
	    goto again;
	}
    }
done:
    endwin();
    return err ? 3 : 0;
}

/*
 * Command line arguments
 *
 *   Modbus/RTU    ttydevice baud,parity,bits,stop,slave,mode [nregs]
 *   Modbus/TCP    ipaddress port [nregs]
 */

int
main(int argc, char **argv)
{
    if ((argc > 1) &&
#ifdef _WIN32
	((argv[1][] == 'C') || (argv[1][0] == 'c'))
#else
	(argv[1][0] == '/')
#endif
       ) {
	int baud = 9600, parity = 'N', bits = 8, stop = 1, slave = 1, mode = 0;

	if (argc > 2) {
	    sscanf(argv[2], "%d,%c,%d,%d,%d,%d",
		   &baud, &parity, &bits, &stop, &slave, &mode);
	}
	if ((parity == 'O') || (parity == 'E') || (parity == 'N')) {
	    /* valid parity */
	} else if (parity == 'o') {
	    parity = 'O';
	} else if (parity == 'e') {
	    parity = 'E';
	} else {
	    parity = 'N';
	}
	mb = modbus_new_rtu(argv[1], baud, parity, bits, stop);
	if (mb != NULL) {
	    modbus_set_slave(mb, slave);
	    if (mode) {
		modbus_rtu_set_serial_mode(mb, MODBUS_RTU_RS485);
	    }
	}
    } else {
	mb = modbus_new_tcp_pi((argc > 1) ? argv[1] : "localhost",
			       (argc > 2) ? argv[2] : "15002");
	istcp = 1;
    }
    if (argc > 3) {
	maxb = strtol(argv[3], NULL, 0);
    }
    if (mb == NULL) {
	fprintf(stderr, "modbus_new_%s: %s\n", istcp ? "tcp_pi" : "rtu",
		modbus_strerror(errno));
	exit(1);
    }
    if (modbus_connect(mb) == -1) {
	fprintf(stderr, "modbus_connect: %s\n", modbus_strerror(errno));
	exit(2);
    }
    if (!istcp) {
#ifdef _WIN32
	Sleep(30000);
#else
	const struct timespec sec3 = { 3, 0 };

	nanosleep(&sec3, NULL);
#endif
    }
    return mbspy();
}

/*
 * Local Variables:
 * mode: c
 * c-basic-offset: 4
 * fill-column: 78
 * tab-width: 8
 * End:
 */
