##############################################################################
#
# Proof-of-concept Tcl interface to libmodbus (http://libmodbus.org)
# using Ffidl and TclOO. Client-side only.
#
# chw April/May 2018
#
##############################################################################

package require Ffidl
package require Ffidlrt
package require TclOO

package provide modbus 0.1

namespace eval ::modbus {
    ::ffidl::typedef modbus_t pointer

    variable lib
    variable lib_5
    switch -exact $::tcl_platform(platform) {
        unix {
	    set lib libmodbus[info sharedlibextension]
	    if {$::tcl_platform(os) eq "OpenBSD"} {
		set lib_5 ${lib}.6.0
	    } else {
		set lib_5 ${lib}.5
	    }
	    # No version number on Android
	    if {[info exists ::env(PACKAGE_CODE_PATH)] &&
		($::env(PACKAGE_CODE_PATH) ne {})} {
		unset lib_5
	    }
	}
	windows {
	    set lib modbus[info sharedlibextension]
	}
	default {
	    error "unsupported platform"
	}
    }
    # Prefer library in script install directory
    # also prefer versioned library names on unix platforms
    set lib [file join [file dirname [info script]] $lib]
    if {[info exists lib_5]} {
	set lib_5 [file join [file dirname [info script]] $lib_5]
	if {[file readable $lib_5]} {
	    set lib $lib_5
	}
    }
    # Use system defaults as fall back
    if {![file readable $lib]} {
	if {[info exists lib_5]} {
	    set lib [file tail $lib_5]
	} else {
	    set lib [file tail $lib]
	}
    }

    # Create the Ffidl callouts, Tcl procs with "_" prefix,
    # library function names with "modbus_" prefix.
    apply [list lib {
	foreach {ret name arglist} {
	    modbus_t new_tcp_pi {pointer-utf8 pointer-utf8}
	    modbus_t new_rtu {pointer-utf8 int int int int}
	    int connect {modbus_t}
	    void close {modbus_t}
	    void free {modbus_t}
	    pointer-utf8 strerror {int}
	    int get_response_timeout {modbus_t pointer-var pointer-var}
	    int set_response_timeout {modbus_t uint32 uint32}
	    int rtu_get_serial_mode {modbus_t}
	    int rtu_set_serial_mode {modbus_t int}
	    int rtu_get_rts {modbus_t}
	    int rtu_set_rts {modbus_t int}
	    int rtu_get_rts_delay {modbus_t}
	    int rtu_set_rts_delay {modbus_t int}
	    int read_bits {modbus_t int int pointer-var}
	    int read_input_bits {modbus_t int int pointer-var}
	    int read_registers {modbus_t int int pointer-var}
	    int read_input_registers {modbus_t int int pointer-var}
	    int write_bit {modbus_t int int}
	    int write_register {modbus_t int uint16}
	    int write_bits {modbus_t int int pointer-var}
	    int write_registers {modbus_t int int pointer-var}
	    int set_slave {modbus_t int}
	    void set_socket {modbus_t int}
	} {
	    ::ffidl::callout ::modbus::_$name $arglist $ret \
		[::ffidl::symbol $lib modbus_$name]
	}
	# Special callouts needed for setchan method
	if {$::tcl_platform(platform) eq "unix"} {
	    # dup(2) system call
	    ::ffidl::callout ::modbus::_dup {int} int \
		[::ffidl::symbol [::ffidl::find-lib c] dup]
	    # Tcl_GetChannel
	    ::ffidl::callout ::modbus::_GetChannel \
		{pointer pointer-utf8 pointer-var} pointer \
		[::ffidl::stubsymbol tcl stubs 151]
	    # Tcl_GetChannelHandle
	    ::ffidl::callout ::modbus::_GetChannelHandle \
		{pointer int pointer-var} int \
		[::ffidl::stubsymbol tcl stubs 153]
	}
    } [namespace current]] $lib

    proc error {{msg "unknown error"}} {
	if {[catch ::ffidl::lasterror num]} {
	    return -code error $msg
	}
	return -code error [::modbus::_strerror $num]
    }
}

oo::class create ::modbus::modbus

oo::define ::modbus::modbus {

    constructor {addr service args} {
	my variable ctx
	if {[llength $args]} {
	    # defaults
	    lassign {N 8 1 0} parity data stop slave
	    # addr    -> device
	    # service -> baud rate
	    # arg0    -> parity
	    # arg1    -> data bits
	    # arg2    -> stop bits
	    # arg3    -> slave address
	    lassign $args parity data stop slave
	    switch -glob -- $parity {
		{[eE]*} {
		    set parity 69
		}
		{[oO]*} {
		    set parity 79
		}
		default {
		    set parity 78
		}
	    }
	    if {[scan $service %d baud] < 1} {
		return -code error "expect integer baud rate"
	    }
	    if {[scan $data %d data] < 1} {
		return -code error "expect integer data bits"
	    }
	    if {[scan $stop %d stop] < 1} {
		return -code error "expect integer stop bits"
	    }
	    if {[scan $stop %d slave] < 1 || $slave < 0} {
		return -code error "expect positive integer slave address"
	    }
	    set ctx [::modbus::_new_rtu $addr $baud $parity $data $stop]
	} else {
	    set ctx [::modbus::_new_tcp_pi $addr $service]
	}
	if {$ctx == 0} {
	    return -code error "unable to get modbus context"
	}
	if {[info exists slave]} {
	    # try to set slave address, ignore errors
	    ::modbus::_set_slave $ctx $slave
	}
    }

    destructor {
	my variable ctx
	if {[info exists ctx]} {
	    ::modbus::_close $ctx
	    ::modbus::_free $ctx
	}
	return ""
    }

    method connect {} {
	my variable ctx
	set code [::modbus::_connect $ctx]
	if {$code == -1} {
	    tailcall ::modbus::error "::modbus::_connect failed"
	}
	return $code
    }

    method close {} {
	my variable ctx
	::modbus::_close $ctx
	return ""
    }

    method setchan {chan} {
	my variable ctx
	if {$::tcl_platform(platform) ne "unix"} {
	    return -code error "unsupported platform"
	}
	set rw [binary format x[ffidl::info sizeof int]]
	set c [::modbus::_GetChannel [::ffidl::info interp] $chan rw]
	if {$c == 0} {
	    return -code error "failed to get channel \"$chan\""
	}
	binary scan $rw [::ffidl::info format int] rw
	# TCL_READABLE
	if {($rw & 0x02) == 0} {
	    return -code error "channel \"$chan\" is not readable"
	}
	# TCL_WRITABLE
	if {($rw & 0x04) == 0} {
	    return -code error "channel \"$chan\" is not writable"
	}
	set fd [binary format x[ffidl::info sizeof pointer]]
	set r [::modbus::_GetChannelHandle $c 0x04 fd]
	if {$r != 0} {
	    return -code error "failed to get file descriptor for \"$chan\""
	}
	binary scan $fd [ffidl::info format pointer] fd
	set fd [::modbus::_dup $fd]
	if {$fd < 0} {
	    tailcall ::modbus::error "::modbus::_dup failed"
	}
	set code [::modbus::_set_socket $ctx $fd]
	if {$code == -1} {
	    tailcall ::modbus::error "::modbus::_set_socket failed"
	}
	return $code
    }

    method response_timeout {args} {
	my variable ctx
	if {[llength $args] > 1} {
	    return -code error "only one optional argument allowed"
	}
	set s [binary format x[ffidl::info sizeof uint32]]
	set u [binary format x[ffidl::info sizeof uint32]]
	if {[llength $args]} {
	    lassign $args ms
	    if {$ms < 0} {
		set ms 0
	    }
	    set sec [expr {int($ms / 1000)}]
	    set usec [expr {int(($ms % 1000) * 1000)}]
	    set code [::modbus::_set_response_timeout $ctx $sec $usec]
	    if {$code == -1} {
		tailcall ::modbus::error \
		    "::modbus::_set_response_timeout failed"
	    }
	    return ""
	}
	set code [::modbus::_get_response_timeout $ctx s u]
	if {$code == -1} {
	    tailcall ::modbus::error "::modbus::_get_response_timeout failed"
	}
	binary scan $s [ffidl::info format uint32] sec
	binary scan $u [ffidl::info format uint32] usec
	return [expr {$sec * 1000 + $usec / 1000}]
    }

    method serial_mode {args} {
	my variable ctx
	if {[llength $args] > 1} {
	    return -code error "only one optional argument allowed"
	}
	if {[llength $args]} {
	    set v [lindex $args 0]
	    set code [::modbus::_rtu_set_serial_mode $ctx $v]
	    if {$code == -1} {
		tailcall ::modbus::error \
		    "::modbus::_rtu_set_serial_mode failed"
	    }
	    return ""
	}
	return [::modbus::_rtu_get_serial_mode $ctx]
    }

    method serial_rts {args} {
	my variable ctx
	if {[llength $args] > 1} {
	    return -code error "only one optional argument allowed"
	}
	if {[llength $args]} {
	    set v [lindex $args 0]
	    set code [::modbus::_rtu_set_rts $ctx $v]
	    if {$code == -1} {
		tailcall ::modbus::error \
		    "::modbus::_rtu_set_rts failed"
	    }
	    return ""
	}
	return [::modbus::_rtu_get_rts $ctx]
    }

    method serial_rts_delay {args} {
	my variable ctx
	if {[llength $args] > 1} {
	    return -code error "only one optional argument allowed"
	}
	if {[llength $args]} {
	    set v [lindex $args 0]
	    set code [::modbus::_rtu_set_rts_delay $ctx $v]
	    if {$code == -1} {
		tailcall ::modbus::error \
		    "::modbus::_rtu_set_rts_delay failed"
	    }
	    return ""
	}
	return [::modbus::_rtu_get_rts_delay $ctx]
    }

    method read_bits {addr {n 1}} {
	my variable ctx
	if {$n <= 0} {
	    return -code error "at least one bit required"
	}
	set v [binary format x[expr {$n * [ffidl::info sizeof uint8]}]]
	set code [::modbus::_read_bits $ctx $addr $n v]
	if {$code == -1} {
	    tailcall ::modbus::error "::modbus::_read_bits failed"
	}
	binary scan $v [ffidl::info format uint8]${n} v
	return $v
    }

    method read_input_bits {addr {n 1}} {
	my variable ctx
	if {$n <= 0} {
	    return -code error "at least one bit required"
	}
	set v [binary format x[expr {$n * [ffidl::info sizeof uint8]}]]
	set code [::modbus::_read_input_bits $ctx $addr $n v]
	if {$code == -1} {
	    tailcall ::modbus::error "::modbus::_read_input_bits failed"
	}
	binary scan $v [ffidl::info format uint8]${n} v
	return $v
    }

    method read_registers {addr {n 1}} {
	my variable ctx
	if {$n <= 0} {
	    return -code error "at least one value required"
	}
	set v [binary format x[expr {$n * [ffidl::info sizeof uint16]}]]
	set code [::modbus::_read_registers $ctx $addr $n v]
	if {$code == -1} {
	    tailcall ::modbus::error "::modbus::_read_registers failed"
	}
	binary scan $v [ffidl::info format uint16]${n} v
	return $v
    }

    method read_input_registers {addr {n 1}} {
	my variable ctx
	if {$n <= 0} {
	    return -code error "at least one value required"
	}
	set v [binary format x[expr {$n * [ffidl::info sizeof uint16]}]]
	set code [::modbus::_read_input_registers $ctx $addr $n v]
	if {$code == -1} {
	    tailcall ::modbus::error "::modbus::_read_input_registers failed"
	}
	binary scan $v [ffidl::info format uint16]${n} v
	return $v
    }

    method write_bit {addr val} {
	my variable ctx
	set code [::modbus::_write_bit $ctx $addr $val]
	if {$code == -1} {
	    tailcall ::modbus::error "::modbus::_write_bit failed"
	}
	return ""
    }

    method write_register {addr val} {
	my variable ctx
	set code [::modbus::_write_register $ctx $addr $val]
	if {$code == -1} {
	    tailcall ::modbus::error "::modbus::_write_register failed"
	}
	return ""
    }

    method write_bits {addr args} {
	my variable ctx
	set n [llength $args]
	if {$n <= 0} {
	    return -code error "at least one bit required"
	}
	set v [binary format [ffidl::info format uint8]${n} $args]
	set code [::modbus::_write_bits $ctx $addr $n v]
	if {$code == -1} {
	    tailcall ::modbus::error "::modbus::_write_bits failed"
	}
	return ""
    }

    method write_registers {addr args} {
	my variable ctx
	set n [llength $args]
	if {$n <= 0} {
	    return -code error "at least one value required"
	}
	set v [binary format [ffidl::info format uint16]${n} $args]
	set code [::modbus::_write_registers $ctx $addr $n v]
	if {$code == -1} {
	    tailcall ::modbus::error "::modbus::_write_registers failed"
	}
	return ""
    }

    method set_slave {id} {
	my variable ctx
	set code [::modbus::_set_slave $ctx $id]
	if {$code == -1} {
	    tailcall ::modbus::error "::modbus::_set_slave failed"
	}
	return ""
    }
}

namespace eval ::modbus {
    proc new {args} {
	uplevel 1 ::modbus::modbus create $args
    }
}
