# fsm.tcl -- Tcl FSM Infrastructure
#
# Copyright (c) 2025 Christian Werner <chw at ch minus werner dot de>
#
# See the file "license.terms" for information on usage and redistribution of
# this file, and for a DISCLAIMER OF ALL WARRANTIES.
#
#############################################################################

# A module to implement finite state machines (FSMs) using a coroutine
# as the basic building block. Inspired by the boost::msm package.
#
# Format of transition list. Each state machine has at least one of
# these flat lists of groups of 9 items:
#
#  level      - level of entry, 0 based
#  name       - name of sub state, empty for top level
#  start      - start state for the level
#  from       - start ("from") state
#  event      - event for transition
#  to         - target ("to") state
#  tolevel    - level to which "to" state belongs
#  actions    - list of actions to be called for transition
#  guard      - guard to be called or empty
#
# Format of annotation list. For each transition list there must
# be an annotation list which can be empty. Flat list of these 2 items:
#
#  state      - state name
#  data       - list of annotations
#
# The annotation data is interpreted as list of one or more elements,
# i.e.
#
#   "interrupt"     - state sets interrupt on entry and clears it
#                     on exit
#   "flag" name     - state sets the flag "name" on entry to true
#                     and on exit to false
#   "entry" name... - on state's entry the entry procs "name" are
#                     evaluated
#   "exit" name...  - on state's exit the exit procs "name" are
#                     evaluated
#
# Together, these lists enable a FSM to be driven by a coroutine,
# where the actions and guards are run in that coroutine's context.
# An event is presented to the coroutine by invoking it either with
# the event as a single word or with a so called cargo dict, which
# is maintained for ancillary data to be carried along both the
# actions, entries, exits, and guards and the caller of the coroutine.
# On input to the FSM, i.e. the invocation of the FSM, the contents
# of the cargo dict are merged into the FSM's internal cargo dict,
# which later is returned as result of the FSM invocation.
#
# The cargo dict receives this information as keys from the FSM:
#
#  coroutine  - fully qualified name of the FSM, i.e. coroutine
#  interrupt  - interrupt state, 0 or 1
#  laststates - like states but previous states
#  states     - list of list of current states of the FSM
#  terminated - termination state, 0 or 1
#
# On invocation of the coroutine, the cargo dict should contain
#
#  event     - temporary, event on input
#
# To turn on tracing of the FSM, the cargo dict can contain
#
#  trace     - command prefix for dealing with single trace messages
#
# To specifically deal with unmatched events, the cargo dict can contain
#
#  nomatch   - command prefix for dealing with an unmatched event
#
# To introspect aspects of the coroutine (the FSM) certain
# predefined "command events", which all start with a dash are
#
#  -actions     - return list of all actions
#  -annotations - return list of annotation lists
#  -cargo       - return current cargo dict
#  -cleardefer  - clear all deferred events
#  -current     - return list of list of current states
#  -defer       - return list of deferred events
#  -entries     - return list of entry actions
#  -events      - return list of all event names
#  -exits       - return list of exit actions
#  -flags       - return flags and their values
#  -guards      - return list of all guards
#  -kill        - kill the FSM
#  -last        - return list of list of last states
#  -puml        - return recorded PlantUML source, if any
#  -reset       - reset the FSM to its initial state
#  -states      - return list of all state names
#  -transitions - return list of transition lists
#  -unset       - unsets cargo items given by the list
#                 in the cargo's "keys" entry
#
# Unknown "command events", i.e. event names starting with
# a dash, are ignored.
#
# Guards are simple Tcl procs, which receive a single
# argument "event", the current event processed. A guard
# name (or flag name) can be prefixed with an exclamation
# mark to invert the logic of the guard condition.
#
# Actions, entries and exits are simple Tcl procs, which
# receive three arguments "op", "event", and "state", where
# "op" is "action", "entry", or "exit", "event" is the
# event being processed, and "state" is the state entered
# or exited. For actions, state is a two element list
# of old state and new state.
#
# Flags are booleans, which become true when a state is
# entered and false when it is exited. The flag's value
# can be queried in actions, entries, exits, and guards
# by the "getflag" proc.
#
# An event can be sent from within an action/guard or outside
# of the coroutine's (FSM's) context by the "sendevent"
# proc, which requires the coroutine's name and the event
# name as arguments.
#
# When the event loop is to be involved, the "postevent"
# proc can be used, which is similar to the "sendevent" proc,
# but uses "after idle".
#
# Deferred events are generated when the event name in
# the transition table starts with a dash and the
# transition has no action. Additionally, a deferred
# event can be programmatically sent by invoking the
# "defer" proc. For each event only a single deferred
# instance is maintained.
#
# To make the cargo dict (which resides in the top level
# stack frame of the coroutine) visible in the scope of
# an action/entry/exit/guard, the "cargodict" proc can be
# used, which is a specialized variation of upvar/uplevel.
#
# To make a "global" variable residing in the FSM (coroutine)
# the proc "fsmvar" can be used. It resembles the "variable"
# command but works on the coroutine's top level stack frame.
# Its life time is bound to the FSM's.
#
# Limited support exists to read the transition and annotation
# list(s) from *.puml (PlantUML) text files. See the various
# procs "parsepuml", "initpuml", and "initpumlfile" for more
# details.
#
# The PlantUML syntax is borrowed from boost::msm, i.e. a
# transitions is written like
#
#   <from-state> "->" <to-state> ":" <event> "/" <action> ... "[" <guard> "]"
#
# All items enclosed in less-than and greater-than symbols
# must match the "regexp \w" pattern. Actions and guards can be
# left out. Guards are enclosed in literal angle brackets and
# share the same namespace as flags, where flag names take
# precedence over guard names. Guards can be inverted by prefixing
# with "!". Special event names are "*" for any event, "[*]" for
# entry and exit, and "[H]" for shallow history of the enclosing
# state machine. Currently, no boolean expression syntax is
# supported for guards.
#
# For state annotations, the syntax is
#
#   <state> ":" <word> ...
#
# with these variations
#
#   <state> ":" "interrupt"
#   <state> ":" "flag" <flag-name>
#   <state> ":" "entry" <action> ... "[" <guard> "]"
#   <state> ":" "exit" <action> ... "[" <guard> "]"
#
# Parsing the *.puml text files is very tolerant, thus it is
# recommended to carefully review the generated transition tables
# and annotation lists.

namespace eval fsm {

    # Procs which are imported into a FSM instance.

    namespace export cargodict cleardefer defer fsmvar getflag \
	postevent self sendevent

    # Handle trace message from running state machine.
    #
    #  msg - the message text

    proc _trace {msg} {
	upvar #1 _trace trace
	if {$trace ne {}} {
	    {*}$trace $msg
	}
    }

    # Internal proc to enqueue an event.
    #
    #  cargo - event name or dict
    #
    # Must be called with state machine's coroutine context.

    proc _queue {cargo} {
	upvar #1 _cargo oldcargo
	if {[llength $cargo] <= 1} {
	    set event $cargo
	    set cargo [dict create event $event]
	} elseif {![dict exists $cargo event]} {
	    return {}
	} else {
	    set event [dict get $cargo event]
	}
	uplevel #1 [list lappend _queue $cargo]
	_trace "Queueing \"$event\""
    }

    # Internal proc to evaluate a guard.
    #
    #  nsc     - namespace
    #  guard   - name of guard
    #  event   - name of event
    #
    # Returns true or false.

    proc _fsmguard {nsc guard event} {
	upvar #1 _flag flag
	if {$guard eq {}} {
	    return 1
	}
	set neg !!
	if {[string equal -length 1 ! $guard]} {
	    set guard [string range $guard 1 end]
	    set neg !
	}
	if {[info exists flag($guard)]} {
	    # guard is a flag
	    return [expr ${neg}[set flag($guard)]]
	}
	set ret [uplevel #1 [list $guard $event]]
	_trace "Guard \"$guard\" = $ret"
	return [expr ${neg}$ret]
    }

    # Internal proc to call entry or exit actions.
    #
    #  nsc     - namespace
    #  op      - "entry" or "exit"
    #  states  - list of states to consider
    #  others  - list of states to exclude
    #  event   - name of event

    proc _fsmee {nsc op states others event} {
	set gop g$op
	upvar #1 _$op e			;# entry or exit
	upvar #1 _$gop g		;# gentry or gexit
	set dostates {}
	foreach state $states {
	    if {$state in $others} {
		continue
	    }
	    lappend dostates $state
	}
	if {$op eq "exit"} {
	    # execute exits in reverse order
	    set dostates [lreverse $dostates]
	}
	foreach state $dostates {
	    if {[info exists g($state)]} {
		if {![_fsmguard $nsc $g($state) $event]} {
		    continue
		}
	    }
	    if {[info exists e($state)]} {
		foreach action $e($state) {
		    uplevel #1 [list $action $op $event $state]
		    _trace "[string totitle $op] \"$action\""
		}
	    }
	}
	return
    }

    # Internal proc to call actions.
    #
    #  nsc       - namespace
    #  actions   - list of actions to invoke
    #  oldstate  - from state
    #  newstate  - to state
    #  event     - name of event

    proc _fsma {nsc actions oldstate newstate event} {
	foreach action $actions {
	    uplevel #1 [list $action action $event [list $oldstate $newstate]]
	    _trace "Action \"$action\""
	}
	return
    }

    # Internal proc to switch flags.
    #
    #  from    - list of from states
    #  to      - list of to states
    #  value   - flag value (0 or 1)

    proc _fsmflags {from to value} {
	upvar #1 _flag flag
	upvar #1 _mapflags mapflags
	set fromset {}
	set toset {}
	foreach state $from {
	    if {$state in $to} {
		continue
	    }
	    if {[info exists mapflags($state)]} {
		lappend fromset {*}$mapflags($state)
	    }
	}
	foreach state $to {
	    if {$state in $from} {
		continue
	    }
	    if {[info exists mapflags($state)]} {
		lappend toset {*}$mapflags($state)
	    }
	}
	foreach f $fromset {
	    if {($f ni $toset) && ($flag($f) != $value)} {
		set flag($f) $value
		_trace "Flag \"$f\" = $value"
	    }
	}
	return
    }

    # Internal proc to run one event through state machine.
    #
    #  event    - current event
    #  retdefer - variable to receive deferred event, if any
    #
    # Returns a number of matches.

    proc _fsmrun {event retdefer} {
	upvar levels levels
	upvar states states
	upvar transitions transitions
	upvar laststates laststates
	upvar cargo cargo
	upvar nsc nsc
	upvar interrupt interrupt
	upvar terminated terminated
	upvar #1 _intflag intflag
	upvar #1 _history history
	upvar $retdefer defer
	set count 0
	set matched 0
	set deferevt 0
	foreach t $transitions {
	    set oldstates [lindex $states $count]
	    set currlvl [lindex $levels $count]
	    unset -nocomplain newstate
	    array unset done
	    array set done {}
	    # Search transition table per decreasing level for
	    # matching transition given event. As last resort,
	    # search entire table regardless of level.
	    # TBD: is this correct?
	    while {$currlvl >= -1} {
		if {$currlvl >= 0} {
		    set currstate [lindex $states $count $currlvl]
		}
		set cdone 0
		foreach {lvl stn start from e to tolvl actions guard} $t {
		    if {[info exists done($cdone)]} {
			incr cdone
			continue
		    }
		    set isdefer 0
		    if {[string index $e 0] eq "-"} {
			set e [string range $e 1 end]
			incr isdefer
		    }
		    if {(($lvl == $currlvl) || ($currlvl < 0)) &&
			(($e eq $event) || ($e eq "*"))} {
			if {$currstate eq $from} {
			    incr done($cdone)
			    if {[_fsmguard $nsc $guard $event]} {
				set newstate $to
				incr matched
				break
			    }
			}
		    }
		    incr cdone
		}
		if {[info exists newstate]} {
		    break
		}
		incr currlvl -1
	    }
	    set currlvl [lindex $levels $count]
	    set currstate [lindex $states $count end]
	    if {[info exists newstate] && $interrupt} {
		if {![info exists intflag($currstate)]} {
		    unset newstate
		    incr matched -1
		}
	    }
	    if {[info exists newstate]} {
		# TBD: this block needs simplification.
		if {($newstate eq {}) && ($stn ne {})} {
		    set newfrom $stn
		    foreach {_ _ _ from _ to tl ac _} $t {
			if {$newfrom eq $from} {
			    # guard required here?
			    set newstate $to
			    set tolvl $tl
			    set actions $ac
			    break
			}
		    }
		}
		if {$tolvl > $currlvl} {
		    set substate {}
		    if {$newstate eq "(H)"} {
			set newstate $stn
			set substate $history($stn)
		    } else {
			# initial state of substate
			set history($newstate) {}
			foreach {_ stn2 start2 _ _ _ _ _ _} $t {
			    if {$stn2 eq $newstate} {
				set substate $start2
				set history($newstate) $substate
				break
			    }
			}
		    }
		    set nst [lindex $states $count]
		    lappend nst $substate
		    lset nst $currlvl $newstate
		    lset states $count $nst
		    set newstate $substate
		    lset levels $count $tolvl
		} elseif {$tolvl < $currlvl} {
		    set nst [lindex $states $count]
		    set stn2 [lindex $nst $tolvl]
		    set history($stn2) [lindex $nst $currlvl]
		    set nst [lrange $nst 0 $tolvl]
		    lset nst $tolvl $newstate
		    lset states $count $nst
		    lset levels $count $tolvl
		} else {
		    set history($stn) $newstate
		    lset states $count $tolvl $newstate
		}
		# setup states field in cargo
		lset laststates $count $oldstates
		dict set cargo laststates $laststates
		dict set cargo states $states
		_trace "Transition [expr {
			($currstate eq {}) ? {[*]} : "\"$currstate\""
		}] -> [expr {
			($newstate eq {}) ? {[*]} : "\"$newstate\""
		}]"
		# handle interrupt flag which lives in _fsm proc
		if {$interrupt && ![info exists intflag($newstate)]} {
		    set interrupt 0
		    dict set cargo interrupt $interrupt
		    _trace "Interrupt = 0"
		} elseif {!$interrupt && [info exists intflag($newstate)]} {
		    set interrupt 1
		    dict set cargo interrupt $interrupt
		    _trace "Interrupt = 1"
		}
		# flags/entries/exits/actions:
		#  1. exits 2. flags off, 3. action, 4. flags on, 5. entries
		_fsmee $nsc exit $oldstates [lindex $states $count] $event
		_fsmflags $oldstates [lindex $states $count] 0
		if {[llength $actions]} {
		    _fsma $nsc $actions $currstate $newstate $event
		} elseif {$isdefer} {
		    incr deferevt
		}
		_fsmflags [lindex $states $count] $oldstates 1
		_fsmee $nsc entry [lindex $states $count] $oldstates $event
		# if new state is empty, the FSM is terminated
		if {$newstate eq {}} {
		    set terminated 1
		    dict set cargo terminated $terminated
		    _trace "Terminated = 1"
		    return $matched
		}
	    }
	    incr count
	}
	if {$deferevt} {
	    if {$event ni $defer} {
		lappend defer $event
		_trace "Defer \"$event\""
	    }
	}
	return $matched
    }

    # Internal proc to implement a state machine.
    #
    #  nsc         - namespace for actions and guards
    #  transitions - transition tables
    #  annotations - annotation tables
    #
    # The coroutine stack frame has been prepared by a lambda
    # to have the cargo dict in "_cargo". This proc runs an
    # endless loop waiting for events which get in per yield or
    # an internal queue.

    proc _fsm {nsc transitions annotations} {
	# Vital variables are in the coroutine's toplevel stack frame,
	# everything else is in local variables.
	upvar #1 _cargo cargo		;# dict: cargo
	upvar #1 _queue queue		;# list: queued events
	upvar #1 _trace trace		;# scalar: trace function
	upvar #1 _defer defer		;# list: deferred events
	upvar #1 _entry entry		;# array: entries per state
	upvar #1 _gentry gentry		;# array: guards for entries
	upvar #1 _exit exit		;# array: exits per state 
	upvar #1 _gexit gexit		;# array: guards for exits
	upvar #1 _intflag intflag	;# array: states with interrupt
	upvar #1 _flag flag		;# array: flags by name
	upvar #1 _mapflags mapflags	;# array: states mapped to flags
	upvar #1 _history history	;# array: history per state
	uplevel #1 {set _queue {}}
	uplevel #1 {set _trace {}}
	uplevel #1 {set _defer {}}
	uplevel #1 {array set _entry {}}
	uplevel #1 {array set _gentry {}}
	uplevel #1 {array set _exit {}}
	uplevel #1 {array set _gexit {}}
	uplevel #1 {array set _intflag {}}
	uplevel #1 {array set _flag {}}
	uplevel #1 {array set _mapflags {}}
	uplevel #1 {array set _history {}}
	set nomatch {}
	# remember initial stack state of coroutine frame
	set vars1 [uplevel #1 {
	    unset nsc
	    unset transitions
	    unset annotations
	    info vars
	}]
	if {[dict exists $cargo puml]} {
	    # keep the source, Luke, if any
	    set puml [dict get $cargo puml]
	    dict unset cargo puml
	}
	# remember initial cargo
	set cargo1 $cargo
	# mangle annotations
	foreach a $annotations {
	    foreach {from anl} $a {
		foreach an $anl {
		    if {$an eq "interrupt"} {
			set intflag($from) 1
			continue
		    }
		    set rest [lassign $an an]
		    if {($an eq "flag") && ($rest ne {})} {
			foreach an $rest {
			    set flag($an) 0
			}
			lappend mapflags($from) {*}$rest
			continue
		    }
		    if {($an eq "entry") && ($rest ne {})} {
			set gentry($from) {}
			set entry($from) {}
			foreach a $rest {
			    if {[string match (*) $a]} {
				set gentry($from) [string range $a 1 end-1]
			    } else {
				lappend entry($from) $a
			    }
			}
			continue
		    }
		    if {($an eq "exit") && ($rest ne {})} {
			set gexit($from) {}
			set exit($from) {}
			foreach a $rest {
			    if {[string match (*) $a]} {
				set gexit($from) [string range $a 1 end-1]
			    } else {
				lappend exit($from) $a
			    }
			}
			continue
		    }
		}
	    }
	}
	# initial setup
	set levels [lrepeat [llength $transitions] 0]
	set states {}
	foreach t $transitions {
	    foreach {lvl stn start from e to tolvl actions guard} $t {
		if {$lvl == 0} {
		    lappend states $start
		    break
		}
		set history($stn) {}
		set history($start) {}
		set history($from) {}
		set history($to) {}
	    }
	}
	set laststates [lrepeat [llength $states] {}]
	dict unset cargo event
	dict set cargo states $states
	dict set cargo laststates $laststates
	dict set cargo coroutine [info coroutine]
	# call initial actions/entries, set flags
	foreach t $transitions {
	    foreach {lvl stn start from e to tolvl actions guard} $t {
		if {($lvl == 0) && ($start eq $to)} {
		    # ignore errors
		    catch {_fsma $nsc $actions {} $start $e}
		    _fsmflags $start {} 1
		    # ignore errors
		    catch {_fsmee $nsc entry $start {} $e}
		    break
		}
	    }
	}
	# main processing loop
	set terminated 0
	set interrupt 0
	dict set cargo interrupt $interrupt
	dict set cargo terminated $terminated
	while {1} {
	    # wait for next event
	    if {[llength $queue]} {
		set queue [lassign $queue newcargo]
	    } elseif {[info exists ret]} {
		# report result of last command
		set newcargo [yield $ret]
		unset ret
	    } else {
		# otherwise report current cargo
		set newcargo [yield $cargo]
	    }
	    if {[llength $newcargo] <= 1} {
		set event $newcargo
		unset newcargo
	    } elseif {![dict exists $newcargo event]} {
		unset newcargo
		continue
	    } else {
		# take out event and merge in new cargo
		set event [dict get $newcargo event]
		dict unset newcargo event
		if {[dict exists $newcargo keys]} {
		    # special for -unset
		    set keys [dict get $newcargo keys]
		    dict unset newcargo keys
		} else {
		    unset -nocomplain keys
		}
		set newcargo [dict merge $cargo $newcargo]
		dict set newcargo states $states
		dict set newcargo laststates $laststates
		dict set newcargo coroutine [info coroutine]
		dict set newcargo interrupt $interrupt
		dict set newcargo terminated $terminated
		set cargo $newcargo
		unset newcargo
		if {[dict exists $cargo trace]} {
		    set trace [dict get $cargo trace]
		}
		if {[dict exists $cargo nomatch]} {
		    set nomatch [dict get $cargo nomatch]
		}
	    }
	    if {$event eq {}} {
		# empty event, do nothing
		continue
	    }
	    if {!$terminated && ![string equal -length 1 $event "-"]} {
		# not terminated and not a command,
		# run state machine with the received event
		_trace "Event \"$event\""
		set newdefer {}
		set matched [_fsmrun $event newdefer]
		if {$matched == 0} {
		    if {$nomatch ne {}} {
			uplevel #1 [list $nomatch $event]
		    }
		    _trace "Event \"$event\" did not match"
		}
		# deal with existing deferred events
		foreach event $defer {
		    set matched [_fsmrun $event newdefer]
		    if {!$matched} {
			lappend newdefer $event
		    }
		}
		# new situation of deferred events
		set defer {}
		foreach event $newdefer {
		    defer $event
		}
		continue
	    }
	    # deal with commands which start with a "-"
	    if {$event eq "-unset"} {
		if {[info exists keys]} {
		    foreach var $keys {
			catch {dict unset cargo {*}$var}
		    }
		    unset keys
		    # restore standard items just in case
		    dict set cargo states $states
		    dict set cargo laststates $laststates
		    dict set cargo coroutine [info coroutine]
		    dict set cargo interrupt $interrupt
		    dict set cargo terminated $terminated
		}
		set ret {}
		continue
	    } elseif {$event eq "-reset"} {
		# cleanup stack
		set vars [uplevel {info vars}]
		foreach var $vars {
		    if {$var ni $vars1} {
			uplevel [list unset -nocomplain $var]
		    }
		}
		set cargo $cargo1
		set queue {}
		set defer {}
		# setup levels and states
		set trace {}
		set nomatch {}
		set levels [lrepeat [llength $transitions] 0]
		set states {}
		foreach t $transitions {
		    foreach {lvl stn start from e to tolvl actions guard} $t {
			if {$lvl == 0} {
			    lappend states $start
			    break
			}
		    }
		}
		set laststates [lrepeat [llength $states] {}]
		dict set cargo states $states
		dict set cargo laststates $laststates
		dict set cargo coroutine [info coroutine]
		set ret {}
		set terminated 0
		set interrupt 0
		dict set cargo interrupt $interrupt
		dict set cargo terminated $terminated
		# call initial actions/entries, set flags
		foreach t $transitions {
		    foreach {lvl stn start from e to tolvl actions guard} $t {
			if {($lvl == 0) && ($start eq $to)} {
			    # ignore errors
			    catch {_fsma $nsc $actions {} $start $e}
			    _fsmflags $start {} 1
			    # ignore errors
			    catch {_fsmee $nsc entry $start {} $e}
			    break
			}
		    }
		}
		continue
	    }
	    if {$event eq "-kill"} {
		# kill entire state machine
		foreach a [after info] {
		    lassign [after info $a] a _
		    if {[string match "[info coroutine] *" $a]} {
			after cancel $id
		    }
		}
		return
	    }
	    if {$event eq "-current"} {
		# return current states
		set ret $states
		continue
	    }
	    if {$event eq "-last"} {
		# return last states
		set ret $laststates
		continue
	    }
	    if {$event eq "-states"} {
		# return list of states
		unset -nocomplain aret
		array set aret {}
		foreach t $transitions {
		    foreach {lvl stn start from e to tolvl actions guard} $t {
			incr aret($stn)
			incr aret($start)
			incr aret($from)
			incr aret($to)
		    }
		}
		array unset aret (H)
		array unset aret {}
		set ret [lsort -dictionary [array names aret]]
		unset aret
		continue
	    }
	    if {$event eq "-events"} {
		# return list of events
		unset -nocomplain aret
		array set aret {}
		foreach t $transitions {
		    foreach {lvl stn start from e to tolvl actions guard} $t {
			if {[string index $e 0] eq "-"} {
			    set e [string range $e 1 end]
			}
			incr aret($e)
		    }
		}
		array unset aret {}
		set ret [lsort -dictionary [array names aret]]
		unset aret
		continue
	    }
	    if {$event eq "-actions"} {
		# return list of actions
		unset -nocomplain aret
		array set aret {}
		foreach t $transitions {
		    foreach {lvl stn start from e to tolvl actions guard} $t {
			foreach a $actions {
			    if {$nsc eq "::"} {
				set a ::$a
			    } else {
				set a ${nsc}::$a
			    }
			    incr aret($a)
			}
		    }
		}
		set ret [lsort -dictionary [array names aret]]
		unset aret
		continue
	    }
	    if {$event eq "-guards"} {
		# return list of guards
		unset -nocomplain aret
		array set aret {}
		foreach t $transitions {
		    foreach {lvl stn start from e to tolvl actions guard} $t {
			if {$guard eq {}} {
			    continue
			}
			if {[string equal -length 1 ! $guard]} {
			    set guard [string range $guard 1 end]
			}
			if {[info exists flag($guard)]} {
			    # guard is a flag
			    continue
			}
			if {$nsc eq "::"} {
			    set guard ::$guard
			} else {
			    set guard ${nsc}::$guard
			}
			incr aret($guard)
		    }
		}
		foreach {t guard} [concat \
			[array get gentry] [array get gexit]] {
		    if {$guard eq {}} {
			continue
		    }
		    if {[string equal -length 1 ! $guard]} {
			set guard [string range $guard 1 end]
		    }
		    if {[info exists flag($guard)]} {
			# guard is a flag
			continue
		    }
		    if {$nsc eq "::"} {
			set guard ::$guard
		    } else {
			set guard ${nsc}::$guard
		    }
		    incr aret($guard)
		}
		set ret [lsort -dictionary [array names aret]]
		continue
	    }
	    if {$event eq "-annotations"} {
		# return annotations
		set ret $annotations
		continue
	    }
	    if {$event eq "-transitions"} {
		# return transitions
		set ret $transitions
		continue
	    }
	    if {$event eq "-defer"} {
		# return current list of deferred events
		set ret [lsort -dictionary $defer]
		continue
	    }
	    if {$event eq "-cleardefer"} {
		# clear list of deferred events
		set defer {}
		set ret $defer
		continue
	    }
	    if {$event eq "-puml"} {
		# return puml input, if present
		set ret {}
		if {[info exists puml]} {
		    set ret $puml
		}
		continue
	    }
	    if {$event eq "-flags"} {
		# return state of flags
		set ret {}
		foreach a [lsort -dictionary [array names flag]] {
		    lappend ret $a $flag($a)
		}
		continue
	    }
	    if {$event eq "-entries"} {
		# return list of entry actions
		unset -nocomplain aret
		foreach {a actions} [array get entry] {
		    foreach a $actions {
			if {$nsc eq "::"} {
			    set a ::$a
			} else {
			    set a ${nsc}::$a
			}
			incr aret($a)
		    }
		}
		set ret [lsort -dictionary [array names aret]]
		continue
	    }
	    if {$event eq "-exits"} {
		# return list of exit actions
		unset -nocomplain aret
		foreach {a actions} [array get exit] {
		    foreach a $actions {
			if {$nsc eq "::"} {
			    set a ::$a
			} else {
			    set a ${nsc}::$a
			}
			incr aret($a)
		    }
		}
		set ret [lsort -dictionary [array names aret]]
		continue
	    }
	    if {$event eq "-cargo"} {
		# fetch current cargo
		continue
	    }
	}
    }

    # Create state machine (coroutine).
    #
    #  name        - name for coroutine, namespace part supported
    #  transitions - transition tables
    #  annotations - annotation tables
    #  cargo       - initial dict value of user defined data

    proc init {name transitions annotations {cargo {}}} {
	set ns [namespace qualifiers $name]
	if {$ns eq {}} {
	    set ns ::
	    set name ::$name
	} elseif {![string equal -length 2 "::" $ns]} {
	    set name ::${ns}::[namespace tail $name]
	}
	if {[llength $transitions] != [llength $annotations]} {
	    return -code error "setup problem"
	}
	if {([llength $cargo] % 2) || [catch {dict keys $cargo}]} {
	    return -code error "cargo is not a valid dict"
	}
	coroutine $name apply [list {nsc transitions annotations cargo} {
	    set _cargo $cargo
	    unset cargo
	    ${nsc}::_fsm [namespace current] $transitions $annotations
	} $ns] [namespace current] $transitions $annotations $cargo
    }

    # Send event to state machine.
    #
    #  name  - name of state machine
    #  event - event name or cargo dict

    proc sendevent {name event} {
	if {[string equal -length 1 $event -]} {
	    return
	}
	set ns [namespace qualifiers $name]
	if {$ns eq {}} {
	    set ns ::
	    set name ::$name
	} elseif {![string equal -length 2 "::" $ns]} {
	    set name ::${ns}::[namespace tail $name]
	}
	if {[catch {info coroutine} coro]} {
	    # outside of any coroutine
	    tailcall $name $event
	}
	if {$coro ne $name} {
	    # called from other coroutine
	    tailcall $name $event
	}
	tailcall [namespace current]::_queue $event
    }

    # Post event to state machine.
    #
    #  name  - name of state machine
    #  event - event name or cargo dict
    #  ms    - optional delay in milliseconds

    proc postevent {name event {ms {}}} {
	if {[string equal -length 1 $event -]} {
	    return
	}
	set ns [namespace qualifiers $name]
	if {$ns eq {}} {
	    set ns ::
	    set name ::$name
	} elseif {![string equal -length 2 "::" $ns]} {
	    set name ::${ns}::[namespace tail $name]
	}
	if {[string is integer -strict $ms] && ($ms >= 0)} {
	    after $ms [list $name $event]
	} else {
	    after idle [list $name $event]
	}
    }

    # Make cargo dict visible in action or guard.
    #
    #  dictname  - name to be used in caller's context
    #
    # Must be called with state machine's coroutine context.

    proc cargodict {dictname} {
	uplevel [list upvar #1 _cargo $dictname]
    }

    # Defer an event.
    #
    #  event  - event to be deferred
    #
    # Must be called with state machine's coroutine context.
    # Returns 1, if event has been deferred, 0 when the event
    # was already present in the set of deferred events.

    proc defer {event} {
	upvar #1 _defer defer
	if {$event eq {}} {
	    return 0
	}
	if {$event ni $defer} {
	    lappend defer $event
	    return 1
	}
	return 0
    }

    # Remove deferred event(s).
    #
    #  args  - optional list of events
    #
    # Must be called with state machine's coroutine context.

    proc cleardefer {args} {
	upvar #1 _defer defer
	if {[llength $args] == 0} {
	    # clear all deferred events
	    set defer {}
	    return
	}
	foreach event $args {
	    set index [lsearch -exact $defer $event]
	    if {$index >= 0} {
		set defer [lreplace $defer $index $index]
	    }
	}
	return
    }

    # Get flag given name.
    #
    #  name  - name of flag
    #
    # Must be called with state machine's coroutine context.
    # For unknown flags, 0 is always returned.

    proc getflag {name} {
	upvar #1 _flag flag
	if {[info exists flag($name)]} {
	    return $flag($name)
	}
	return 0
    }

    # Define/declare variable in FSM, similar to "variable".
    #
    #  name   - name of variable
    #  value  - optional initial value
    #
    # Must be called with state machine's coroutine context.

    proc fsmvar {name args} {
	if {[string match _* $name]} {
	    return -code error "variable name may not start with underscore"
	}
	set len [llength $args]
	if {$len > 1} {
	    return -code error "more than one value supplied"
	} elseif {$len} {
	    lassign $args args
	    uplevel #1 [list set $name $args]
	}
	uplevel [list upvar #1 $name $name]
    }

    # Syntactic sugar: return name of FSM.
    #
    # Must be called with state machine's coroutine context,
    # e.g. "postevent [self] Howdy".

    proc self {} {
	tailcall info coroutine
    }

    # Parse *.puml string and make transition and annotation lists
    #
    #  input  - input string, e.g. from a *.puml file
    #
    # Returns a list of transition lists and corresponding
    # annotation lists.

    proc parsepuml {input} {
	uplevel #0 {package require tcLex}
	set lexer [lindex [info level 0] 0]_[info cmdcount]
	lexer $lexer -lines -resultvar puml -prescript {
	    set puml {}
	    set stack {}
	    set states {}
	    set level 0
	    set osection 0
	    set isection 0
	    set olevel -1
	    set name {}
	    set line {}
	    # empty annotation, placeholder
	    lappend states [list $osection,$level,$isection {} - :]
	} -exclusiveconditions {
	    ign
	} -postscript {
	    if {[llength $states]} {
		lappend puml {*}$states
	    }
	} -- {
	    {} "\[ \t\]*'.*\n" {} {
		# comment, ignored
	    }
	    {} {\\n} {} {
		# backslash newline, ignored, counts as blank
	    }
	    {} "skinparam.*\{" {} {
		[lexer current] begin ign
	    }
	    ign ".*\n" {line} {
		# ignore block
		set line [string trimright $line]
		if {[string index $line end] eq "\}"} {
		    # closing brace of ign block
		    # TBD: nesting???
		    [lexer current] end
		}
	    }
	    {} "state\[ \t\]+(\\w+).*\{" {_ newname} {
		# start of state block
		incr level
		lappend stack [list $osection $isection $name]
		set name $newname
		if {[llength $states]} {
		    lappend puml {*}$states
		    set states {}
		}
		set line {}
	    }
	    {} "\[ \t\]*\}" {} {
		# closing brace of state block
		if {[llength $states]} {
		    lappend puml {*}$states
		    set states {}
		}
		set line {}
		incr level -1
		lassign [lindex $stack end] osection isection name
		incr isection
		set stack [lrange $stack 0 end-1]
	    }
	    {} {[\[]\*\]} {} {
		# start condition [*]
		lappend line {}
	    }
	    {} {[-]+>} {} {
		# arrow
		lappend line >
	    }
	    {} "-\[^ \t\n\]+->" {} {
		# arrow with arrow spec
		lappend line >
	    }
	    {} {[:/*]} word {
		# colon, slash, or asterisk
		lappend line $word
	    }
	    {} {[\[][!]?\w+\]} word {
		# [guard] -> (guard) or [!guard] -> (!guard)
		lappend line ([string trim $word {[]}])
	    }
	    {} {[-]?\w+} word {
		# [-]action
		lappend line $word
	    }
	    {} "\[ \t\]*(--|\\|\\|)\[ \t\]*\n" {} {
		# orthogonal region separator
		if {$olevel < 0} {
		    set olevel $level
		} elseif {$level != $olevel} {
		    error "nested orthogonal regions not supported"
		}
		incr osection
		set isection 0
		if {[llength $states]} {
		    lappend puml {*}$states
		    set states {}
		}
		# empty annotation, placeholder
		lappend states [list $osection,$level,$isection {} - :]
	    }
	    {} "\n" {} {
		# end of line
		if {[llength $line]} {
		    lappend states \
			[list $osection,$level,$isection $name {*}$line]
		    set line {}
		}
	    }
	    {} {<<(\w+)>>} {_ word} {
		error "unsupported <<$word>>"
	    }
	    {} {\[H\*\]} {word} {
		error "unsupported $word"
	    }
	}

	# collapse continuation lines first
	regsub -all -- {\\\n} $input { } input
	# run lexer on input
	if {[catch {$lexer eval $input} ret]} {
	    rename $lexer {}
	    return -code error $ret
	}
	rename $lexer {}
	if {[llength $ret] == 0} {
	    return -code error "invalid input"
	}
	# post process list from lexer, put annotations aside
	set nret {}
	set anno {}
	foreach elem $ret {
	    lassign $elem idx name start op
	    if {$op eq ">"} {
		lappend nret $elem
	    } elseif {$op eq ":"} {
		lappend anno $elem
	    }
	}
	set ret $nret
	# now find state names
	array set sms {}
	array set tbl {}
	foreach elem $ret {
	    lassign $elem idx name
	    set sms($name) {}
	    if {[llength $elem] > 3} {
		lappend tbl($idx) [lrange $elem 1 end]
	    }
	}
	# check for start conditions
	foreach {idx ent} [array get tbl] {
	    if {[llength $ent] == 0} {
		# just in case
		unset tbl($idx)
		continue
	    }
	    unset -nocomplain name
	    set count 0
	    foreach elem $ent {
		lassign $elem name start op target
		if {$start eq {} && $op eq ">"} {
		    incr count
		}
	    }
	    if {![info exists name]} {
		return -code error "bogus table \"$idx $ent\""
	    }
	    if {$count == 0} {
		return -code error "no start condition in \"$name\""
	    }
	    if {$count > 1} {
		return -code error "too many start conditions in \"$name\""
	    }
	}
	# check for events
	foreach {idx ent} [array get tbl] {
	    foreach elem $ent {
		set actions [lassign $elem name start op target evt]
		if {$evt eq "/"} {
		    set evt {}
		}
		if {($op eq ">") && ($evt eq {}) && ($start ne {})} {
		    return -code error \
			"empty event in \"$name\" at $start -> $target"
		}
	    }
	}
	# check for guards
	foreach {idx ent} [array get tbl] {
	    foreach elem $ent {
		set actions [lassign $elem name start op target _]
		if {$op eq ">"} {
		    set nguards 0
		    foreach action $actions {
			if {[string match (*) $action]} {
			    incr nguards
			}
		    }
		    if {$nguards > 1} {
			return -code error \
		    "more than one guard in \"$name\" at $start -> $target"
		    }
		}
	    }
	}
	# make list of start conditions
	set sconds {}
	foreach {idx ent} [array get tbl] {
	    foreach elem $ent {
		lassign $elem name start op target
		if {$start eq {} && $op eq ">"} {
		    lappend sconds [list $idx $name $target]
		}
	    }
	}
	# find all states, then filter annotations accordingly
	array set sn {}
	foreach {idx ent} [array get tbl] {
	    foreach elem $ent {
		lassign $elem name start _ target
		if {$name ne {}} {
		    incr sn($name)
		}
		if {$start ne {}} {
		    incr sn($start)
		}
		if {$target ne {}} {
		    incr sn($target)
		}
	    }
	}
	set snames [array names sn]
	set nanno {}
	foreach elem $anno {
	    lassign $elem addr name sname
	    if {$sname in $snames} {
		lappend nanno $elem
	    } elseif {$sname eq "-"} {
		lappend nanno $elem
	    }
	}
	set anno $nanno
	# postprocess and make result as list of transition tables
	# per independent states and list of annotations for it
	return [list [_postproc_t $ret $sconds] [_postproc_a $anno]]
    }

    # Internal helper, reformats single transition table into lists of lists.

    proc _postproc_t {trl stc} {
	array set tr {}
	array set nm {}
	set allactions {}
	set allguards {}
	# find minimum level
	set minlvl [apply {{} {
	    upvar trl trl
	    foreach elem $trl {
		lassign $elem addr
		scan $addr %d,%d _ level
		incr min($level)
	    }
	    return [lindex [lsort -integer [array names min]] 0]
	}}]
	if {$minlvl > 0} {
	    # reduce levels to start at 0
	    apply {{} {
		upvar minlvl minlvl
		upvar trl trl
		upvar stc stc
		set ntrl {}
		foreach elem $trl {
		    lassign $elem addr
		    scan $addr %d,%d,%d osection level isection
		    set level [expr {$level - $minlvl}]
		    set addr $osection,$level,$isection
		    lset elem 0 $addr
		    lappend ntrl $elem
		}
		set trl $ntrl
		set nstc {}
		foreach elem $stc {
		    lassign $elem addr
		    scan $addr %d,%d,%d osection level isection
		    set level [expr {$level - $minlvl}]
		    set addr $osection,$level,$isection
		    lset elem 0 $addr
		    lappend nstc $elem
		}
		set stc $nstc
	    }}
	}
	foreach elem $trl {
	    set rest [lassign $elem addr name start _ target _ event]
	    set actions {}
	    set guard {}
	    if {$event eq "/"} {
		set event {}
	    }
	    foreach elem $rest {
		if {[string match (*) $elem]} {
		    set guard [string range $elem 1 end-1]
		    set nguard $guard
		    if {[string equal -length 1 ! $nguard]} {
			set nguard [string range $nguard 1 end]
		    }
		    if {$nguard ni $allguards} {
			lappend allguards $nguard
		    }
		} elseif {$elem ne "/"} {
		    lappend actions $elem
		    if {$elem ni $allactions} {
			lappend allactions $elem
		    }
		}
	    }
	    set nm($addr) $name
	    # find target level
	    scan $addr %d,%d,%s _ tolvl _
	    foreach elem $trl {
		lassign $elem tgtaddr name _ _ tgt
		if {$name eq $target} {
		    scan $tgtaddr %d,%d,%s _ tolvl _
		    break
		}
	    }
	    lappend tr($addr) $start $event $target $tolvl $actions $guard
	}
	# cross check actions vs. guards
	# TBD: add entries/exits including guards to this test
	foreach action $allactions {
	    if {$action in $allguards} {
		return -code error "ambiguous action/guard \"$action\""
	    }
	}
	# start conditions
	array set st {}
	foreach elem $stc {
	    set rest [lassign $elem addr]
	    set st($addr) $rest
	}
	array set ntr {}
	unset minlvl
	foreach addr [lsort -dictionary [array names tr]] {
	    scan $addr %d,%d,%s count level rest
	    lassign $st($addr) _ lstart
	    foreach {start event target tolvl actions guard} $tr($addr) {
		lappend ntr($count) $level $nm($addr) $lstart \
		    $start $event $target $tolvl $actions $guard
		if {![info exists minlvl($count)] ||
		    ($level < $minlvl($count))} {
		    set minlvl($count) $level
		}
	    }
	}
	set ret {}
	foreach count [lsort -integer [array names ntr]] {
	    if {$minlvl($count) != 0} {
		return -code error \
		    "orthogonal regions only allowed in outmost state"
	    }
	    lappend ret $ntr($count)
	}
	return $ret
    }

    # Internal helper, reformats single annotation table into lists of lists.

    proc _postproc_a {anl} {
	array set an {}
	foreach elem $anl {
	    set rest [lassign $elem addr name state _]
	    lappend an($addr,$state) $rest
	}
	array set nan {}
	foreach addr [lsort -dictionary [array names an]] {
	    scan $addr %d,%d,%d,%s count _ _ state
	    set nan($count) {}
	}
	foreach addr [lsort -dictionary [array names an]] {
	    scan $addr %d,%d,%d,%s count _ _ state
	    if {$state ne "-"} {
		lappend nan($count) $state $an($addr)
	    }
	}
	set ret {}
	foreach count [lsort -integer [array names nan]] {
	    lappend ret $nan($count)
	}
	return $ret
    }

    # Create state machine (coroutine) from *.puml string.
    #
    #  name        - name for coroutine, namespace part supported
    #  puml        - *.puml input string
    #  cargo       - initial dict value of user defined data
    #
    # Wraps "init" and "parsepuml" for convenience.

    proc initpuml {name puml {cargo {}}} {
	if {[catch {
		lassign [parsepuml $puml] transitions annotations
	} msg]} {
	    return -code error $msg
	}
	if {[llength $transitions] == 0} {
	    # reject empty table(s)
	    return -code error "no transitions found"
	}
	# keep the source, Luke!
	dict set cargo puml $puml
	tailcall [namespace current]::init $name \
	    $transitions $annotations $cargo
    }

    # Create state machine (coroutine) from *.puml file.
    #
    #  name        - name for coroutine, namespace part supported
    #  file        - name of *.puml file
    #  cargo       - initial dict value of user defined data
    #
    # Wraps "initpuml" for convenience.

    proc initpumlfile {name file {cargo {}}} {
	try {
	    set f [open $file r]
	    # max. 100000 chars, should be enough anyway
	    set puml [read $f 100000]
	    close $f
	} on error {result opts} {
	    catch {close $f}
	    return -code error -options $opts $result
	}
	tailcall [namespace current]::initpuml $name $puml $cargo
    }

}

package provide fsm 0.1
