#!/usr/bin/tclsh
#
# Copyright 2018 Brad Lanam Walnut Creek CA
# Copyright 2019-2020 Brad Lanam Pleasant Hill CA
#
# Version history:
#   1.52  2021-1-17
#       Mac OS: Fix menu colors for 8.6.11.
#       Mac OS: Fix menu relief.
#   1.51  2020-12-15
#       Get default relief from standard menus.
#   1.50  2020-10-26
#       Fix another crash checking for -keepopen item.
#   1.49  2020-10-24
#       Fix a possible crash.
#       Fix crash checking for -keepopen item.
#   1.48  2020-9-25
#       Added a -keepopen on a per item basis.
#   1.47  2020-8-28
#       Missing mousewheel binding for scrolling areas.
#       Fixed scrollbar arrow buttons not working.
#   1.46  2020-6-9
#       allow index arithmetic
#       add 'invoked' menu index to return invoked item.
#   1.45  2020-4-30
#       fix configure of background color
#   1.44  2020-4-30
#       fix change of background color
#   1.43  2020-4-9
#       popup menus: fix bug: would immediately close after initial open.
#   1.42  2020-4-9
#       popup menus: fix bug: would immediately close after initial open.
#   1.41  2020-2-25
#       fix bugs with cascades introduced in 1.40
#         (-precommand, -yalign, -xalign)
#   1.40  2020-2-24
#       popup menus: fix bug: would immediately close after initial open.
#       styling issue: fix bug with non-unique styles.
#       clone command is now working.
#         The widget entry type cannot be cloned.
#       Added tearoff entry type.
#         It is a standard label, the text or image can be configured.
#         The -tearoff option automatically adds a tearoff item
#           (backwards compatibility).
#         -tearoffcommand is available.
#         -title is available.  Default title is empty.
#         I expect this is not fully backwards compatible.
#       tearoff command : create your own interface to tear off menus.
#   1.39  2020-1-10
#       Prevent a outofrange crash in updownhandler.
#   1.38  2019-12-25
#       Fixed a bug in the delete() method.
#       Fixed a typo bug in _unpostMenu.
#       Fixed a bug in invokeitem setting the wrong state.
#   1.37  2019-12-3
#       Add appearance/isdark call back in using a temporary toplevel.
#   1.36  2019-10-31
#       Remove appearance/isdark call.  Only works on existing windows.
#   1.35  2019-10-17
#       Fixed close check handling.
#       Fixed combobox click completely outside of menu window.
#       Added -clicktoopen option.
#   1.34  2019-10-17
#       Fixed mouse handling issues with overlapping menus.
#   1.33  2019-10-15
#       Fix bug in delete.
#   1.32  2019-9-29
#       Fix a bug in deactivateitem, when the item is not present.
#   1.31  2019-9-29
#       Fix a bug win movement check when there was no prior movement.
#   1.30  2019-9-25
#       More MacOS color fixes.
#       Fix color/styling issues.
#   1.29  2019-9-18
#       Change -relief default to always be raised.
#       Fix bug deleting items when the layout is not set up.
#       Fix per-menu configuration handling for
#         -font, -background, -foreground.
#       More MacOS color fixes.
#       Protect grab calls.
#   1.28  2019-9-16
#       Fix gap on delete.
#       Fix reconfigure on insert or delete.
#   1.27  2019-9-16
#       Fix column weight issues.
#       MacOS: toplevels are ok for 10.14+
#       MacOS: revert to old dark mode test for 10.14.
#   1.26 2019-9-10
#       Internal cleanup.
#       Fix sizing issues.
#   1.25
#       Change color defaults for Mac OS.
#   1.24
#       Mac OS dark mode colors.
#   1.23
#       Fix cascade unpost handling.
#   1.22
#       Handle change of item selection on insert/add/delete.
#   1.21 2018-12-03
#       Fix grab handling issue.
#       Fix close check handling.
#   1.20 2018-12-03
#       Resolve issues with menu display and configuration.
#   1.19 2018-12-03
#       Added -bindaccelerators option.
#       Added -stickycascade option.
#       Fixed display when moving off of active item in certain cases.
#       Fixed initial display of menu.
#   1.18 2018-12-02
#       Mac OS X: Force -mode frame, as toplevels cannot be supported.
#   1.17 2018-11-29
#       Change manual invoke processing to work when the menu is inactive.
#         This matches the original menu function.  May create some odd
#         states if run on a sub-cascade.
#       Fixed overlapping menu handling.
#   1.16 2018-11-29
#       Allow vertical separators in menubar.
#       Add new -gap option for items.
#   1.15 2018-10-25
#       Fix mac os x: enter/leave processing does not work properly.
#   1.14 2018-10-22
#       Fix various bugs with user 'invoke' method.
#   1.13 2018-10-22
#       Turn off mouse activation of menus in the wait state so that
#         overlapping menus work (changed in 1.12).
#       -xalign and -yalign can have tag[+-]offset
#       Fix menutop/menuleft display (call postprocess).
#   1.12 2018-10-21
#       Initialize styles once only.
#       Fix x and y offsets.
#       Fix borderwidth and relief on menubar/menuleft menus.
#       Fix placement of active frame in scrolling window.
#       Fix enter/leave handling.
#       Fix accelerator keys.
#       Fix F10.
#       Fix left arrow item positioning.
#       Fix checkbutton selection.
#   1.11 2018-10-1
#       Fix mistake.
#   1.10 2018-10-1
#       Rewrite styling and active state handling (pass 1).
#       Some option cleanup.
#   1.9 2018-10-1
#       Fix incorrect default for -borderwidth.
#       Fix bad copy/paste in enterLeaveHandler.
#       Fix column spans with -hidemargin.
#       Fix non-global grab.
#       Fix enter/leave handling.
#       Fix press handler/menu.enter/grab.
#       Fix column weighting with -hidemargin.
#   1.8 2018-10-1
#       Fix crash when window closed.
#   1.7 2018-10-1
#       Remove tailcall.
#       Rewrite grab handling.
#       Rewrite enter/leave handling.
#       Fix invoke of item when -keepopen is false.
#   1.6 2018-9-30
#       Set menu colors from the current theme.
#       Fix -font handling.
#       Fix menus not closing on checkbutton select.
#       Button press will activate the menu and item.
#   1.5 2018-9-30
#       Fixed -borderwidth and -relief options for items.
#       Added -mode {toplevel|frame} option.
#       Cleaned up some stacking order issues
#         (fixes the activeframe covering menu items for scrolling menus).
#   1.4 2018-9-29
#       Added -precommand for cascades.
#       Added -yalign for cascades.
#       Added -xalign for cascades.
#       Fixed bug in entrycget.
#       Fixed 'delete' operation to match 'menu'.
#       Fixed bug where the width was lost on a redisplay.
#       Fixed display when initial number of items did not exceed -maxheight.
#       Fixed various bugs with empty menus.
#   1.3 2018-9-22
#       Fixed initialization to only execute on mac os x.
#       Fixed incorrect state for menus after invoking an item.
#       Brought state table documentation up to date.
#   1.2 2018-9-21
#       Fixed -padx/-pady so they will work for the main menubar.
#       Set the -acceleratorprefix to \u2318 for Mac OS X.
#         This is converted to 'Meta' (hope that's right).
#       Changed default -acceleratorprefix to 'Alt-' (due to mac os x changes).
#       Fixed the colors on mac os x.  Dark mode and graphite themes
#         are supported.
#       Documentation
#   1.1 2018-9-20
#       Fixed active highlight for scrolled menus
#   1.0 2018-9-20
#       initial release
#
# Menu options:
#   See the 'menu' documentation for most options.
#   New and ignored options are below.
#
#   Existing options:
#     -font, -background, -foreground : If specified for the main menu,
#       configures the defaults for all menus.  If specified for a
#       sub-menu, configures the default for that menu only.
#   Required for top menu:
#     -type menubar
#           standard horizontal menubar
#     -type menuleft
#           vertical menubar
#   New options:
#     -acceleratoractiveforeground <color>
#           active foreground for accelerator text
#     -acceleratorfont <font>
#           font to use for accelerator text
#     -acceleratorforeground <color>
#           foreground for accelerator text
#     -acceleratorprefix <text>
#           The default accelerator prefix defaults to 'Alt-'.
#           Defaults to \u2318 on Mac OS X.
#           A combination of Alt, Meta, Super, Control, Shift.
#           Control may be abbreviated as 'Ctrl'.
#           If you change this, be sure to have a '-' appended
#           (as mac os x normally displays as \u2318D, not \u2318-D).
#     -activerelief <relief>
#           sets the active relief for menu items
#     -bindaccelerators <boolean>
#           default: true
#           If false, do not bind the accelerator keys.
#     -clicktoopen <boolean>
#           Default: false
#           If true, all cascade menus are automatically sticky, and must
#           be clicked to open.
#     -columnbreak <value>
#           set the maximum number of items per column
#           change to 0 to turn off.
#     -hideaccelerators <boolean>
#           hide accelerator text
#     -hidearrow <boolean>
#           hide cascade arrows
#     -keepopen <boolean>
#           the menu will not close when a selection is selected
#     -maxheight <height>
#           maximum number of entries.
#           if the number of entries is exceeded, a scrollbar
#           will be added.
#           Note that once -maxheight has been set, it
#           cannot be re-configured and set to zero.
#     -mode {toplevel|frame}
#           default is toplevel
#           -mode frame will create the menus as frames rather than as
#             toplevel windows.  These frames will be subject to clipping.
#           To use -mode frame, the menus must be children of the toplevel,
#             not children of the menubar.
#           Mac OS X does not support -mode toplevel and forces -mode frame.
#     -padx <value>
#           padding for items
#     -pady <value>
#           padding for items
#     -stickycascade <boolean>
#           Default: false
#           If true, clicking a cascade menu item switches the menu to sticky
#           mode.  The cascade menu will remain open, new cascades must be
#           clicked on to open.
#
# New Entry Types:
#   tearoff, widget
#
# New menu indexing command:
#   .mymenu index invoked
#       Returns the menu index for the invoked entry.
# menu indexes may include math:
#   .mymenu index active-1
#   .mymenu index last-2
#   .mymenu index @y150+1
#
# Entry options:
#   Label text:
#     -label <text>
#         may include an '&' sign to indicate that the
#         following character is the accelerator key.  The appropriate
#         -underline <index> option will be generated for the item.
#         e.g. '-label E&xit' generates '-underline 1'.
#   New options:
#     -activerelief
#         sets the active relief for the menu item
#     -gap true/false
#         defaults to false.
#         create a gap to the left/top of the item.
#         used to create right justified menu items in the
#         top bar and other configurations with gaps.
#   New options for cascades:
#     -precommand <command>
#         <command> is executed before the cascade is posted.
#         This allows dynamic creation of a menu.
#     -xalign [{activeleft|activeright|menuleft|menuright}][<value>]
#         sets the x-alignment for a cascade menu
#         default is menuright
#         <value> is an adjustment in pixels.
#         It can be positive or negative.
#     -yalign [{activetop|menutop}][<value>]
#         sets the y-alignment for a cascade menu
#         default is activetop
#         <value> is an adjustment in pixels.
#         It can be positive or negative.
#   Ignored:
#     -bitmap, -selectcolor
#
# Styling:
#   2018-10-21 : The styles are created when the first menu is instantiated.
#
#   Flexmenu.TLabel
#   Accelerator.Flexmenu.TLabel
#   Flexmenu.TRadiobutton
#   Flexmenu.TCheckbutton
#   Flexmenu.Horizontal.TSeparator
#   Flexmenu.Vertical.TSeparator
#

package provide flexmenu 1.52

proc ::flexmenu { nm args } {
  ::flexmenuobj new $nm {*}$args
  return $nm
}

namespace eval ::flexmenucmd {
  variable vars

  # duplicate this from colorutils in order to keep flexmenu a
  # stand-alone package.
  proc rgbToHexStr { rgblist } {
    set fmt #%04x%04x%04x
    set t [format $fmt {*}$rgblist]
    return $t
  }

  proc handler { nm o args } {
    $o {*}$args
  }

  proc init { } {
    variable vars

    set vars(darwin.appearance) #0079ff
    set vars(darwin.darkmode) false
    set vars(darwin.version) 0.0
    set vars(global.dictkey) 0
    set vars(mousewheel.bound) false

    if { $::tcl_platform(os) eq "Darwin" } {
      set vars(darwin.version) [exec sw_vers -productVersion]

      set tw .txyz
      toplevel $tw
      wm withdraw $tw
      # the window must exist to get the information from it.
      update idletasks

      if { [info exists ::sysvars::v(darwin.appearance)] } {
        set vars(darwin.appearance) \
            [rgbToHexStr $::sysvars::v(darwin.appearance)]
        set vars(darwin.darkmode) $::sysvars::v(darwin.darkmode)
      } else {
        set vars(darwin.appearance) \
            [rgbToHexStr [winfo rgb $tw systemControlAccentColor]]
        # sierra and high sierra dark mode only apply to the top menu bar.
        if { [package vcompare $::flexmenucmd::vars(darwin.version) 10.14] >= 0 } {
          set tdm [::tk::unsupported::MacWindowStyle isdark $tw]
          if { $tdm } {
            set vars(darwin.darkmode) true
          }
        }
      }

      set vars(darwin.menu.bg) \
          [rgbToHexStr [winfo rgb $tw systemWindowBackgroundColor]]
      set vars(darwin.menu.fg) \
          [rgbToHexStr [winfo rgb $tw systemTextColor]]
      set vars(darwin.menu.fg.disabled) \
          [rgbToHexStr [winfo rgb $tw systemDisabledControlTextColor]]
      destroy $tw
    }
  }

  proc initStyling { } {
    variable vars

    set rc false
    if { [info exists vars(styling.initialized)] } {
      set rc true
    }
    set vars(styling.initialized) true
    return $rc
  }

  proc topixels { pad } {
    if { [regexp {^(\d+)p$} $pad all val] } {
      set pad [expr {round([tk scaling]*$val)}]
    }
    return $pad
  }

  proc nextkey { } {
    variable vars

    set rc $vars(global.dictkey)
    incr vars(global.dictkey)
    return $rc
  }

  proc bindWheel { } {
    variable vars

    if { ! $vars(mousewheel.bound) } {
      bind all <MouseWheel> +[list ::flexmenucmd::wheelHandler %W %D]
      if { $::tcl_platform(platform) ne "windows" } {
        bind all <Button-4> +[list ::flexmenucmd::wheelHandler %W -1]
        bind all <Button-5> +[list ::flexmenucmd::wheelHandler %W 1]
      }
      set vars(mousewheel.bound) true
    }
  }

  proc wheelHandler { w d } {
    set last {}
    while { $w ne {} && $last ne $w } {
      if { [winfo class $w] eq "Flexmenu" } {
        if { [$w cget -maxheight] > 0 } {
          if { $::tcl_platform(platform) eq "windows" } {
            set d [expr {int(-$d / 120)}]
          }
          if { $::tcl_platform(os) eq "Darwin" } {
            set d [expr {int(-$d)}]
          }
          $w scroll $d
        }
        break
      }
      set last $w
      set w [winfo parent $w]
    }
  }

  ::flexmenucmd::init
}

::oo::class create ::flexmenuobj {
  constructor { nm args } {
    my variable vars

    set tm [::menu .flexmenutmp]
    set theme [::ttk::style theme use]
    if { [info commands ::ttk::theme::${theme}::setMenuColors] ne {} } {
      ::ttk::theme::${theme}::setMenuColors $tm
    } else {
      if { $theme ne "winnative" &&
          $theme ne "xpnative" &&
          $theme ne "vista" &&
          ! [regexp {^aqua(light|dark)?$} $theme] } {
        foreach {k} {-background -foreground -font} {
          set c [::ttk::style lookup . $k]
          if { $c ne {} } {
            $tm configure $k $c
          }
        }
        set afg [::ttk::style lookup TEntry -selectforeground focus]
        if { $afg ne {} } {
          $tm configure -selectcolor $afg
          $tm configure -activeforeground $afg
        }
        set abg [::ttk::style lookup TEntry -selectbackground focus]
        if { $abg ne {} } {
          $tm configure -activebackground $abg
        }
        set c [::ttk::style lookup TEntry -foreground disabled]
        if { $c ne {} } {
          $tm configure -disabledforeground $c
        }
      }
    }

    foreach {k} {-activebackground -activeforeground -background
        -foreground -font -disabledforeground -cursor -activeborderwidth
        -takefocus -relief} {
      set c [$tm cget $k]
      if { [regexp {^aqua(light|dark)?$} $theme] } {
        if { $k eq "-activebackground" } {
          set c $::flexmenucmd::vars(darwin.appearance)
        }
        if { $k eq "-activeforeground" } {
          set c $::flexmenucmd::vars(darwin.menu.fg)
        }
        if { $k eq "-background" } {
          set c $::flexmenucmd::vars(darwin.menu.bg)
        }
        if { $k eq "-foreground" } {
          set c $::flexmenucmd::vars(darwin.menu.fg)
        }
        if { $k eq "-disabledforeground" } {
          set c $::flexmenucmd::vars(darwin.menu.fg.disabled)
        }
        if { $k eq "-relief" } {
          set c raised
          if { $::flexmenucmd::vars(darwin.darkmode) } {
            set c solid
          }
        }
      }

      set vars($k) $c
    }
    destroy $tm

    set vars(-bordercolor) [::ttk::style lookup TFrame -bordercolor]
    if { $::flexmenucmd::vars(darwin.darkmode) } {
      set vars(-bordercolor) #3a3a3a
    }
    # as the background color cannot be set on mac os
    # for ttk widgets, set the activebackground to the same as the background.
    if { $::tcl_platform(os) eq "Darwin" } {
      set vars(-activebackground) $vars(-background)
    }

    set vars(-acceleratorfont) $vars(-font)
    set vars(-acceleratorforeground) $vars(-foreground)
    set vars(-acceleratoractiveforeground) $vars(-activeforeground)
    set vars(-acceleratorprefix) Alt-
    if { $::tcl_platform(os) eq "Darwin" } {
      set vars(-acceleratorprefix) \u2318
    }
    set vars(-activeborderwidth) 0
    set vars(-activerelief) flat
    set vars(-bindaccelerators) true
    # do not scale borderwidth
    set vars(-borderwidth) 1
    set vars(-columnbreak) 0
#    set vars(-font) TkMenuFont
    set vars(-hidearrows) false
    set vars(-hideaccelerators) false
    set vars(-keepopen) false
    set vars(-maxheight) 0
    set vars(-mode) toplevel
    if { $::tcl_platform(os) eq "Darwin" &&
        [package vcompare $::flexmenucmd::vars(darwin.version) 10.14] < 0 } {
      # -mode toplevel did not work for earlier mac os versions.
      # there are still issues with -mode toplevel.
      set vars(-mode) frame
    }
    set vars(-tornoff) false
    set vars(-padx) {}
    set vars(-pady) {}
    set vars(-postcommand) {}
    set vars(-stickycascade) false
    set vars(-clicktoopen) false
    set vars(-style) {}
    set vars(-tearoff) false
    set vars(-title) {}
    set vars(-type) submenu

    my varsinit

    # right arrows:
    #      \u21d2 \u21e2 \u25b8 \u25ba \u25b6* \u27a1 \u27a4* \u27a7
    set vars(-rightarrowchar) \u27a4
    # down arrows:
    #   \u2b07 \u2b63
    set vars(-downarrowchar) \u2b63 ; # not in use

    # a list of all non-managed widgets
    set vars(widgetlist) {}

    set vars(states) [dict create]
    set vars(states) {
        active              0
        cascade.post.active 1
        close               2
        inactive            3
        inuse               4
        wait                5
        }
    set vars(states.rev) [dict create]
    dict for {state v} $vars(states) {
      dict set vars(states.rev) $v $state
    }
    set vars(menu.state) [my _setState inactive]

    set vars(actions) [dict create]
    set vars(actions) {
        key.close       0
        key.invoke      1
        key.move        2
        menu.close      3
        menu.enter      4
        menu.leave      5
        menu.open       6
        menu.post       7
        menu.unpost     8
        mouse.click.out 9
        mouse.invoke    10
        mouse.move      11
        mouse.move.sticky 12
        none            13
        }
    set vars(actions.rev) [dict create]
    dict for {action v} $vars(actions) {
      dict set vars(actions.rev) $v $action
    }
    set vars(menu.action) [my _setAction none]

    my _initStyling

    set vars(name) $nm
    set vars(innerframe) {}
    set vars(packframe) {}
    set vars(activeframe) {}
    set vars(textframe) {}
    set vars(scrollbar) {}

    my configure {*}$args

    set vars(window.type) frame
    if { $vars(-type) eq "submenu" } {
      set vars(window.type) toplevel
      if { $vars(-mode) eq "frame" } {
        set vars(window.type) frame
      }
      set vars(cascades.active) true
    } else {
      set vars(defaultpadx) $vars(defaultmenubarpadx)
      set vars(defaultpady) $vars(defaultmenubarpady)
    }

    set vars(itemframe) [$vars(window.type) $vars(name) -class Flexmenu]
    $vars(itemframe) configure \
        -background $vars(-background) \
        -borderwidth 1 \
        -highlightthickness 0 \
        -relief $vars(-relief)
    set vars(packframe) $vars(itemframe)
    if { $vars(-type) eq "submenu" && $vars(-mode) eq "toplevel" } {
      wm withdraw $vars(itemframe)
      set vars(withdrawn) true
    }
    if { $vars(-type) eq "submenu" &&
        $vars(-mode) eq "toplevel" &&
        ! $vars(-tornoff) } {
      wm overrideredirect $vars(itemframe) yes
    }

    # in order to handle -tearoff and -tearoffcommand
    my configure {*}$args

    if { $vars(-tornoff) } {
      wm title $vars(name) $vars(-title)
      wm protocol $vars(name) WM_DELETE_WINDOW [list [self] tearoffmenuexit]
    }

    if { $vars(-type) ne "submenu" } {
      bind $vars(itemframe) <Configure> [list [self] movecheck %W %x %y]
    }

    set vars(itemframe.cmd) ${vars(name)}_flexmenu
    rename $vars(name) $vars(itemframe.cmd)
    interp alias {} $vars(name) {} ::flexmenucmd::handler $vars(name) [self]

    grid columnconfigure $vars(itemframe) 999 -weight 0 -minsize 2p

    bind $vars(itemframe) <Map> [list [self] setconfigure map %W -1]
    incr vars(config.count.base) 1
    incr vars(config.count) 1

    if { $vars(-type) ne "submenu" } {
      bind all <F10> +[list [self] openmenu]
    }
    if { $vars(-type) eq "menubar" } {
      set vars(-columnbreak) 1
    }

    if { $vars(-maxheight) != 0 } {
      ::flexmenucmd::bindWheel

      # for scrollable menus,
      # itemframe
      #   textframe scrollbar
      #     innerframe as a window in textframe (packframe)
      catch { font create flexmenusmall }
      font configure flexmenusmall -size 2
      set tf [text $vars(itemframe).tcont \
          -font flexmenusmall \
          -wrap none \
          -background $vars(-background) \
          -relief flat \
          -highlightthickness 0 \
          -yscrollcommand [list $vars(itemframe).scrollbar set] \
          -pady 0 \
          -padx 0 \
          -width 10 \
          -height 10 \
          -cursor {} \
          ]
      set if [frame $vars(itemframe).innerframe \
          -background $vars(-background) \
          -relief flat]
      # the normal yview scroll x units is not working properly
      # with the layout
      set sb [::ttk::scrollbar $vars(itemframe).scrollbar \
          -orient vertical \
          -takefocus 0 \
          -command [list [self] yview $tf] \
          ]
      set vars(textframe) $tf
      set vars(scrollbar) $sb
      set vars(innerframe) $if
      grid columnconfigure $vars(itemframe) 0 -weight 0 -minsize 2p
      grid columnconfigure $vars(itemframe) 1 -weight 0
      grid rowconfigure $vars(itemframe) 0 -weight 1
      $tf configure -state normal
      $tf window create 0.0 -window $if -padx 0 -pady 0
      $tf configure -state disabled

      set vars(packframe) $vars(innerframe)
      grid $vars(textframe) -in $vars(itemframe) -row 0 -column 0 -sticky news
      grid $vars(scrollbar) -in $vars(itemframe) -row 0 -column 1 -sticky ns
    }

    set vars(activeframe) [frame $vars(packframe).activeframe \
        -background $vars(-activebackground) \
        -relief $vars(-activerelief) \
        -borderwidth $vars(-activeborderwidth)]
    bind $vars(activeframe) <ButtonRelease-1> [list [self] invokeactive activeframe]
    bind $vars(activeframe) <ButtonPress-1> [list [self] pressHandler %W]
    bind $vars(activeframe) <Key-space> [list [self] invokeactive activeframe]

    return $vars(name)
  }

  method yview { w args } {
    my variable vars

    lassign $args cmd val type
    if { $type eq "units" } {
      lassign [$w yview] toploc bottomloc
      # just make the assumption that all items are the same height
      set dictkey [lindex $vars(item.map) 0]
      set h [dict get $vars(items) $dictkey height]
      set perc [expr {double($h)/double([winfo height $vars(innerframe)])}]
      if { $val < 0 } {
        set perc [expr {-$perc}]
      }
      set loc [expr {$toploc+$perc}]
      if { $loc < 0.0 } {
        set loc 0.0
      }
      if { $loc > 100.0 } {
        set loc 100.0
      }
      $w yview moveto $loc
    } else {
      $w yview {*}$args
    }
  }

  method varsinit { } {
    my variable vars

    set vars(activatemenu.in) false
    set vars(active.dictkey) -1
    set vars(active.dictkey.last) -1 ; # for <<MenuSelect>>
    set vars(active.itemno) -1
    set vars(bind.closecheck) false
    set vars(bind.mousehandler) false
    set vars(cascade.posted) [list]   ; # dictionary keys of posted menu
    set vars(closecheck.active) false
    set vars(closecheck.skip) false
    set vars(confchanged) true
    set vars(config.count) 0
    set vars(config.count.base) 0
    set vars(defaultmenubarpadx) 4p
    set vars(defaultmenubarpady) 4p
    set vars(defaultpadx) 2p
    set vars(defaultpady) 1p
    set vars(dictkey.new) -1
    set vars(dictkey.prior) -1
    set vars(invoke.window) {}
    set vars(isconfigured) 0
    set vars(item.count) 0
    set vars(item.map) [list]
    set vars(itemno.invoked) -1
    set vars(itemno.new) -1
    set vars(itemno.prior) -1
    set vars(items) [dict create]
    set vars(key.skip) false
    set vars(keysym) {}
    set vars(keysym.last) {} ; # for ISO_Next_Group
    set vars(keysym.meta) {}
    set vars(last.x) {}
    set vars(last.y) {}
    # menu.active indicates that either the mouse pointer
    # is in the menu, or the arrow keys are active for the
    # menu.
    set vars(menu.active) false
    set vars(menu.binds) false
    # menu.posted indicates that the menu has been drawn
    set vars(menu.posted) false
    set vars(parent) {}
    set vars(postprocess) false
    set vars(postprocess.in) false
    set vars(sticky.mode) false
    set vars(tearoff.dictkey) {}
    set vars(tframeheight) 0
    set vars(tframewidth) 0
    set vars(waitpostprocess.in) false
    set vars(withdrawn) false
  }

  # styling

  method _initStyling { } {
    my variable vars

    if { [::flexmenucmd::initStyling] } {
      return
    }

    ::ttk::style configure Flexmenu.TLabel \
        -relief flat \
        -padding 0
    my _setStyling Flexmenu.TLabel

    ::ttk::style configure Accelerator.Flexmenu.TLabel \
        -relief flat
    my _setStyling Accelerator.Flexmenu.TLabel

    ::ttk::style configure Flexmenu.TCheckbutton \
        -font $vars(-font) \
        -background $vars(-background)
    my _setStyling Flexmenu.TCheckbutton

    ::ttk::style configure Flexmenu.TRadiobutton \
        -font $vars(-font) \
        -background $vars(-background)
    my _setStyling Flexmenu.TRadiobutton

    ::ttk::style configure Flexmenu.Horizontal.TSeparator \
        -background $vars(-background)
    ::ttk::style configure Flexmenu.Vertical.TSeparator \
        -background $vars(-background)
    return
  }

  # for use by the main menu for initialization only.
  # not used by submenus
  method _setStyling { style {dictkey {}} } {
    my variable vars

    set afgkey -activeforeground
    set fgkey -foreground
    set fkey -font
    if { [regexp Accelerator $style] } {
      set fgkey -acceleratorforeground
      set afgkey -acceleratoractiveforeground
      set fkey -acceleratorfont
    }
    foreach {nm k} [list \
        bg -background \
        fg $fgkey \
        abg -activebackground \
        afg $afgkey \
        dfg -disabledforeground \
        font $fkey \
        ] {
      if { [info exists vars($k)] } {
        set $nm $vars($k)
      }
      if { $dictkey ne {} &&
          [dict exists $vars(items) $dictkey $k] } {
        set $nm [dict get $vars(items) $dictkey $k]
      }
    }

    ::ttk::style configure $style \
        -background $bg \
        -foreground $fg \
        -font $font
    ::ttk::style map $style \
        -background [list {user1 !disabled} $abg \
            {hover !disabled} $abg \
            {focus !disabled} $abg \
            ] \
        -foreground [list {user1 !disabled} $afg \
            {hover !disabled} $afg \
            {focus !disabled} $afg \
            disabled $dfg \
            ]
  }

  # for use by submenus, items and post initialization configuration
  method _confStyling { style conflist {dictkey {}} } {
    my variable vars

    set clist [list]
    set domap false
    set abg {}
    set bg [::ttk::style map $style -background]
    if { $bg eq {} } {
      regsub {^[^\.]+\.} $style {} basestyle
      set bg [::ttk::style map $basestyle -background]
    }
    if { $bg ne {} } {
      set abg [lindex $bg 1]
    }
    set fg [::ttk::style map $style -foreground]
    if { $fg eq {} } {
      regsub {^[^\.]+\.} $style {} basestyle
      set fg [::ttk::style map $basestyle -foreground]
    }
    set afg {}
    set dfg {}
    foreach {k v} $fg {
      if { $k eq "disabled" } {
        set dfg $v
      } else {
        set afg $v
      }
    }

    foreach {k v} $conflist {
      if { $k eq "-activebackground" } {
        set abg $v
        set domap true
      } elseif { $k eq "-activeforeground" } {
        set afg $v
        set domap true
      } else {
        lappend clist $k $v
      }
    }

    if { [llength $clist] > 0 } {
      ::ttk::style configure $style {*}$clist
    }
    if { $domap } {
      set fglist [list \
          {user1 !disabled} $afg \
          {hover !disabled} $afg \
          {focus !disabled} $afg \
          ]
      if { $dfg ne {} } {
        lappend fglist disabled $dfg
      }
      ::ttk::style map $style \
          -background [list {user1 !disabled} $abg \
            {hover !disabled} $abg \
            {focus !disabled} $abg \
            ] \
          -foreground $fglist
    }
  }

  # state handling

  method _getState { } {
    my variable vars

    return [dict get $vars(states.rev) $vars(menu.state)]
  }

  method _setState { state } {
    my variable vars

    # force an error if the key does not exist
    set v [dict get $vars(states) $state]
    set vars(menu.state) $v
    return $vars(menu.state)
  }

  method _isState { state } {
    my variable vars

    # force an error if the key does not exist
    set v [dict get $vars(states) $state]
    set rc [expr {$vars(menu.state) == $v}]
    return $rc
  }

  method _getAction { } {
    my variable vars

    return [dict get $vars(actions.rev) $vars(menu.action)]
  }

  method _setAction { action } {
    my variable vars

    # force an error if the key does not exist
    set v [dict get $vars(actions) $action]
    set vars(menu.action) $v
    return $vars(menu.action)
  }

  # if any of the actions in the list are true, return true
  method _isAction { args } {
    my variable vars

    set v [my _getAction]
    return [expr {$v in $args}]
  }

  # State table:
  #
  # menubar:  open: down  close: up    move: left, right
  # menuleft: open: right close: left  move: up, down
  #
  # key.close:        key: close a cascade (with appropriate key or escape)
  # key.invoke:       key: open cascade/item with key
  # key.move:         key: moving within the menu
  # menu.close:       key: escape
  # menu.enter:       mouse: enters menu; activate parent menu
  # menu.leave:       mouse: moved outside menu
  # menu.open:        key: F10
  # menu.post:        menu is posted
  # menu.unpost:      menu is unposted
  # mouse.click.out:  mouse: click outside the menu
  # mouse.invoke:     mouse: invoke item
  # mouse.move:       mouse: move with the menu
  # mouse.move.sticky: mouse: move with the menu while in sticky mode
  #
  # The vars(cascade.posted) list is used to determine additional
  # actions.
  #
  #   state         type        action            new state
  #
  # All menus start in an inactive state.
  # Sub-menus go straight to a wait state after they are posted.
  # Non-sub menus go active if a movement action is made.
  #   inactive      submenu      menu.post             wait
  #   inactive      !submenu     mouse.move, key.move  active
  #   inactive      !submenu     menu.open             active (F10)
  #   inactive      !submenu     cascade/key.invoke    cascade.post.active
  #   inactive      !submenu     !cascade/key.invoke   active
  #   inactive      !submenu     cascade/mouse.invoke  cascade.post.active
  #   inactive      !submenu     !cascade/mouse.invoke active
  # The active state is for non-sub-menus only.  No cascade has
  # yet been opened.
  #   active        !submenu     key.close             inactive
  #   active        !submenu     menu.close            inactive
  #   active        !submenu     menu.leave            active
  #   active        !submenu     key.move              active
  #   active        !submenu     mouse.move            active
  #   active        !submenu     cascade/mouse.invoke  inuse
  #   active        !submenu     cascade/key.invoke    cascade.post.active
  #   active        !submenu     cascade/menu.open     cascade.post.active
  #   active        !submenu     !cascade/mouse.invoke active
  #   active        !submenu     !cascade/key.invoke   active
  #   active        !submenu     menu.unpost           inactive
  # The cascade.post.active state is a intermediate state.
  # It will post a cascade, then select the first item in the cascade
  # and activate the cascade.
  #   cascade.post.active !submenu -                   wait
  # The inuse state is an active menu.  Cascades are automaticlly opened
  # on mouse movement.
  #   inuse         -           key.move              inuse
  #   inuse         -           mouse.move            inuse
  #   inuse         submenu      menu.leave            wait
  #   inuse         !submenu     menu.leave/no cascade inactive
  #   inuse         -           cascade/key.invoke    cascade.post.active
  #   inuse         -           cascade/mouse.invoke  inuse
  #   inuse         -           !cascade/key.invoke/-keepopen
  #                                                   inuse
  #   inuse         -           !cascade/mouse.invoke/-keepopen
  #                                                    inuse
  #   inuse         -           !cascade/key.invoke/!-keepopen
  #                                                   inactive (close parent)
  #   inuse         -           !cascade/mouse.invoke/!-keepopen
  #                                                   inactive (close parent)
  #   inuse         -           key.close             close
  #   inuse         -           menu.close            close
  #   inuse         submenu      menu.unpost           inactive
  # The wait state is used when another menu is active
  # or the mouse is not in frame.
  # A menu.enter message is processed to make it active again.
  #   wait          -           key.move          inuse
  #   wait          -           menu.enter        inuse
  #       mouse.move is not processed, otherwise overlapping
  #       menus will not work properly.
  #   wait          -           mouse.move        -
  #   wait          -           mouse.click.out
  #                             no cascade open   close
  #   wait          submenu      menu.close        close
  #   wait          submenu      menu.unpost       inactive
  #   wait          -           key.invoke        wait
  # The close state is an intermediate state.
  #   close         -           -                 inactive
  #
  method _processState { } {
    my variable vars

    set continueProcessing true
    set previousstate $vars(menu.state)
    set nextstate {}
    while { $continueProcessing } {
      set continueProcessing false

      set currstate [my _getState]
      set type $vars(-type)
      set itemType {}
      if { $vars(active.dictkey) != -1 &&
          ! [dict exists $vars(items) $vars(active.dictkey)] } {
        return
      }
      if { ! [my _isState close] && $vars(active.dictkey) != -1 } {
        set itemType [dict get $vars(items) $vars(active.dictkey) type]
      }
# keep
#puts "ps: $vars(itemframe) state:$currstate action:[my _getAction] itemtype:$itemType"
      switch -exact -- $currstate {
        inactive {
          if { $type eq "submenu" } {
            if { [my _isAction menu.post] } {
              set currstate [my _setState wait]
            }
          } else {
            if { [my _isAction mouse.move mouse.move.sticky key.move] } {
              my _activateMenu
              set currstate [my _setState active]
            } elseif { [my _isAction menu.open] } {
              set vars(dictkey.new) [lindex $vars(item.map) 0]
              set vars(itemno.new) 0
              my _activateMenu
              set itemType [my _setActiveItem]
              if { $itemType eq "cascade" } {
                my _postCascade
                set currstate [my _setState cascade.post.active]
                set continueProcessing true
              } else {
                set currstate [my _setState active]
              }
            } elseif { [my _isAction key.invoke mouse.invoke] } {
              my _activateMenu
              set itemType [my _setActiveItem]
              if { $itemType eq "cascade" } {
                my _postCascade
                set currstate [my _setState cascade.post.active]
                set continueProcessing true
              } else {
                set dictkey $vars(active.dictkey)
                set itemno $vars(active.itemno)
                my _invokeItem $dictkey $itemno
                set currstate [my _setState active]
              }
            }
          } ; # not submenu
        }
        active {
          # the active state is only used by menubar and menuleft
          # top level menus.  it is the state where the menu is
          # active, but no cascade has yet been opened.
          if { [my _isAction key.close menu.close] } {
            my _unpostCascade
            set currstate [my _setState inactive]
            set continueProcessing true
          } elseif { [my _isAction menu.leave] } {
            my _deactivateItem
          } elseif { [my _isAction key.move mouse.move mouse.move.sticky] } {
            my _activateMenu
            my _setActiveItem
          } elseif { [my _isAction mouse.invoke key.invoke menu.open] } {
            set itemType [my _setActiveItem]
            if { $itemType eq "cascade" } {
              my _postCascade
              if { [my _isAction key.invoke menu.open] } {
                set currstate [my _setState cascade.post.active]
                set continueProcessing true
              } else {
                set currstate [my _setState inuse]
              }
            } else {
              set dictkey $vars(active.dictkey)
              set itemno $vars(active.itemno)
              my _invokeItem $dictkey $itemno
            }
          } elseif { [my _isAction menu.unpost] } {
            my _unpostCascade
            my _deactivateMenu
            set currstate [my _setState inactive]
          }
        }
        cascade.post.active {
          my _postCascade
          my _deactivateMenu
          my _selectFirst [my _isAction key.invoke]
          set currstate [my _setState wait]
        }
        inuse {
          if { [my _isAction mouse.move.sticky] } {
            my _setActiveItem
          } elseif { [my _isAction mouse.move key.move] } {
            my _unpostCascade
            set itemType [my _setActiveItem]
            if { $itemType eq "cascade" &&
                ! [my _isAction key.move] } {
              my _postCascade
            }
          } elseif { [my _isAction menu.leave] } {
            my _deactivateItem
            my _deactivateMenu
            set currstate [my _setState wait]
            if { $type ne "submenu" &&
                [llength $vars(cascade.posted)] == 0 } {
              set currstate [my _setState inactive]
            }
          } elseif { [my _isAction key.invoke mouse.invoke] } {
            set itemType [my _setActiveItem]
            if { $itemType eq "cascade" } {
              if { [my _isAction key.invoke mouse.invoke] } {
                set currstate [my _setState cascade.post.active]
                set continueProcessing true
              } else {
                set currstate [my _setState inuse]
              }
            } else {
              set dictkey $vars(active.dictkey)
              set itemno $vars(active.itemno)
              set itemkeepopen false
              if { $vars(active.dictkey) != -1 &&
                  [dict exists $vars(items) $vars(active.dictkey) -keepopen] } {
                set itemkeepopen [dict get $vars(items) $vars(active.dictkey) -keepopen]
              }
              if { ! $vars(-keepopen) && ! $itemkeepopen } {
                # need to close the menu before invoking.
                # invokeItem will work on a non-active menu.
                # this makes things messy due to
                # active.dictkey and active.itemno
                if { $type eq "submenu" } {
                  my _deactivateItem
                  my _deactivateMenu
                  my _unpostMenu
                  if { $vars(parent) ne {} } {
                    $vars(parent) close -internal
                  }
                  set currstate [my _setState inactive]
                }
              }
              my _invokeItem $dictkey $itemno
            }
          } elseif { [my _isAction key.close menu.close] } {
            if { $vars(parent) ne {} } {
              $vars(parent) activatemenu
            }
            my _unpostCascade
            set currstate [my _setState close]
            set continueProcessing true
          } elseif { [my _isAction menu.unpost] } {
            my _unpostCascade
            my _deactivateItem
            my _deactivateMenu
            my _unpostMenu
            set currstate [my _setState inactive]
          }
        }
        wait {
          if { [my _isAction key.move] } {
            my _activateMenu
            set itemType [my _setActiveItem]
            set currstate [my _setState inuse]
          } elseif { [my _isAction menu.enter] } {
            my _activateMenu
            set currstate [my _setState inuse]
          } elseif { [my _isAction mouse.click.out] } {
            if { [llength $vars(cascade.posted)] == 0 } {
              set currstate [my _setState close]
              set continueProcessing true
            }
          } elseif { [my _isAction menu.close] } {
            set currstate [my _setState close]
            set continueProcessing true
          } elseif { [my _isAction menu.unpost] } {
            my _unpostCascade
            my _deactivateItem
            my _deactivateMenu
            my _unpostMenu
            set currstate [my _setState inactive]
          } elseif { [my _isAction key.invoke mouse.invoke] } {
            my _unpostCascade
            my _activateMenu
            set itemType [my _setActiveItem]
            set dictkey $vars(active.dictkey)
            set itemno $vars(active.itemno)
            set itemkeepopen false
            if { $vars(active.dictkey) != -1 &&
                [dict exists $vars(items) $vars(active.dictkey) -keepopen] } {
              set itemkeepopen [dict get $vars(items) $vars(active.dictkey) -keepopen]
            }
            if { ! $vars(-keepopen) && ! $itemkeepopen } {
              # need to close the menu before invoking.
              # invokeItem will work on a non-active menu.
              if { $type eq "submenu" } {
                my _deactivateItem
                my _deactivateMenu
                my _unpostMenu
                if { $vars(parent) ne {} } {
                  $vars(parent) close -internal
                }
                set currstate [my _setState inactive]
              }
            }
            set currstate [my _setState inuse]
            my _invokeItem $dictkey $itemno
          }
        }
        close {
          my _unpostCascade
          my _deactivateMenu
          if { $type eq "submenu" } {
            my _deactivateItem
            my _unpostMenu
            if { $vars(parent) ne {} } {
              $vars(parent) close -internal
            }
          }
          set currstate [my _setState inactive]
        }
      } ; # switch on state

    } ; # while continue process
    if { $previousstate != $vars(menu.state) } {
# keep
#puts "   ps: $vars(itemframe) now [my _getState] ([my _getAction]) "
    }

    my _setAction none
  }

  # state transition helpers

  method _setActiveItem { } {
    my variable vars

    if { $vars(dictkey.new) != -1 &&
        $vars(dictkey.new) != $vars(active.dictkey) } {
      set vars(dictkey.prior) $vars(active.dictkey)
      set vars(itemno.prior) $vars(active.itemno)
      my _deactivateItem
      set vars(active.dictkey) $vars(dictkey.new)
      set vars(active.itemno) $vars(itemno.new)
      my _activateItem
    }
    set itemType {}
    if { $vars(active.dictkey) != -1 && $vars(active.dictkey) ne {} } {
      set itemType [dict get $vars(items) $vars(active.dictkey) type]
    }
    return $itemType
  }

  method _unpostCascade { } {
    my variable vars

    if { [llength $vars(cascade.posted)] == 0 } {
      return
    }

    set dictkey [lindex $vars(cascade.posted) 0]
    if { $dictkey == -1 } {
      return
    }

    if { ! [dict exists $vars(items) $dictkey] } {
      return
    }

    set menu [dict get $vars(items) $dictkey -menu]
    if { $menu ne {} } {
      $menu unpost
    }

    set vars(cascade.posted) [lrange $vars(cascade.posted) 1 end]
    $menu unpost

    if { $vars(menu.active) } {
      if { [winfo exists $vars(itemframe)] &&
          [winfo viewable $vars(itemframe)] &&
          $vars(-type) eq "submenu" &&
          ! $vars(-tornoff) } {
        # take the grab for the current menu
        grab set -global $vars(itemframe)
        set vars(closecheck.active) true
      }
    }
  }

  method _postCascade { } {
    my variable vars

    my execprecmd $vars(active.dictkey)

    set menu [dict get $vars(items) $vars(active.dictkey) -menu]
    if { $menu eq {} } {
      return
    }

    set state [dict get $vars(items) $vars(active.dictkey) -state]
    if { $state eq "disabled" } {
      return
    }

    set yalign [dict get $vars(items) $vars(active.dictkey) -yalign]
    set xalign [dict get $vars(items) $vars(active.dictkey) -xalign]

    if { $vars(-type) eq "menubar" } {
      set x [dict get $vars(items) $vars(active.dictkey) x.left]
      set y [dict get $vars(items) $vars(active.dictkey) y.bottom]
      set bw [::flexmenucmd::topixels 1]
      incr y $bw
    } else {
      set xalignval {}
      set xaligntxt menuright
      regexp {^(menuleft|activeleft|activeright|menuright)?([+-]?\d+)?$} \
          $xalign all xaligntxt xalignval
      switch -exact -- $xaligntxt {
        menuleft {
          set x [dict get $vars(items) [lindex $vars(item.map) 0] x.left]
        }
        activeleft {
          set x [dict get $vars(items) $vars(active.dictkey) x.left]
        }
        activeright {
          set x [dict get $vars(items) $vars(active.dictkey) x.right]
        }
        menuright -
        default {
          set x [winfo width $vars(itemframe)]
        }
      }
      if { $xalignval ne {} } {
        incr x $xalignval
      }

      set yalignval {}
      set yaligntxt activetop
      regexp {^(menutop|activetop)?([+-]?\d+)?$} $yalign all yaligntxt yalignval
      switch -exact -- $yaligntxt {
        menutop {
          set y [dict get $vars(items) [lindex $vars(item.map) 0] y.top]
        }
        activetop -
        default {
          set y [dict get $vars(items) $vars(active.dictkey) y.top]
        }
      }
      if { $yalignval ne {} } {
        incr y $yalignval
      }
    }
    set w $vars(itemframe)
    # rootx/rooty work for both linux and windows
    set tx [winfo rootx $w]
    set ty [winfo rooty $w]
    # the tk menu commands use absolute coordinates.
    incr x $tx
    incr y $ty

    my execcmd $vars(active.dictkey)
    if { [lindex $vars(cascade.posted) end] != $vars(active.dictkey) } {
      lappend vars(cascade.posted) $vars(active.dictkey)
    }
    $menu post $x $y $vars(itemframe)
    set vars(closecheck.active) false
  }

  method _selectFirst { keyskip } {
    my variable vars

    set menu [dict get $vars(items) $vars(active.dictkey) -menu]
    if { $menu ne {} } {
      $menu selectfirst $keyskip
    }
  }

  method _invokeItem { dictkey itemno } {
    my variable vars

    set reset false
    set vars(itemno.invoked) $itemno
    if { $dictkey eq {} } {
      set dictkey $vars(active.dictkey)
    }

    if { $dictkey != -1 } {
      set type [dict get $vars(items) $dictkey type]
      switch -exact -- $type {
        cascade {
          my postcascade $itemno
        }
        check -
        checkbutton -
        radio -
        radiobutton {
          set iarg -invoke
          if { $vars(invoke.window) ne {} &&
              [winfo class $vars(invoke.window)] eq "TCheckbutton" } {
            set iarg {}
          }
          my execbutton $dictkey $iarg
        }
        tearoff {
          my tearoff $dictkey
        }
        command {
          my execcmd $dictkey
        }
      }
    }
  }

  # helper routines

  method _addBindTag { w btnm } {
    set bt [bindtags $w]
    if { [lsearch -exact $bt $btnm] == -1 } {
      bindtags $w [concat $bt $btnm]
    }
  }

  method dispmenubar { } {
    my variable vars

    $vars(itemframe.cmd) configure \
        -relief flat \
        -borderwidth $vars(-borderwidth)
    my _postMenu  ; # already packed by user
    if { ! $vars(postprocess) } {
      my waitpostprocess {}
    }
  }

  method _getdictkey { itemno } {
    my variable vars

    if { $itemno < 0 } {
      # error: release grab
      if { [winfo exists $vars(itemframe)] } {
        grab release $vars(itemframe)
      }
      set vars(closecheck.active) false
      throw {MENU INDEX OUTOFRANGE {menu index out of range}} {bad menu entry index}
      return
    }
    if { $itemno >= $vars(item.count) } {
      set itemno end
    }
    return [lindex $vars(item.map) $itemno]
  }

  method _inWindow { w } {
    my variable vars

    set rc 0
    if { ! [winfo exists $w] } {
      return $rc
    }

    lassign [winfo pointerxy $w] x y
    set bxl [winfo rootx $w]
    set byt [winfo rooty $w]
    set bxr [expr {$bxl+[winfo width $w]}]
    set byb [expr {$byt+[winfo height $w]}]
    if { $x >= $bxl && $x <= $bxr && $y >= $byt && $y <= $byb } {
      incr rc
    }
    return $rc
  }

  # bind handlers

  method openmenu { } {
    my variable vars

    my _setAction menu.open
    my _processState
  }

  method closemenu { } {
    my variable vars

    my _setAction menu.close
    my _processState
  }

  method mouseinvoke { win dictkey } {
    my variable vars

    set vars(invoke.window) $win
    my _setAction mouse.invoke
    my _processState
    set vars(invoke.window) {}
    return -code ok
  }

  method keyinvoke { win {dictkey {}} } {
    my variable vars

    if { $dictkey eq {} } {
      if { $vars(active.dictkey) == -1 } {
        return
      }
      set vars(dictkey.new) $vars(active.dictkey)
      set vars(itemno.new) $vars(active.itemno)
    } else {
      set vars(dictkey.new) $dictkey
      set vars(itemno.new) -1
    }
    set vars(invoke.window) $win

    set keepopen [dict get $vars(items) $vars(dictkey.new) -keepopen]
    if { ! $keepopen } {
      my _setAction key.invoke
      my _processState
    }
    set vars(invoke.window) {}
    return -code ok
  }

  method upDownHandler { dir tag } {
    my variable vars

    if { $vars(key.skip) } {
      set vars(keysym) $vars(keysym.meta)
      set vars(key.skip) false
      return -code ok
    }
    if { ! $vars(menu.active) } {
      return -code ok
    }

    if { ! $vars(postprocess) } {
      my waitpostprocess [list my upDownHandler $dir $tag]
      return -code ok
    }
    if { $vars(active.itemno) == -1 } {
      return -code ok
    }

    if { $dir == 1 &&
        $vars(-type) eq "menubar" } {
      set dictkey [my _getdictkey $vars(active.itemno)]
      if { [dict get $vars(items) $dictkey type] eq "cascade" } {
        my _setAction key.invoke
        my _processState
      }
      return -code ok
    }

    set itemno $vars(active.itemno)

    if { $dir == -1 &&
        $itemno == 0 &&
        $vars(parent) ne {} &&
        [$vars(parent) cget -type] ne "submenu" } {
      my _setAction key.close
      my _processState
      return -code ok
    }

    incr itemno $dir
    if { $itemno < 0 || $itemno >= $vars(item.count) } {
      return -code ok
    }

    set dictkey [my _getdictkey $itemno]
    while { $dictkey != -1 && [my _skipItem $dictkey] } {
      incr itemno $dir
      if { $itemno < 0 || $itemno >= $vars(item.count) } {
        return -code ok
      }
      set dictkey [my _getdictkey $itemno]
    }

    set vars(dictkey.new) $dictkey
    set vars(itemno.new) $itemno
    my _setAction key.move
    my _processState
    return -code ok
  }

  method rightLeftHandler { dir tag } {
    my variable vars

    if { $vars(key.skip) } {
      set vars(keysym) $vars(keysym.meta)
      set vars(key.skip) false
      return -code ok
    }
    if { ! $vars(menu.active) } {
      return -code ok
    }

    if { ! $vars(postprocess) } {
      my waitpostprocess [list my rightLeftHandler $dir $tag]
      return -code ok
    }

    if { $vars(active.dictkey) == -1 || $vars(active.dictkey) eq {} } {
      if { $vars(dictkey.prior) != -1 } {
        set dictkey $vars(dictkey.prior)
      } else {
        set dictkey [lindex $vars(item.map) 0]
      }
      set vars(dictkey.new) $dictkey
      set vars(itemno.new) 0
      my _setAction key.move
      my _processState
      return -code ok
    }

    if { $dir == 1 &&
        $vars(-type) ne "menubar" &&
        [dict get $vars(items) $vars(active.dictkey) type] eq "cascade" } {
      set menu [dict get $vars(items) $vars(active.dictkey) -menu]
      if { $menu ne {} } {
        my _setAction key.invoke
        my _processState
      }
      return -code ok
    }

    set in false
    set currcol 0
    set ytop 0
    set ndictkey -1
    set nidx -1
    set otcol -1
    set idx $vars(active.itemno)
    set currcol [dict get $vars(items) $vars(active.dictkey) startcol]
    set ytop [dict get $vars(items) $vars(active.dictkey) y.top]
    set ybottom [dict get $vars(items) $vars(active.dictkey) y.top]
    incr idx $dir

    if { $dir == -1 &&
        $currcol == 0 &&
        $vars(parent) ne {} } {
      my _setAction key.close
      my _processState
      return -code ok
    }

    while { $idx >= 0 && $idx < $vars(item.count) } {
      set dictkey [lindex $vars(item.map) $idx]
      set tcol [dict get $vars(items) $dictkey startcol]
      if { ($dir == 1 && $currcol < $tcol) ||
          ($dir == -1 && $currcol > $tcol) } {
        if { $otcol == -1 } { set otcol $tcol }
        if { ($dir == 1 && $otcol < $tcol) ||
            ($dir == -1 && $otcol > $tcol) } {
          # do not want to go more than one column
          break
        }
        set tytop [dict get $vars(items) $dictkey y.top]
        set tybottom [dict get $vars(items) $dictkey y.bottom]

        # if there's no good match, just take
        # the last in the column
        set ndictkey $dictkey
        set nidx $idx
        if { ($dir == 1 && $ytop <= $tytop) ||
            ($dir == -1 && $ytop >= $tytop) } {
          break
        }
      }
      incr idx $dir
    }
    if { $ndictkey != -1 } {
      set vars(dictkey.new) $ndictkey
      set vars(itemno.new) $nidx
      my _setAction key.move
      my _processState
    }
    return -code ok
  }

  # post/unpost routines

  method setconfigure { type w dictkey } {
    my variable vars

    incr vars(isconfigured)
    if { $type eq "map" } {
      bind $w <Map> {}
    } else {
      bind $w <Configure> {}
      if { $dictkey != -1 } {
        dict set vars(items) $dictkey changed false
      }
    }
    if { $vars(isconfigured) >= $vars(config.count) } {
      if { $vars(-type) ne "submenu" } {
        after idle [list [self] dispmenubar]
      } else {
        after idle [list [self] waitpostprocess {}]
      }
      set vars(config.count) 0
      set vars(config.count.base) 0
    }
  }

  method waitpostprocess { cmd } {
    my variable vars

    if { $vars(postprocess) } {
      set vars(waitpostprocess.in) false
    }
    if { $vars(isconfigured) < $vars(config.count) ||
        $vars(postprocess.in) || $vars(waitpostprocess.in) } {
      after 100 [list [self] waitpostprocess $cmd]
      return
    }
    if { ! $vars(postprocess) } {
      set vars(waitpostprocess.in) true
      my _postprocess
    }
    if { $cmd ne {} } {
      {*}$cmd
    }
    set vars(waitpostprocess.in) false
  }

  method _postprocessBackgrounds { } {
    my variable vars

    # this is need to support -background and -foreground, but
    # to reduce the number of widgets, a background
    # is only created if the background is changed from the default
    foreach {dictkey} $vars(item.map) {
      set bg [dict get $vars(items) $dictkey -background]

      if { $bg ne $vars(-background) } {
        if { ! [winfo exists $vars(itemframe).$dictkey-bgframe] } {
          set bgf [frame $vars(packframe).$dictkey-bgframe \
              -background [dict get $vars(items) $dictkey -background] \
              -relief [dict get $vars(items) $dictkey -relief] \
              -borderwidth [dict get $vars(items) $dictkey -borderwidth]]
          dict set vars(items) $dictkey backgroundframe $bgf
        } else {
          set bgf [dict get $vars(items) $dictkey backgroundframe]
        }

        $bgf configure -width [dict get $vars(items) $dictkey width]
        $bgf configure -height [dict get $vars(items) $dictkey height]
        set tx [dict get $vars(items) $dictkey x.left]
        set ty [dict get $vars(items) $dictkey y.top]
        place $bgf -in $vars(itemframe) \
            -relx 0.0 -rely 0.0 \
            -x $tx -y $ty
        lower $bgf
      }
    }
  }

  method _postprocessSetData { startitem x mx } {
    my variable vars

    set padx [expr {$vars(-padx) eq {} ? $vars(defaultpadx) : $vars(-padx)}]
    set padx [::flexmenucmd::topixels $padx]

    for {set itemno $startitem} {$itemno < $vars(item.count)} {incr itemno} {
      set dictkey [lindex $vars(item.map) $itemno]
      if { $itemno != $startitem &&
          [dict get $vars(items) $dictkey -columnbreak] } {
        break
      }
      dict set vars(items) $dictkey x.left [expr {$x-$padx-1}]
      dict set vars(items) $dictkey x.right [expr {$mx+$padx}]
      dict set vars(items) $dictkey width [expr {$mx-$x+2*$padx}]
      if { $vars(-maxheight) != 0 } {
        set vars(tframewidth) [expr {max($vars(tframewidth), \
            [dict get $vars(items) $dictkey width])}]
      }
    }
  }

  method _postprocess { } {
    my variable vars

    if { $vars(postprocess.in) } {
      return
    }
    set vars(postprocess.in) true

    set pady [expr {$vars(-pady) eq {} ? $vars(defaultpady) : $vars(-pady)}]
    if { [regexp {^(\d+)p$} $pady all val] } {
      set pady [expr {round([tk scaling]*$val)}]
    }

    set bw [::flexmenucmd::topixels 1]

    set currcol 0
    set row 0
    set x 9999
    set mx 0
    set startitem 0
    set vars(tframeheight) 0
    set vars(tframewidth) 0
    for {set itemno 0} {$itemno < $vars(item.count)} {incr itemno} {
      set dictkey [lindex $vars(item.map) $itemno]
      if { $itemno != $startitem &&
          [dict get $vars(items) $dictkey -columnbreak] } {
        my _postprocessSetData $startitem $x $mx
        incr currcol
        set row 0
        set x 9999
        set mx 0
        set startitem $itemno
      }

      set wlist [dict get $vars(items) $dictkey widgetlist]

      set y 9999
      set my 0
      foreach {w} $wlist {
        set cx [winfo x $w]
        set cy [winfo y $w]
        if { [winfo parent $w] ne $vars(packframe) } {
          # if the widget is not owned by the item frame, we need
          # to do more calculations to find its location.
          set ty [winfo rooty $w]
          set ify [winfo rooty $vars(packframe)]
          set cy [expr {$ty - $ify}]

          set tx [winfo x $w]
          set ifx [winfo x $vars(packframe)]
          set cx [expr {$tx - $ifx}]
        }
        set x [expr {min($cx,$x)}]
        set mx [expr {max([winfo width $w]+$cx,$mx)}]
        set y [expr {min($cy,$y)}]
        set my [expr {max([winfo height $w]+$cy,$my)}]
      }

      dict set vars(items) $dictkey y.top [expr {$y-$pady-1}]
      dict set vars(items) $dictkey y.bottom [expr {$my+$pady-1}]
      dict set vars(items) $dictkey height [expr {$my-$y+2*$pady}]
      if { $vars(-maxheight) != 0 && $row < $vars(-maxheight) } {
        incr vars(tframeheight) [dict get $vars(items) $dictkey height]
      }
      incr row
    }
    my _postprocessSetData $startitem $x $mx

    if { $vars(-maxheight) != 0 } {
      set sz [expr {[font metrics flexmenusmall -ascent] + \
          [font metrics flexmenusmall -descent]}]
      $vars(textframe) configure \
          -width [expr {round(double($vars(tframewidth)) / \
              double([font measure flexmenusmall 0])+2)}] \
          -height [expr {round(double($vars(tframeheight)) / double($sz))}]
    }
    my _postprocessBackgrounds

    set vars(postprocess) true
    set vars(postprocess.in) false
  }

  method _packitem { dictkey wlist startcol row cspan colidx } {
    my variable vars

    set nwlist [list]
    set padx [expr {$vars(-padx) eq {} ? $vars(defaultpadx) : $vars(-padx)}]
    set pady [expr {$vars(-pady) eq {} ? $vars(defaultpady) : $vars(-pady)}]

    if { [dict get $vars(items) $dictkey type] eq "tearoff" } {
      if { $vars(-tearoff) } {
        grid $wlist -in $vars(packframe) -padx 5p -pady 0 -sticky ew \
            -column $startcol -row $row
        grid configure [lindex $wlist end] -columnspan $cspan
        lappend nwlist $wlist
      }
    } elseif { [dict get $vars(items) $dictkey type] eq "separator" } {
      if { $vars(-type) eq "menubar" } {
        grid $wlist -in $vars(packframe) -pady 2p -padx 0 -sticky ns \
            -column $startcol -row $row
        grid configure [lindex $wlist end] -columnspan $cspan
        lappend nwlist $wlist
      } else {
        grid $wlist -in $vars(packframe) -padx 5p -pady 0 -sticky ew \
            -column $startcol -row $row
        grid configure [lindex $wlist end] -columnspan $cspan
        lappend nwlist $wlist
      }
    } else {
      foreach {w} $wlist {
        if { $w ne {} } {
          grid $w -in $vars(packframe) -padx $padx -pady $pady \
              -sticky w -column $startcol -row $row
          lappend nwlist $w
        }
        if { $startcol == $colidx } {
          if { $w ne {} } {
            grid configure $w -columnspan $cspan
          }
          incr startcol $cspan
        } else {
          incr startcol
        }
      }
      if { $vars(-type) eq "submenu" &&
          ! [dict get $vars(items) $dictkey -hidemargin] } {
        set w [lindex $wlist [expr {$colidx-1}]]
        if { $w ne {} } {
          grid configure $w -padx $padx -sticky e
        }
      }
    }
    dict set vars(items) $dictkey widgetlist $nwlist
  }

  method _layoutmenu { } {
    my variable vars

    if { ! $vars(confchanged) } {
      return
    }

    set vars(config.count) $vars(config.count.base)
    set vars(isconfigured) 0

    set startcol 0
    set currcol 0
    set maxcol 0
    set row 0
    set colbreak [expr {$vars(-columnbreak) < 1 ? 0 : $vars(-columnbreak)}]

    if { $vars(-maxheight) != 0 } {
      if { [llength $vars(item.map)] > $vars(-maxheight) } {
        grid $vars(scrollbar) -in $vars(itemframe) -row 0 -column 1 -sticky ns
      } else {
        grid forget $vars(scrollbar)
      }
    }

    set haverightlabel false
    set hidemarginmixed false
    set thidemargin 0
    # want the right arrows for menuleft
    if { $vars(-type) ne "menubar" || $vars(-hidearrows) } {
      foreach {dictkey} $vars(item.map) {
        set info [dict get $vars(items) $dictkey]
        if { [dict get $vars(items) $dictkey -hidemargin] } {
          incr thidemargin
        }
        if { [dict get $vars(items) $dictkey rightlabel] ne {} } {
          set haverightlabel true
        }
      }
    }
    if { $thidemargin != 0 && $thidemargin != [llength $vars(item.map)] } {
      set hidemarginmixed true
    }

    set itemno 0
    foreach {dictkey} $vars(item.map) {
      set info [dict get $vars(items) $dictkey]
      set cbreak [dict get $info -columnbreak]

      if { ($colbreak != 0 && $itemno != 0 && $itemno % $colbreak == 0) ||
          $cbreak } {
        incr currcol
        incr startcol $maxcol
        set row 0
        dict set vars(items) $dictkey -columnbreak true
      }

      set gap [dict get $info -gap]
      set type [dict get $info type]
      set wkeylist [dict get $info wkeylist]
      set cspan [dict get $info cspan]
      set wlist [list]
      set colidx 0
      set hidemargin [dict get $vars(items) $dictkey -hidemargin]

      # -marginimage (or mainwidget for cb/rb)
      # cb/rb can also have an empty key in wkeylist
      if { ! $hidemargin } {
        if { $type ne "separator" && $type ne "tearoff" } {
          set wkey [lindex $wkeylist 0]
          if { $wkey ne {} } {
            set w [dict get $info $wkey]
            if { $w ne {} } {
              $w configure -cursor [dict get $vars(items) $dictkey -cursor]
              lappend wlist $w
              incr colidx
            } elseif { $vars(-type) eq "submenu" } {
              lappend wlist {}
              incr colidx
            }
          } elseif { $vars(-type) eq "submenu" } {
            lappend wlist {}
            incr colidx
          }
        }
      } else {
        # with -hidemargin, the first column is removed
        # marginimage
        if { $hidemarginmixed } {
          set cspan [expr {max(1,$cspan+1)}]
        } else {
          set cspan [expr {max(1,$cspan-1)}]
        }
      }

      # main label or widget
      set wkey [lindex $wkeylist 1]
      if { $wkey ne {} } {
        set w [dict get $vars(items) $dictkey $wkey]
        if { $w ne {} } {
          $w configure -cursor [dict get $vars(items) $dictkey -cursor]
          lappend wlist $w
          set ismapped [winfo ismapped $w]
          if { ! $ismapped } {
            bind $w <Map> [list [self] setconfigure map %W $dictkey]
            incr vars(config.count) 1
          }
          if { (! $ismapped && ! $vars(withdrawn)) ||
              [dict get $vars(items) $dictkey changed] } {
            bind $w <Configure> [list [self] setconfigure conf %W $dictkey]
            incr vars(config.count) 1
          }
        }
      }

      # acceleratorwidget
      set w [dict get $vars(items) $dictkey acceleratorwidget]
      if { $w ne {} &&
          $vars(-type) eq "submenu" &&
          ! $vars(-hideaccelerators) } {
        $w configure -cursor [dict get $vars(items) $dictkey -cursor]
        lappend wlist $w
      } elseif { $haverightlabel } {
        if { $type ne "separator" && $type ne "tearoff" } {
          lappend wlist {}
        }
      }

      # rightlabel
      if { $haverightlabel } {
        if { $vars(-type) ne "menubar" && [llength $wkeylist] >= 3 } {
          set w [dict get $vars(items) $dictkey [lindex $wkeylist 2]]
          if { $w ne {} } {
            $w configure -cursor [dict get $vars(items) $dictkey -cursor]
            lappend wlist $w
          }
        } else {
          set cspan [expr {max(1,$cspan-1)}]
        }
      } else {
        if { ! $hidemarginmixed } {
          set cspan [expr {max(1,$cspan-1)}]
        }
      }

      if { $gap } {
        # create an empty column that stretches
        grid columnconfigure $vars(packframe) $startcol -weight 10
        incr startcol
      }
      # for re-configuration of the menu
      grid columnconfigure $vars(packframe) $startcol -weight 0

      set ccount [llength $wlist]
      set maxcol [expr {max($maxcol,$ccount)}]
      dict set vars(items) $dictkey startcol $startcol
      dict set vars(items) $dictkey columncount $ccount
      dict set vars(items) $dictkey row $row
      if { $hidemarginmixed } {
        grid columnconfigure $vars(packframe) 1 -weight 1
      }
      my _packitem $dictkey $wlist $startcol $row $cspan $colidx

      incr row
      incr itemno
    }

    set vars(postprocess) false
    set vars(confchanged) false
  }

  method _unpostMenu { {nounpostflag {}} } {
    my variable vars

    if { ! [winfo exists $vars(itemframe)] } {
      return
    }

    # no more menu, release grab
    if { [winfo exists $vars(itemframe)] } {
      grab release $vars(itemframe)
    }
    set vars(closecheck.active) false
    if { $vars(-bindaccelerators) } {
      my _bindAcceleratorKeys unbind
    }

    if { $vars(-type) eq "submenu" } {
      if { $vars(-mode) eq "toplevel" } {
        wm withdraw $vars(itemframe)
      } else {
        # -mode frame
        place forget $vars(itemframe)
      }
      set vars(withdrawn) true
      if { $vars(parent) ne {} } {
        $vars(parent) unpostcascade
      }
      # leave parent set
      set vars(menu.posted) false
      set vars(menu.active) false
      set vars(dictkey.prior) -1
      set vars(active.dictkey) -1
      set vars(active.itemno) -1
      set vars(itemno.prior) -1
      set vars(isconfigured) 0
    } else {
      set vars(cascade.posted) [lrange $vars(cascade.posted) 1 end]
    }
  }

  method disableSkip { } {
    my variable vars

    set vars(closecheck.skip) false
  }

  method _postMenu { {x {}} {y {}} {parent {}} } {
    my variable vars

    if { $vars(menu.active) } {
      return
    }

    if { $vars(-postcommand) ne {} } {
      try {
        {*}$vars(-postcommand)
      } on error {err res} {
        puts "-postcommand: $res"
      }
    }

    set vars(parent) $parent

    my _updateButtonDisplay
    my _layoutmenu

    if { $vars(-type) eq "submenu" } {
      set bw [::flexmenucmd::topixels 1]

      if { $vars(-mode) eq "toplevel" } {
        set geom {}
        append geom +$x
        incr y $bw
        append geom +$y
        # set the geometry ahead of time in case the wm supports that
        wm geometry $vars(itemframe) $geom
        wm deiconify $vars(itemframe)
        focus $vars(itemframe)
        after idle [list wm geometry $vars(itemframe) $geom]
        raise $vars(itemframe)
      } else {
        # -mode frame
        set tw [winfo toplevel $vars(itemframe)]
        set w $tw
        set tx [winfo rootx $w]
        set ty [winfo rooty $w]
        incr x [expr {-1*$tx}]
        incr y [expr {-1*$ty}]
        incr y $bw
        place $vars(itemframe) -in $tw \
            -relx 0.0 -rely 0.0 \
            -x $x -y $y
        raise $vars(itemframe)
      }
      set vars(withdrawn) false

      # check for mouse clicks outside of the menu
      # torn off menus have no close check.
      if { ! $vars(bind.closecheck) && ! $vars(-tornoff) } {
        bind all <ButtonRelease-1> +[list [self] closeCheck release %W]
        set vars(bind.closecheck) true
      }
      if { [winfo exists $vars(itemframe)] &&
          $vars(-type) eq "submenu" } {
        # the button release will generate a mouse.click.out event.
        # skip the next closecheck.
        # this is really annoying...
        set vars(closecheck.skip) true
        after 500 [list [self] disableSkip]
        set vars(closecheck.active) true
      }
    }

    # mouse motion
    if { ! $vars(bind.mousehandler) } {
      bind all <Motion> +[list [self] mouseHandler %W %X %Y]
      set vars(bind.mousehandler) true

      bind $vars(itemframe) <Enter> +[list [self] enterLeaveHandler %W enter %m]
      bind $vars(itemframe) <Leave> +[list [self] enterLeaveHandler %W leave %m]
    }

    set vars(menu.posted) true

    if { $vars(parent) ne {} } {
      my _setAction menu.post
      my _processState
    }
    if { $vars(-type) ne "submenu" &&
        $vars(-bindaccelerators) } {
      my _bindAcceleratorKeys bind
    }
  }

  # menu activation and bindings

  method _bindAcceleratorKeys { flag } {
    my variable vars

    set tag all
    if { $vars(-type) eq "submenu" } {
      set tag $vars(itemframe)
    }
    for {set itemno 0} {$itemno < $vars(item.count)} {incr itemno} {
      set dictkey [lindex $vars(item.map) $itemno]
      set keymap [dict get $vars(items) $dictkey keymap]
      foreach {k v} $keymap {
        if { $k eq {} } {
          continue
        }
        if { $flag eq "unbind" } { set v {} }
        bind all <$k> $v
      }
    }
  }

  method _bindKeys { } {
    my variable vars

    if { $vars(menu.binds) } {
      return
    }
    set vars(menu.binds) true

    if { $vars(-type) ne "menubar" } {
      bind all <<PrevLine>> +[list [self] upDownHandler -1 prevline]
      bind all <Up> +[list [self] upDownHandler -1 prevline]
    }
    bind all <<NextLine>> +[list [self] upDownHandler 1 nextline]
    bind all <Down> +[list [self] upDownHandler 1 nextline]

    bind all <<NextChar>> +[list [self] rightLeftHandler 1 nc]
    bind all <Right> +[list [self] rightLeftHandler 1 right]
    if { $vars(-type) ne "menuleft" } {
      bind all <<PrevChar>> +[list [self] rightLeftHandler -1 pc]
      bind all <Left> +[list [self] rightLeftHandler -1 left]
    }

    bind all <space> +[list [self] keyinvoke %W]
    bind all <Return> +[list [self] keyinvoke %W]
    bind all <KP_Enter> +[list [self] keyinvoke %W]
    bind all <Escape> +[list [self] closemenu]
  }

  method _deactivateMenu { } {
    my variable vars

    set vars(keysym) $vars(keysym.meta)
    set vars(menu.active) false
  }

  method _activateMenu { } {
    my variable vars

    if { $vars(menu.active) } {
      return
    }
    if { $vars(activatemenu.in) } {
      return
    }
    set vars(activatemenu.in) true

    set vars(keysym) $vars(keysym.meta)
    my _bindKeys
    if { $vars(-type) eq "submenu" &&
        $vars(-bindaccelerators) } {
      my _bindAcceleratorKeys bind
    }

    if { $vars(parent) ne {} } {
      # the parent will no longer own the grab
      $vars(parent) releasegrab
    }

    if { [winfo exists $vars(itemframe)] &&
        [winfo viewable $vars(itemframe)] &&
        $vars(-type) eq "submenu" &&
        ! $vars(-tornoff) } {
      # menu is activated, get the grab
      grab set -global $vars(itemframe)
      set vars(closecheck.active) true
    }

    set vars(menu.active) true
    set vars(activatemenu.in) false
  }

  # internal methods for communication between menus

  method releasegrab { } {
    my variable vars

    if { [winfo exists $vars(itemframe)] } {
      grab release $vars(itemframe)
    }
    set vars(closecheck.active) false
  }

  method unpostcascade { } {
    my variable vars

    set vars(cascade.posted) [lrange $vars(cascade.posted) 1 end]
  }

  method close { {intflag {}} } {
    my variable vars

    if { $vars(-tornoff) && $intflag eq "-internal" } {
      return
    }
    my _setAction mouse.click.out
    my _processState
  }

  method activatemenu { } {
    my variable vars

    my _setAction menu.enter
    my _processState
  }

  method updatebuttondisplay { } {
    my variable vars

    my _updateButtonDisplay
  }

  method setskipfalse { } {
    my variable vars

    set vars(key.skip) false
  }

  method selectfirst { keyskip } {
    my variable vars

    if { $keyskip } {
      if { $vars(active.dictkey) == -1 } {
        set vars(key.skip) true
        after 100 [list [self] setskipfalse]
      }
    }
    set vars(dictkey.new) [lindex $vars(item.map) 0]
    set vars(itemno.new) 0
    my _setAction key.move
    my _processState
  }

  # item activation and mouse handling

  method pressHandler { w } {
    my variable vars

    if { [winfo exists $w] } {
      my _setAction menu.enter
      my _processState
      lassign [winfo pointerxy $vars(itemframe)] x y
      if { $vars(-stickycascade) || $vars(-clicktoopen) } {
        if { $vars(-type) eq "submenu" ||
            ($vars(-type) ne "submenu" && [my _getState] ne "active") } {
          set vars(sticky.mode) true
        }
      }
      my mouseHandler $w $x $y true
    }
  }

  method _leaveMenu { } {
    my variable vars

    if { ! $vars(menu.active) } {
      return
    }
    # grab is left intact
    my _setAction menu.leave
    my _processState
  }

  method _enterMenu { } {
    my variable vars

    if { $vars(menu.active) } {
      return
    }
    if { [winfo exists $vars(itemframe)] &&
        [winfo viewable $vars(itemframe)] &&
        $vars(-type) eq "submenu" &&
        ! $vars(-tornoff) } {
      # when a menu is entered, get the grab
      grab set -global $vars(itemframe)
      set vars(closecheck.active) true
    }
    my _setAction menu.enter
    my _processState
  }

  method enterLeaveHandler { w type mode } {
    my variable vars

    if { $vars(itemframe) ne $w } {
      return -code ok
    }

    if { ! $vars(menu.posted) } {
      return -code ok
    }

    if { ! [winfo exists $w] } {
      if { $vars(postprocess) } {
        my _deactivateItem
      }
      return -code ok
    }

    if { $vars(postprocess.in) } {
      return -code ok
    }

    if { ! $vars(postprocess) } {
      my waitpostprocess [list my enterLeaveHandler $w $type $mode]
      return -code ok
    }

    if { $vars(activatemenu.in) } {
      return -code ok
    }

    if { $mode ne "NotifyNormal" } {
      return -code ok
    }

    if { $type eq "leave" } {
      my _leaveMenu
    }

    if { $type eq "enter" } {
      my _enterMenu
    }
    return -code ok
  }

  method mouseHandler { w x y {pressflag false} } {
    my variable vars

    if { ! $vars(menu.posted) } {
      return -code ok
    }

    if { ! [winfo exists $w] } {
      if { $vars(postprocess) } {
        my _deactivateItem
      }
      return -code ok
    }

    if { $vars(postprocess.in) } {
      return -code ok
    }

    if { ! $vars(postprocess) } {
      my waitpostprocess [list [self] mouseHandler $w $x $y $pressflag]
      return -code ok
    }

    if { $vars(activatemenu.in) } {
      return -code ok
    }

    set if $vars(itemframe)
    set pf $vars(packframe)
    if { ! [winfo exists $vars(packframe)] } {
      return -code ok
    }

    set bxl [winfo rootx $if]
    set byt [winfo rooty $if]
    set bxr [expr {$bxl+[winfo width $if]}]
    set byb [expr {$byt+[winfo height $if]}]

    # When a menu has the grab, the enter/leave events are not
    # triggered, so check here also.

    # Only invoke entermenu if the mouse is actually within
    # this menu.
    set match false
    lassign [winfo pointerxy $w] x y
    set cw [winfo containing $x $y]
    while { $cw ne "." && $cw ne {} } {
      if { $cw eq $vars(itemframe) } {
        set match true
        break
      }
      set cw [winfo parent $cw]
    }

    if { $x >= [expr {$bxl}] && $x <= [expr {$bxr}] &&
        $y >= [expr {$byt}] && $y <= [expr {$byb}] } {
      if { $match && ! $vars(menu.active) } {
        my _enterMenu
      }
    } else {
      if { $vars(menu.active) } {
        my _leaveMenu
      }
    }

    set bxl [winfo rootx $pf]
    set byt [winfo rooty $pf]
    set bxr [expr {$bxl+[winfo width $pf]}]
    set byb [expr {$byt+[winfo height $pf]}]

    foreach {dictkey} $vars(item.map) {
      set th [dict get $vars(items) $dictkey y.top]
      set bh [dict get $vars(items) $dictkey y.bottom]
      set tw [dict get $vars(items) $dictkey x.left]
      set bw [dict get $vars(items) $dictkey x.right]
      if { $x >= [expr {$tw+$bxl}] && $x <= [expr {$bw+$bxl}] &&
          $y >= [expr {$th+$byt}] && $y <= [expr {$bh+$byt}] } {
        if { $dictkey == $vars(active.dictkey) } {
          return -code ok
        }

        set action mouse.move
        if { $vars(sticky.mode) || $vars(-clicktoopen) } {
          # the button release will invoke the cascade.
          set action mouse.move.sticky
        }
        set vars(dictkey.new) $dictkey
        set vars(itemno.new) -1
        my _setAction $action
        my _processState
        break
      }
    }
    return -code ok
  }

  method movecheck { w x y } {
    my variable vars

    if { $vars(last.x) eq {} } {
      set vars(last.x) $x
      set vars(last.y) $y
      return
    }

    if { $w eq $vars(itemframe) &&
        $x != $vars(last.x) || $y != $vars(last.y) &&
        $vars(dictkey.prior) != -1 } {
      set menu [dict get $vars(items) $vars(dictkey.prior) -menu]
      if { $menu ne {} } {
        set geom {}
        append geom +$x
        append geom +$y
        wm geometry $menu $geom
      }
      set vars(last.x) $x
      set vars(last.y) $y
      set vars(postprocess) false
    }
  }

  method closeCheck { type w } {
    my variable vars

    if { ! [info exists vars(itemframe)] } {
      return -code ok
    }
    if { ! $vars(menu.posted) } {
      return -code ok
    }
    if { ! $vars(closecheck.active) } {
      return -code ok
    }
    if { $vars(closecheck.skip) } {
      set vars(closecheck.skip) false
      return -code ok
    }

    set if $vars(itemframe)
    if { ! [winfo exists $if] } {
      return -code ok
    }
    if { ! [winfo viewable $if] } {
      return -code ok
    }

    set val 0

    # this checks for widgets that expand beyond the menu's confines
    # such as a combobox.  A click in a combobox outside the menu
    # should not cause a menu event.
    set cw $w
    set tw [winfo toplevel $vars(itemframe)]
    while { $cw ne $tw && $cw ne {} } {
      if { $cw in $vars(widgetlist) } {
        return
      }
      set cw [winfo parent $cw]
    }

    if { [my _inWindow $vars(itemframe)] } {
      incr val
    }

    if { $val == 0 } {
      my _setAction mouse.click.out
      my _processState
      if { $vars(parent) ne {} } {
        $vars(parent) releasegrab
      }
      if { [winfo exists $vars(itemframe)] } {
        grab release $vars(itemframe)
      }
      set vars(closecheck.active) false
    }
    return -code ok
  }

  # item activation

  method _deactivateItem { } {
    my variable vars

    set vars(keysym) $vars(keysym.meta)

    if { $vars(active.dictkey) == -1 } {
      return
    }
    if { ! [winfo exists $vars(itemframe)] } {
      return
    }

    if { $vars(active.dictkey) != -1 &&
        ! [dict exists $vars(items) $vars(active.dictkey)] } {
      return
    }
    if { $vars(active.dictkey) != -1 &&
        [dict get $vars(items) $vars(active.dictkey) -state] ne "active" } {
      return
    }

    # deactivate
    dict set vars(items) $vars(active.dictkey) -state normal

    if { $vars(active.dictkey) != -1 } {
      set wlist [dict get $vars(items) $vars(active.dictkey) widgetlist]

      foreach {ch} $wlist {
        if { [info commands $ch] ne {} } {
          if { [catch {$ch state {!user1 !active}}] } {
            catch {$ch configure -state {!user1 !active}}
          }
        }
      }
      lower $vars(activeframe) ; # for mac os x
      place forget $vars(activeframe)
      set vars(dictkey.prior) $vars(active.dictkey)
      set vars(itemno.prior) $vars(active.itemno)
    }

    set vars(active.dictkey) -1
    set vars(active.itemno) -1
  }

  method _skipItem { dictkey } {
    my variable vars

    set rc false
    if { $dictkey eq {} ||
        [dict get $vars(items) $dictkey -state] eq "disabled" } {
      set rc true
    }
    return $rc
  }

  method _activateItem { } {
    my variable vars

    if { ! $vars(postprocess) || $vars(activatemenu.in) } {
      my waitpostprocess [list my _activateItem]
      return
    }

    if { ! $vars(menu.active) || $vars(active.dictkey) == -1 } {
      return false
    }

    if { [my _skipItem $vars(active.dictkey)] } {
      return false
    }

    # activate
    set wlist [dict get $vars(items) $vars(active.dictkey) widgetlist]
    foreach {ch} $wlist {
      if { [info commands $ch] ne {} } {
        if { [catch {$ch state {user1 active}}] } {
          catch {$ch configure -state {user1 active}}
        }
      }
    }

    foreach {nm k} {abg -activebackground ar -activerelief
        abw -activeborderwidth} {
      set $nm $vars($k)
      if { [dict exists $vars(items) $vars(active.dictkey) $k] } {
        set $nm [dict get $vars(items) $vars(active.dictkey) $k]
      }
    }

    $vars(activeframe) configure \
        -cursor [dict get $vars(items) $vars(active.dictkey) -cursor] \
        -background $abg \
        -relief $ar \
        -borderwidth $abw
    $vars(activeframe) configure -width \
        [dict get $vars(items) $vars(active.dictkey) width]
    $vars(activeframe) configure -height \
        [dict get $vars(items) $vars(active.dictkey) height]
    set tx [dict get $vars(items) $vars(active.dictkey) x.left]
    set ty [dict get $vars(items) $vars(active.dictkey) y.top]
    if { $vars(-maxheight) != 0 } {
      set bw [$vars(itemframe) cget -borderwidth]
      set bw [::flexmenucmd::topixels $bw]
      incr ty $bw
    }
    place $vars(activeframe) -in $vars(packframe) \
        -relx 0.0 -rely 0.0 \
        -x $tx -y $ty
    lower $vars(activeframe)
    if { [dict exists $vars(items) $vars(active.dictkey) backgroundframe] } {
      lower [dict get $vars(items) $vars(active.dictkey) backgroundframe]
    }
    if { $vars(active.itemno) == -1 } {
      set vars(active.itemno) [lsearch -exact -integer $vars(item.map) $vars(active.dictkey)]
    }
    dict set vars(items) $vars(active.dictkey) -state active

    if { $vars(active.dictkey) != $vars(active.dictkey.last) } {
      event generate $vars(itemframe) <<MenuSelect>>
      set vars(active.dictkey.last) $vars(active.dictkey)
    }

    return true
  }

  # invocation handlers

  method _updateButtonDisplay { } {
    my variable vars

    set bconfchanged false
    foreach {dictkey} $vars(item.map) {
      set type [dict get $vars(items) $dictkey type]
      if { $type ne "radiobutton" && $type ne "checkbutton" } {
        continue
      }

      set button [dict get $vars(items) $dictkey mainwidget]
      set lab [dict get $vars(items) $dictkey label]
      if { $button ne {} && $lab ne {} } {
        set oimg [$button cget -image]
        set img [dict get $vars(items) $dictkey -image]
        set si [dict get $vars(items) $dictkey -selectimage]
        if { $img ne {} && $si ne {} } {
          set var [$button cget -variable]
          set sel false
          try {
            set sel [set $var]
          } on error {err res} {
          }
          if { $type eq "checkbutton" } {
            set arg -onvalue
          }
          if { $type eq "radiobutton" } {
            set arg -value
          }
          if { $sel eq [$button cget $arg] } {
            set image $si
          } else {
            set image $img
          }
          if { $image ne $oimg } {
            $lab configure -image $image
            set bconfchanged true
          }
        } ; # if both image and select image are specified
      } ; # if there is a label
    } ; # for each item

    if { $bconfchanged } {
      set vars(confchanged) true
      my _layoutmenu
    }

    if { $vars(parent) ne {} } {
      $vars(parent) updatebuttondisplay
    }
  }

  method execpostcascade { dictkey } {
    my variable vars

    set itemno [lsearch -exact -integer $vars(item.map) $dictkey]
    if { $itemno != -1 } {
      set vars(dictkey.new) $dictkey
      set vars(itemno.new) $itemno
      my _setAction mouse.invoke
      my _processState
    }
  }

  method execbutton { dictkey {invokeflag {}} } {
    my variable vars

    if { [dict get $vars(items) $dictkey -state] eq "disabled" } {
      return
    }

    set button [dict get $vars(items) $dictkey mainwidget]
    if { $button eq {} } {
      return
    }

    # the standard checkbutton/radiobutton bindings will trigger,
    # don't invoke twice.
    if { $invokeflag eq "-invoke" } {
      $button invoke
    }

    my _updateButtonDisplay
  }

  method execprecmd { dictkey } {
    my execcmd $dictkey -precommand
  }

  method execcmd { dictkey {which -command} args } {
    my variable vars

    if { ! [dict exists $vars(items) $dictkey] } {
      return
    }
    if { [dict get $vars(items) $dictkey -state] eq "disabled" } {
      return
    }

    set c [dict get $vars(items) $dictkey $which]
    if { $c eq {} } {
      return
    }

    try {
      {*}$c {*}$args
    } on error {err res} {
      puts "flexmenu: execcmd: $res"
    }
  }

  method scroll { d } {
    my variable vars

    my yview $vars(textframe) scroll $d units
  }

  # item commands and configuration

  method _iteminit { type idx } {
    my variable vars

    set dictkey [::flexmenucmd::nextkey]

    if { $idx eq "end" } {
      set itemno $vars(item.count)
      lappend vars(item.map) $dictkey
    } else {
      set itemno $idx
      if { $idx == 0 } {
        set vars(item.map) [list $dictkey {*}$vars(item.map)]
      } else {
        set vars(item.map) [concat [lrange $vars(item.map) 0 $idx-1] \
            $dictkey [lrange $vars(item.map) $idx end]]
      }
    }
    incr vars(item.count)
    dict set vars(items) $dictkey type $type

    dict set vars(items) $dictkey -accelerator {}
    dict set vars(items) $dictkey acceleratorwidget {}
    dict set vars(items) $dictkey -background $vars(-background)
    dict set vars(items) $dictkey -borderwidth 0
    dict set vars(items) $dictkey changed true
    dict set vars(items) $dictkey -columnbreak false
    dict set vars(items) $dictkey -command {}
    dict set vars(items) $dictkey -cursor {}
    dict set vars(items) $dictkey -font $vars(-font)
    dict set vars(items) $dictkey -foreground $vars(-foreground)
    dict set vars(items) $dictkey -gap false
    dict set vars(items) $dictkey -hidemargin false
    dict set vars(items) $dictkey -image {}
    dict set vars(items) $dictkey -indicatoron true
    dict set vars(items) $dictkey keymap {}
    dict set vars(items) $dictkey label {}
    dict set vars(items) $dictkey mainwidget {}
    dict set vars(items) $dictkey -marginimage {}
    dict set vars(items) $dictkey -menu {}
    dict set vars(items) $dictkey -precommand {}
    dict set vars(items) $dictkey -relief flat
    dict set vars(items) $dictkey rightlabel {}
    dict set vars(items) $dictkey -selectimage {}
    dict set vars(items) $dictkey -state normal
    dict set vars(items) $dictkey -style {}
    dict set vars(items) $dictkey -text {}
    dict set vars(items) $dictkey -widget {}
    dict set vars(items) $dictkey widgetlist {}
    dict set vars(items) $dictkey wkeylist {}
    dict set vars(items) $dictkey -xalign menuright
    dict set vars(items) $dictkey -yalign activetop
    dict set vars(items) $dictkey -keepopen false

    return $dictkey
  }

  method _mkacclab { dictkey txt } {
    my variable vars

    set lab [dict get $vars(items) $dictkey acceleratorwidget]
    if { $lab ne {} } {
      $lab configure -text $txt
    } else {
      set lab [::ttk::label $vars(packframe).$dictkey-a \
          -takefocus 0 \
          -text $txt \
          -anchor w \
          -style Accelerator.Flexmenu.TLabel]
    }
    dict set vars(items) $dictkey acceleratorwidget $lab
    return $lab
  }

  method _mkkeyname { k } {
    my variable vars

    regsub Ctrl $k Control k
    regsub \u2318 $k Command- k
    set op tolower
    if { [regexp Shift $k] } {
      set op toupper
    }
    if { [regexp {(.*)-(\w+)} $k all prefix sfx] } {
      set k "$prefix-Key-[string $op $sfx]"
    }
    return $k
  }

  method _confmarginimage { dictkey k v } {
    my variable vars

    set mi [dict get $vars(items) $dictkey $k]
    if { $mi ne {} } {
      destroy $mi
    }
    set mi [::ttk::label $vars(packframe).$dictkey-mi \
        -image $v \
        -takefocus 0 \
        -padding 0 \
        -justify center \
        -anchor center \
        -compound image \
        -style $vars(-style)Flexmenu.TLabel]
    dict set vars(items) $dictkey $k $mi
    return $mi
  }

  method _menuLabel {str {char &}} {
    set curr -1
    while 1 {
      set curr [string first $char $str $curr]
      if { $curr == -1 } {
        break
      } elseif { [string index $str $curr+1] ne $char } {
        break
      } else {
        incr curr
        set str [string replace $str $curr $curr]
      }
    }
    return [list [string replace $str $curr $curr] -underline $curr];
  }

  method _conftext { dictkey lab k v } {
    my variable vars

    set uargs [lassign [my _menuLabel $v] v]
    lassign $uargs underlinek underlinev
    $lab configure $k $v
    set knm {}
    if { $underlinev != -1 } {
      # the underline value will be processed later to generate
      # the accelerator.
      $lab configure $underlinek $underlinev
    }
    dict set vars(items) $dictkey $k $v
    return $knm
  }

  method _confunderline { dictkey lab script } {
    my variable vars

    if { [dict exists vars(items) $dictkey -accelerator] &&
        [dict get vars(items) $dictkey -accelerator] ne {} } {
      # if the user has defined an accelerator, skip the -underline value
      return
    }

    set v [$lab cget -underline]
    if { $v ne {} } {
      set char [string index [$lab cget -text] $v]
      if { $char ne {} } {
        set knm ${vars(-acceleratorprefix)}$char
        set acclab [my _mkacclab $dictkey $knm]
        set knm [my _mkkeyname $knm]
        dict lappend vars(items) $dictkey keymap \
            [list $knm [list [self] keyinvoke %W $dictkey]]
        bind $acclab <ButtonRelease-1> [list [self] mouseinvoke %W $dictkey]
        bind $acclab <ButtonPress-1> [list [self] pressHandler %W]
      }
    }
  }

  method _confstate { dictkey wkeylist } {
    my variable vars

    set state [dict get $vars(items) $dictkey -state]

    set wdashstate $state
    set wstate $state
    if { $state ne "disabled" } {
      set wdashstate {!disabled !active}
      set wstate normal
    }

    set wlist [dict get $vars(items) $dictkey acceleratorwidget]
    lappend wlist [dict get $vars(items) $dictkey -marginimage]
    lappend wlist [dict get $vars(items) $dictkey rightlabel]
    foreach {wkey} $wkeylist {
      if { $wkey ne {} } {
        lappend wlist [dict get $vars(items) $dictkey $wkey]
      }
    }

    foreach {w} $wlist {
      if { $w eq {} } {
        continue
      }
      try {
        if { $state ne "active" } {
          # there's no simple way to determine if the widget
          # uses 'configure -state' or the state command
          # or if it supports a state at all.
          if { ! [regexp TLabel $s] } {
            # setting a label to disable on mac os x changes its
            # background color.
            try {
              $w configure -state $wdashstate
            } on error {} {
              try {
                $w state $wstate
              } on error {} {}
            }
          }
        }
      } on error {err res} {
        if { $state ne "active" } {
          try {
            $w configure -state $wdashstate
          } on error {err res} {
            try {
              $w state $wstate
            } on error {err res} {
            }
          }
        } ; # state not active
      } ; # unable to fetch style
    } ; # foreach widget
  }

  method _createlabel { dictkey } {
    my variable vars

    set lab [::ttk::label $vars(packframe).$dictkey-l \
        -text {} \
        -anchor w \
        -style $vars(-style)Flexmenu.TLabel]
    dict set vars(items) $dictkey label $lab
    return $lab
  }

  method _confcascade { dictkey args } {
    my variable vars

    set lab [dict get $vars(items) $dictkey label]
    set nstyle [$lab cget -style]

    foreach {k v} $args {
      switch -exact -- $k {
        -bitmap -
        -indicatoron -
        -selectcolor -
        -selectimage -
        -widget {
          # most of these are here simply to make the cloning process easier
          ;
        }
        -text {
          my _conftext $dictkey $lab $k $v
        }
        -marginimage {
          my _confmarginimage $dictkey $k $v
        }
        -borderwidth -
        -relief {
          dict set vars(items) $dictkey $k $v
          # the relief is only used for the background frame
          if { $v eq "flat" } {
            dict set vars(items) $dictkey -borderwidth 0
          }
        }
        -activebackground -
        -activeforeground -
        -font -
        -background -
        -foreground {
          dict set vars(items) $dictkey $k $v

          set nstyle $dictkey.Flexmenu.TLabel
          ::ttk::style configure $nstyle $k $v

          set vars(postprocess) false
        }
        -keepopen -
        -columnbreak -
        -command -
        -cursor -
        -gap -
        -hidemargin -
        -menu -
        -precommand -
        -state -
        -xalign -
        -yalign {
          dict set vars(items) $dictkey $k $v
        }
        -accel -
        -accelerator {
          dict set vars(items) $dictkey -accelerator $v
          set acclab [my _mkacclab $dictkey $v]
          set knm [my _mkkeyname $v]
          dict lappend vars(items) $dictkey keymap \
              [list $knm [list [self] keyinvoke %W $dictkey]]
          bind $acclab <ButtonRelease-1> [list [self] mouseinvoke %W $dictkey]
          bind $acclab <ButtonPress-1> [list [self] pressHandler %W]
        }
        default {
          $lab configure $k $v
        }
      }
    }

    my _confunderline $dictkey $lab [list [self] mouseinvoke %W $dictkey]

    foreach {k} {-marginimage label acceleratorwidget rightlabel} {
      set item [dict get $vars(items) $dictkey $k]
      if { $item ne {} } {
        $item configure -style $nstyle
      }
    }

    my _confstate $dictkey [list label rightlabel]
  }

  method _cascade { idx args } {
    my variable vars

    set dictkey [my _iteminit cascade $idx]

    set ralab [::ttk::label $vars(packframe).$dictkey-ral \
        -text $vars(-rightarrowchar) \
        -anchor e \
        -style $vars(-style)Flexmenu.TLabel]
    dict set vars(items) $dictkey rightlabel $ralab
    bind $ralab <ButtonRelease-1> [list [self] execpostcascade $dictkey]
    bind $ralab <ButtonPress-1> [list [self] pressHandler %W]
    set lab [my _createlabel $dictkey]
    bind $lab <ButtonRelease-1> [list [self] execpostcascade $dictkey]
    bind $lab <ButtonPress-1> [list [self] pressHandler %W]

    my _confcascade $dictkey {*}$args

    set mi [dict get $vars(items) $dictkey -marginimage]
    if { $mi ne {} } {
      bind $mi <ButtonRelease-1> [list [self] execpostcascade $dictkey]
      bind $mi <ButtonPress-1> [list [self] pressHandler %W]
    }
    dict set vars(items) $dictkey wkeylist [list -marginimage label rightlabel]
    dict set vars(items) $dictkey cspan 1
  }

  method _confcheckbutton { dictkey args } {
    my variable vars

    set cb [dict get $vars(items) $dictkey mainwidget]
    set lab [dict get $vars(items) $dictkey label]
    set nstyle [$lab cget -style]
    set ncbstyle [$cb cget -style]

    foreach {k v} $args {
      switch -exact -- $k {
        -bitmap -
        -command -
        -marginimage -
        -menu -
        -precommand -
        -selectcolor -
        -widget -
        -xalign -
        -yalign {
          # most of these are here simply to make the cloning process easier
          ;
        }
        -text {
          my _conftext $dictkey $lab $k $v
        }
        -borderwidth -
        -relief {
          dict set vars(items) $dictkey $k $v
          # the relief is only used for the background frame
        }
        -activebackground -
        -activeforeground -
        -font -
        -background -
        -foreground {
          dict set vars(items) $dictkey $k $v

          set nstyle $dictkey.Flexmenu.TLabel
          ::ttk::style configure $nstyle $k $v

          set ncbstyle $dictkey.Flexmenu.TCheckbutton
          ::ttk::style configure $ncbstyle $k $v

          set vars(postprocess) false
        }
        -keepopen -
        -state -
        -columnbreak -
        -indicatoron -
        -selectimage -
        -cursor -
        -gap -
        -hidemargin {
          dict set vars(items) $dictkey $k $v
        }
        -image {
          dict set vars(items) $dictkey $k $v
          $lab configure $k $v
        }
        -accel -
        -accelerator {
          dict set vars(items) $dictkey -accelerator $v
          set acclab [my _mkacclab $dictkey $v]
          set knm [my _mkkeyname $v]
          dict lappend vars(items) $dictkey keymap \
              [list $knm [list [self] keyinvoke %W $dictkey]]
          bind $acclab <ButtonRelease-1> [list [self] mouseinvoke %W $dictkey]
          bind $acclab <ButtonPress-1> [list [self] pressHandler %W]
        }
        -compound {
          $lab configure $k $v
        }
        default {
          $cb configure $k $v
        }
      }
    }

    my _confunderline $dictkey $lab [list [self] mouseinvoke %W $dictkey]

    foreach {k} {-marginimage label acceleratorwidget} {
      set item [dict get $vars(items) $dictkey $k]
      if { $item ne {} } {
        $item configure -style $nstyle
      }
    }
    set cb [dict get $vars(items) $dictkey mainwidget]
    $cb configure -style $ncbstyle

    set wkeylist [list]
    if { [dict get $vars(items) $dictkey -indicatoron] } {
      lappend wkeylist mainwidget
    } else {
      lappend wkeylist {}
    }
    lappend wkeylist label
    dict set vars(items) $dictkey wkeylist $wkeylist
    my _confstate $dictkey $wkeylist
  }

  method _checkbutton { idx args } {
    my variable vars

    set dictkey [my _iteminit checkbutton $idx]

    set cb [::ttk::checkbutton $vars(packframe).$dictkey-cb \
        -style Flexmenu.TCheckbutton \
        -takefocus 0]
    dict set vars(items) $dictkey normalstyle Flexmenu.TCheckbutton
    dict set vars(items) $dictkey disabledstyle Disabled.Flexmenu.TCheckbutton
    dict set vars(items) $dictkey activestyle Active.Flexmenu.TCheckbutton
    bind $cb <ButtonRelease-1> [list [self] keyinvoke %W $dictkey]
    bind $cb <ButtonPress-1> [list [self] pressHandler %W]
    bind $cb <Key-space> [list [self] keyinvoke %W $dictkey]
    dict set vars(items) $dictkey mainwidget $cb

    set lab [my _createlabel $dictkey]
    bind $lab <ButtonRelease-1> [list [self] mouseinvoke %W $dictkey]
    bind $lab <ButtonPress-1> [list [self] pressHandler %W]

    my _confcheckbutton $dictkey {*}$args
    dict set vars(items) $dictkey cspan 1
  }

  method _confcommand { dictkey args } {
    my variable vars

    set lab [dict get $vars(items) $dictkey label]
    set nstyle [$lab cget -style]

    foreach {k v} $args {
      switch -exact -- $k {
        -bitmap -
        -indicatoron -
        -menu -
        -precommand -
        -selectcolor -
        -selectimage -
        -widget -
        -xalign -
        -yalign {
          # most of these are here simply to make the cloning process easier
          ;
        }
        -text {
          my _conftext $dictkey $lab $k $v
        }
        -marginimage {
          if { $v ne {} } {
            my _confmarginimage $dictkey $k $v
          }
        }
        -borderwidth -
        -relief {
          dict set vars(items) $dictkey $k $v
          # the relief is only used for the background frame
        }
        -font -
        -activebackground -
        -activeforeground -
        -background -
        -foreground {
          dict set vars(items) $dictkey $k $v

          set nstyle $dictkey.Flexmenu.TLabel
          my _confStyling $nstyle [list $k $v] $dictkey
          set vars(postprocess) false
        }
        -keepopen -
        -command -
        -state -
        -columnbreak -
        -cursor -
        -gap -
        -hidemargin {
          dict set vars(items) $dictkey $k $v
        }
        -accel -
        -accelerator {
          dict set vars(items) $dictkey -accelerator $v
          set acclab [my _mkacclab $dictkey $v]
          set knm [my _mkkeyname $v]
          dict lappend vars(items) $dictkey keymap \
              [list $knm [list [self] keyinvoke %W $dictkey]]
          bind $acclab <ButtonRelease-1> [list [self] mouseinvoke %W $dictkey]
          bind $acclab <ButtonPress-1> [list [self] pressHandler %W]
        }
        default {
          $lab configure $k $v
        }
      }
    }
    my _confunderline $dictkey $lab [list [self] mouseinvoke %W $dictkey]

    foreach {k} {-marginimage label acceleratorwidget} {
      set item [dict get $vars(items) $dictkey $k]
      if { $item ne {} } {
        $item configure -style $nstyle
      }
    }

    my _confstate $dictkey [list label]
  }

  method _command { idx args } {
    my variable vars

    set dictkey [my _iteminit command $idx]

    set lab [my _createlabel $dictkey]
    bind $lab <ButtonRelease-1> [list [self] mouseinvoke %W $dictkey]
    bind $lab <ButtonPress-1> [list [self] pressHandler %W]

    my _confcommand $dictkey {*}$args

    set mi [dict get $vars(items) $dictkey -marginimage]
    if { $mi ne {} } {
      bind $mi <ButtonRelease-1> [list [self] mouseinvoke %W $dictkey]
      bind $mi <ButtonPress-1> [list [self] pressHandler %W]
    }
    dict set vars(items) $dictkey wkeylist [list -marginimage label]
    dict set vars(items) $dictkey cspan 1
  }

  method _confradiobutton { dictkey args } {
    my variable vars

    set rb [dict get $vars(items) $dictkey mainwidget]
    set lab [dict get $vars(items) $dictkey label]
    set nstyle [$lab cget -style]
    set nrbstyle [$rb cget -style]

    foreach {k v} $args {
      switch -exact -- $k {
        -bitmap -
        -command -
        -marginimage -
        -menu -
        -precommand -
        -selectcolor -
        -widget -
        -xalign -
        -yalign {
          # most of these are here simply to make the cloning process easier
          ;
        }
        -text {
          my _conftext $dictkey $lab $k $v
        }
        -borderwidth -
        -relief {
          dict set vars(items) $dictkey $k $v
          # the relief is only used for the background frame
        }
        -activebackground -
        -activeforeground -
        -font -
        -background -
        -foreground {
          dict set vars(items) $dictkey $k $v

          set nstyle $dictkey.Flexmenu.TLabel
          ::ttk::style configure $nstyle $k $v

          set nrbstyle $dictkey.Flexmenu.TRadiobutton
          ::ttk::style configure $nrbstyle $k $v

          set vars(postprocess) false
        }
        -keepopen -
        -state -
        -columnbreak -
        -indicatoron -
        -selectimage -
        -cursor -
        -gap -
        -hidemargin {
          dict set vars(items) $dictkey $k $v
        }
        -image {
          dict set vars(items) $dictkey $k $v
          $lab configure $k $v
        }
        -accel -
        -accelerator {
          dict set vars(items) $dictkey -accelerator $v
          set acclab [my _mkacclab $dictkey $v]
          set knm [my _mkkeyname $v]
          dict lappend vars(items) $dictkey keymap \
              [list $knm [list [self] keyinvoke %W $dictkey]]
          bind $acclab <ButtonRelease-1> [list [self] mouseinvoke %W $dictkey]
          bind $acclab <ButtonPress-1> [list [self] pressHandler %W]
        }
        -compound {
          $lab configure $k $v
        }
        default {
          $rb configure $k $v
        }
      }
    }
    my _confunderline $dictkey $lab [list [self] mouseinvoke %W $dictkey]

    foreach {k} {-marginimage label acceleratorwidget} {
      set item [dict get $vars(items) $dictkey $k]
      if { $item ne {} } {
        $item configure -style $nstyle
      }
    }
    set rb [dict get $vars(items) $dictkey mainwidget]
    $rb configure -style $nrbstyle

    set wkeylist [list]
    if { [dict get $vars(items) $dictkey -indicatoron] } {
      lappend wkeylist mainwidget
    } else {
      lappend wkeylist {}
    }
    lappend wkeylist label
    dict set vars(items) $dictkey wkeylist $wkeylist

    my _confstate $dictkey $wkeylist
  }

  method _radiobutton { idx args } {
    my variable vars

    set dictkey [my _iteminit radiobutton $idx]

    set rb [::ttk::radiobutton $vars(packframe).$dictkey-rb \
        -style Flexmenu.TRadiobutton \
        -takefocus 0]
    dict set vars(items) $dictkey normalstyle Flexmenu.TRadiobutton
    dict set vars(items) $dictkey disabledstyle Disabled.Flexmenu.TRadiobutton
    dict set vars(items) $dictkey activestyle Active.Flexmenu.TRadiobutton
    bind $rb <ButtonRelease-1> [list [self] keyinvoke %W $dictkey]
    bind $rb <ButtonPress-1> [list [self] pressHandler %W]
    bind $rb <Key-space> [list [self] keyinvoke %W $dictkey]
    dict set vars(items) $dictkey mainwidget $rb

    set lab [my _createlabel $dictkey]
    bind $lab <ButtonRelease-1> [list [self] mouseinvoke %W $dictkey]
    bind $lab <ButtonPress-1> [list [self] pressHandler %W]

    my _confradiobutton $dictkey {*}$args

    dict set vars(items) $dictkey cspan 1
  }

  method _separator { idx args } {
    my variable vars

    set dictkey [my _iteminit separator $idx]
    if { $vars(-type) eq "menubar" } {
      set sep [::ttk::separator $vars(packframe).$dictkey-sep \
          -orient vertical \
          -style Vertical.Flexmenu.TSeparator]
      dict set vars(items) $dictkey cspan 1
    } else {
      set sep [::ttk::separator $vars(packframe).$dictkey-sep \
          -orient horizontal \
          -style Horizontal.Flexmenu.TSeparator]
      dict set vars(items) $dictkey cspan 5
    }
    dict set vars(items) $dictkey mainwidget $sep
    dict set vars(items) $dictkey wkeylist [list {} mainwidget]
    dict set vars(items) $dictkey -state disabled
  }

  method tearoffmenuexit { } {
    my variable vars

    my _setAction mouse.click.out
    my _processState
  }

  method tearoff { {dictkey {}} } {
    my variable vars

    set pnm [string map {. {}} $vars(packframe)]
    set mnm .${pnm}-tearoff
    if { ! [winfo exists $mnm] } {
      set newmenu [my clone $mnm]
      set tx [winfo rootx $vars(packframe)]
      set ty [winfo rooty $vars(packframe)]
      $newmenu configure -keepopen true
      $newmenu post [expr {$tx+50}] [expr {$ty+50}]
      if { $dictkey eq {} } {
        set dictkey $vars(active.dictkey)
      }
      if { $dictkey != -1 } {
        my execcmd $dictkey
      }
    }
  }

  method _conftearoff { dictkey args } {
    my variable vars

    set tearoff [dict get $vars(items) $dictkey mainwidget]
    set nstyle [$tearoff cget -style]

    foreach {k v} $args {
      switch -exact -- $k {
        -tearoffcommand -
        -command {
          dict set vars(items) $dictkey -command $v
        }
        -text {
          my _conftext $dictkey $tearoff $k $v
        }
        -borderwidth -
        -relief {
          dict set vars(items) $dictkey $k $v
          # the relief is only used for the background frame
        }
        -activebackground -
        -activeforeground -
        -font -
        -background -
        -foreground {
          dict set vars(items) $dictkey $k $v

          set nstyle $dictkey.Flexmenu.TLabel
          ::ttk::style configure $nstyle $k $v

          set vars(postprocess) false
        }
        -state {
          dict set vars(items) $dictkey $k $v
        }
        -compound -
        -image {
          $tearoff configure $k $v
        }
        default {
          ;
        }
      }
    }

    $tearoff configure -style $nstyle

    set wkeylist mainwidget
    my _confstate $dictkey $wkeylist
  }

  method _tearoff { idx args } {
    my variable vars

    set dictkey [my _iteminit tearoff $idx]
    set vars(tearoff.dictkey) $dictkey
    set tearoff [::ttk::label $vars(packframe).$dictkey-tearoff \
        -takefocus 0 \
        -justify center \
        -anchor center \
        -style Flexmenu.TLabel \
        ]
    bind $tearoff <ButtonRelease-1> [list [self] mouseinvoke %W $dictkey]
    bind $tearoff <ButtonPress-1> [list [self] pressHandler %W]
    set vars(-tearoff) true
    dict set vars(items) $dictkey cspan 5
    dict set vars(items) $dictkey mainwidget $tearoff
    dict set vars(items) $dictkey wkeylist [list {} mainwidget]

    my _conftearoff $dictkey {*}$args -text {- - - -}
  }

  method _confwidget { dictkey args } {
    my variable vars

    set nstyle $vars(-style)Flexmenu.TLabel
    set mi [dict get $vars(items) $dictkey -marginimage]
    if { $mi ne {} } {
      set nstyle [$mi cget -style]
    }

    foreach {k v} $args {
      switch -exact -- $k {
        -accel -
        -accelerator -
        -bitmap -
        -command -
        -image -
        -indicatoron -
        -menu -
        -precommand -
        -selectcolor -
        -selectimage -
        -xalign -
        -yalign {
          # most of these are here simply to make the cloning process easier
          ;
        }
        -marginimage {
          my _confmarginimage $dictkey $k $v
        }
        -borderwidth -
        -relief {
          dict set vars(items) $dictkey $k $v
          # the relief is only used for the background frame
        }
        -activebackground -
        -activeforeground -
        -font -
        -background -
        -foreground {
          dict set vars(items) $dictkey $k $v
          set nstyle $dictkey.Flexmenu.TLabel
          ::ttk::style configure $nstyle $k $v
        }
        -keepopen -
        -state -
        -columnbreak -
        -cursor -
        -gap -
        -hidemargin {
          dict set vars(items) $dictkey $k $v
        }
        -widget {
          dict set vars(items) $dictkey mainwidget $v
          dict set vars(items) $dictkey -widget $v
          lappend vars(widgetlist) $v
        }
        default {
          if { [dict exists $vars(items) $dictkey mainwidget] } {
            set widget [dict get $vars(items) $dictkey mainwidget]
            $widget configure $k $v
          }
        }
      }
    }

    foreach {k} {-marginimage} {
      set item [dict get $vars(items) $dictkey $k]
      if { $item ne {} } {
        $item configure -style $nstyle
      }
    }

    my _confstate $dictkey [list mainwidget]
  }

  method _widget { idx args } {
    my variable vars

    set dictkey [my _iteminit widget $idx]

    my _confwidget $dictkey {*}$args

    set mi [dict get $vars(items) $dictkey -marginimage]
    dict set vars(items) $dictkey wkeylist [list -marginimage mainwidget]
    dict set vars(items) $dictkey cspan 3
  }

  method _getindex { index } {
    my variable vars

    # 2020-6-9
    # allow indexes of the form: active+1, last-2, etc.
    # 'none' cannot have math performed on it.
    # menu text matches cannot have math performed.

    set itemno -1
    switch -regexp -- $index {
      ^\\d+ {
        set itemno $index
        regsub {^\d+} $index $itemno index
      }
      ^end -
      ^last {
        set itemno [expr {$vars(item.count)-1}]
        regsub {^(end|last)} $index $itemno index
      }
      ^invoked {
        set itemno $vars(itemno.invoked)
        regsub {^invoked} $index $itemno index
      }
      ^active {
        set itemno $vars(active.itemno)
        regsub {^active} $index $itemno index
      }
      ^none$ {
        set itemno -1
        regsub {^none} $index $itemno index
      }
      ^@\\d+ {
        if { ! $vars(postprocess) } {
          return -1
        }
        regsub {^@} $index {} y
        set itemno 0
        set found false
        foreach {dictkey} $vars(item.map) {
          set ty [dict get $vars(items) $dictkey y.top]
          set by [dict get $vars(items) $dictkey y.bottom]
          if { $y >= $ty && $y <= $by } {
            set found true
            break
          }
          incr itemno
        }
        if { ! $found } {
          set itemno -1
        }
        regsub {^@\d+} $index $itemno index
      }
      default {
        set itemno 0
        set found false
        foreach {dictkey} $vars(item.map) {
          set l [dict get $vars(items) $dictkey label]
          if { $l ne {} } {
            set txt [$l cget -text]
            if { [string match $index $txt] } {
              set found true
              break
            }
          }
          incr itemno
        }
        if { ! $found } {
          set itemno -1
        }
        set index $itemno
      }
    } ; # switch

    set itemno [expr $index]
    # if the calculation is out of range,
    # change it to the standard error return.
    if { $itemno < 0 || $itemno >= $vars(item.count) } {
      set itemno -1
    }
    return $itemno
  } ; # _getindex

  method _add { idx cmd args } {
    my variable vars

    regsub -all -- {-label} $args -text args

    switch -exact -- $cmd {
      cascade {
        my _cascade $idx {*}$args
      }
      check -
      checkbutton {
        my _checkbutton $idx {*}$args
      }
      command {
        my _command $idx {*}$args
      }
      radio -
      radiobutton {
        my _radiobutton $idx {*}$args
      }
      sep -
      separator {
        my _separator $idx {*}$args
      }
      tearoff {
        if { $vars(-type) eq "submenu" } {
          my _tearoff $idx {*}$args
        }
      }
      widget {
        my _widget $idx {*}$args
      }
      default {
        puts stderr "error: unknown add sub-command: $cmd"
      }
    }
    set vars(confchanged) true
    if { $vars(menu.posted) } {
      for {set i $idx} {$i < $vars(item.count)} {incr i} {
        dict set vars(items) [my _getdictkey $i] changed true
      }

      set vars(dictkey.new) -1
      set vars(itemno.new) -1
      set vars(active.dictkey) -1
      set vars(active.itemno) -1
      set vars(dictkey.prior) -1
      set vars(itemno.prior) -1
      set vars(active.dictkey.last) -1

      my _updateButtonDisplay
      my _layoutmenu

      set w $vars(itemframe)
      after idle [list [self] mouseHandler $w [winfo pointerx $w] [winfo pointery $w]]
    }
  }

  # helper routines

  method invokeactive { tag } {
    my variable vars

    my invoke $vars(active.itemno)
  }

  # user space routines

  method activate { index } {
    my variable vars

    set itemno [my _getindex $index]
    if { $itemno != -1 } {
      set dictkey [my _getdictkey $itemno]
      set vars(dictkey.new) $dictkey
      set vars(itemno.new) $itemno
      my _setAction key.move
      my _processState
    }
  }

  method add { cmd args } {
    my _add end $cmd {*}$args
  }

  method clonehelper { vlist ilist {rtflag {}} } {
    my variable vars

    my varsinit
    foreach {k v} $vlist {
      if { [string index $k 0] eq "-" &&
          $k ne "-type" &&
          $k ne "-tornoff" &&
          $k ne "-style" &&
          $k ne "-mode" } {
        set vars($k) $v
      }
    }
    foreach {item info} $ilist {
      set type [dict get $info type]
      if { $rtflag eq "-removetearoff" && $type eq "tearoff" } {
        continue
      }
      array set a $info
      my add $type {*}[array get a -*]
    }

    # clone all cascades.
    foreach {dictkey} $vars(item.map) {
      set type [dict get $vars(items) $dictkey type]
      if { $type eq "cascade" } {
        set oldcascade [dict get $vars(items) $dictkey -menu]
        regsub {.*\.} $oldcascade {} nm
        set newcascade [$oldcascade clone $vars(name).$nm -cascade]
        dict set vars(items) $dictkey -menu $newcascade
      }
    }
  }

  method clone { mnm args } {
    my variable vars

    set iscascade false
    set istearoff true
    foreach {a} $args {
      switch -exact -- $a {
        -cascade {
          set iscascade true
          set istearoff false
        }
      }
    }

    set rtflag {}
    set margs {}
    if { ! $iscascade } {
      # configure $newmenu to be a top menu
      if { $istearoff } {
        set margs [list -mode toplevel -type submenu \
            -tornoff true -title $vars(-title)]
        set rtflag -removetearoff
      }
    }

    set newmenu [::flexmenu $mnm {*}$margs]
    $newmenu clonehelper [array get vars] [dict get $vars(items)] $rtflag
    return $newmenu
  }

  method cget { opt } {
    my variable vars

    set rc {}
    if { [info exists vars($opt)] } {
      set rc $vars($opt)
    }
    return $rc
  }

  method _updatetearoff { flag } {
    my variable vars

    if { $flag &&
        $vars(packframe) ne {} &&
        $vars(-type) eq "submenu" &&
        $vars(tearoff.dictkey) eq {} } {
      my _tearoff 0
    }
    if { ! $flag &&
        $vars(-type) eq "submenu" &&
        $vars(tearoff.dictkey) ne {} } {
      set idx [lsearch -exact -integer $vars(item.map) $vars(tearoff.dictkey)]
      my delete $idx
    }
  }

  method configure { args } {
    my variable vars

    if { [llength $args] == 0 } {
      set resp [array get vars -*]
      return $resp
    }

    regsub -all -- {-label} $args -text args

    foreach {k v} $args {
      switch -exact -- $k {
        -mode {
          if { $::tcl_platform(os) eq "Darwin" &&
              [package vcompare $::flexmenucmd::vars(darwin.version) 10.14] < 0 } {
            # -mode toplevel did not work for earlier mac os versions.
            # still does not work great...
            set v frame
          }
          set vars($k) $v
        }
        -font -
        -activebackground -
        -activeforeground -
        -background -
        -foreground {
          set vars($k) $v
          set nm {}
          if { $vars(-type) eq "submenu" } {
            regsub -all {\.} $vars(name) {} nm
            set nm $nm.
          }
          my _confStyling ${nm}Flexmenu.TLabel [list $k $v]
          my _confStyling ${nm}Accelerator.Flexmenu.TLabel [list $k $v]
          my _confStyling ${nm}Flexmenu.TCheckbutton [list $k $v]
          my _confStyling ${nm}Flexmenu.TRadiobutton [list $k $v]
          if { [info exists vars(itemframe)] &&
              [info exists vars(itemframe.cmd)] &&
              $k eq "-background" } {
            $vars(itemframe.cmd) configure $k $v
            if { $vars(-maxheight) != 0 } {
              $vars(itemframe).tcont configure $k $v
              $vars(itemframe).innerframe configure $k $v
            }
            foreach {dictkey} $vars(item.map) {
              dict set vars(items) $dictkey $k $v
            }
          }
          if { $vars(-type) eq "submenu" } {
            set vars(-style) $nm
          }
        }
        -tearoff {
          set vars($k) $v
          my _updatetearoff $v
        }
        -tearoffcommand {
          if { $vars(tearoff.dictkey) ne {} } {
            my _conftearoff $vars(tearoff.dictkey) $k $v
          }
        }
        default {
          set vars($k) $v
        }
      }
    }

    set vars(confchanged) true
    if { $vars(menu.posted) } {
      my _updateButtonDisplay
      my _layoutmenu
    }
  }

  method delete { firstidx {lastidx {}} } {
    my variable vars

    if { $vars(menu.posted) } {
      my _deactivateItem
    }

    if { $lastidx eq {} } {
      set lastidx $firstidx
    }
    set first [my _getindex $firstidx]
    set last [my _getindex $lastidx]

    set ilist $vars(item.map)

    # anything to the left of the deletion point will not get
    # re-configured.
    set c [expr {-$first}]
    set vars(config.count.base) $c

    for {set itemno $first} {$itemno <= $last} {incr itemno} {
      if { $itemno == -1 } {
        continue
      }

      lset ilist $itemno {}
      set dictkey [my _getdictkey $itemno]
      set wkeylist [dict get $vars(items) $dictkey wkeylist]
      foreach {wkey} $wkeylist {
        if { $wkey ne {} } {
          if { [dict get $vars(items) $dictkey -gap] } {
            if { [dict exists $vars(items) $dictkey startcol] } {
              set col [dict get $vars(items) $dictkey startcol]
              incr col -1
              grid columnconfigure $vars(packframe) $col -weight 0
            }
          }
          set w [dict get $vars(items) $dictkey $wkey]
          if { $w ne {} } {
            destroy $w
          }
        }
      }
      dict unset vars(items) $dictkey
    }

    set nlist [list]
    set itemno 0
    foreach {dictkey} $ilist {
      if { $dictkey ne {} } {
        lappend nlist $dictkey
        if { $itemno >= $first } {
          dict set vars(items) $dictkey changed true
        }
      }
      incr itemno
    }

    set vars(item.map) $nlist
    set vars(item.count) [llength $vars(item.map)]
    set vars(confchanged) true

    set vars(active.dictkey) -1
    set vars(active.itemno) -1

    if { $vars(menu.posted) } {
      my _updateButtonDisplay
      my _layoutmenu

      set vars(dictkey.new) -1
      set vars(itemno.new) -1
      set vars(dictkey.prior) -1
      set vars(itemno.prior) -1
      set vars(active.dictkey.last) -1

      set w $vars(itemframe)
      after idle [list [self] mouseHandler $w [winfo pointerx $w] [winfo pointery $w]]
    }
  }

  method entrycget { index opt } {
    my variable vars

    set itemno [my _getindex $index]
    if { $itemno == -1 } {
      return
    }

    set dictkey [my _getdictkey $itemno]
    if { [dict exists $vars(items) $dictkey $opt] } {
      set val [dict get $vars(items) $dictkey $opt]
      if { $val ne {} } {
        return $val
      }
    }

    set w {}
    set lab {}
    set type [dict get $vars(items) $dictkey type]

    switch -exact -- $type {
      cascade {
        set w [dict get $vars(items) $dictkey label]
      }
      check -
      checkbutton {
        set w [dict get $vars(items) $dictkey mainwidget]
        set lab [dict get $vars(items) $dictkey label]
      }
      command {
        set w [dict get $vars(items) $dictkey label]
      }
      radio -
      radiobutton {
        set w [dict get $vars(items) $dictkey mainwidget]
        set lab [dict get $vars(items) $dictkey label]
      }
      sep -
      separator {
        ;
      }
      tearoff {
        ;
      }
      widget {
        set w [dict get $vars(items) $dictkey mainwidget]
      }
    }

    if { $lab ne {} && $opt eq "-label" } {
      set val [$lab cget -text]
    }
    set lopts {-compound -width -underline -wraplength -state}
    if { $val eq {} && $lab ne {} && $opt in $lopts } {
      set val [$lab cget $opt]
    }
    if { $val eq {} && $w ne {} } {
      set val [$w cget $opt]
    }

    return $val
  }

  method entryconfigure { index args } {
    my variable vars

    set itemno [my _getindex $index]
    if { $itemno == -1 } {
      return
    }
    set dictkey [my _getdictkey $itemno]
    set type [dict get $vars(items) $dictkey type]

    if { [llength $args] == 0 } {
      # this is a mess....
      set resp [list]
      set w [dict get $vars(items) $dictkey mainwidget]
      if { $w ne {} } {
        if { $type eq "radiobutton" } {
          foreach {k} {-variable -compound -value -textvariable} {
            lappend resp $k [$w cget $k]
          }
        }
        if { $type eq "checkbutton" } {
          foreach {k} {-variable -compound -textvariable -onvalue -offvalue} {
            lappend resp $k [$w cget $k]
          }
        }
      }
      set w [dict get $vars(items) $dictkey label]
      if { $w ne {} } {
        foreach {k} {-text -compound -width -underline -wraplength} {
          lappend resp $k [$w cget $k]
        }
      }
      if { [dict exists $resp -text] } {
        dict set resp -label [dict get $resp -text]
        dict unset resp -text
      }
      dict for {k v} [dict get $vars(items) $dictkey] {
        if { [regexp {^\-} $k] } {
          lappend resp $k $v
        }
      }
      return $resp
    }

    regsub -all -- {-label} $args -text args

    set lab [dict get $vars(items) $dictkey label]
    foreach {k v} $args {
      switch -exact -- $type {
        cascade {
          my _confcascade $dictkey {*}$args
        }
        check -
        checkbutton {
          my _confcheckbutton $dictkey {*}$args
        }
        command {
          my _confcommand $dictkey {*}$args
        }
        radio -
        radiobutton {
          my _confradiobutton $dictkey {*}$args
        }
        sep -
        separator {
          ;
        }
        tearoff {
          my _conftearoff $dictkey {*}$args
        }
        widget {
          my _confwidget $dictkey {*}$args
        }
      }
    }

    set vars(confchanged) true
    if { $vars(menu.posted) } {
      my _updateButtonDisplay
      my _layoutmenu
      ### the changed flag for the item may need to be set.
      ### the item display may also need resetting.
    }
  }

  method index { index } {
    return [my _getindex $index]
  }

  method insert { index type args } {
    my variable vars

    set itemno [my _getindex $index]
    if { $itemno != -1 } {
      # anything to the left of the insertion point will not get
      # re-configured.
      set c [expr {-$itemno}]
      set vars(config.count.base) $c

      if { $vars(menu.posted) } {
        my _deactivateItem
      }
      my _add $itemno $type {*}$args
    }
  }

  method invoke { index } {
    my variable vars

    set rc false
    set itemno [my _getindex $index]
    if { $itemno != -1 } {
      set dictkey [my _getdictkey $itemno]
      set currstate [my _getState]
      if { $currstate eq "inactive" } {
        # if the menu is not open, simply process the command
        my _invokeItem $dictkey $itemno
      } else {
        # if the menu is open, process a state change so
        # that the menu status/etc. will be updated.
        set vars(dictkey.new) $dictkey
        set vars(itemno.new) $itemno
        my _setAction mouse.invoke
        my _processState
      }
      set rc true
    }
    return $rc
  }

  method post { x y {parent {}} } {
    my _postMenu $x $y $parent
    my _setAction menu.post
    my _processState
  }

  method postcascade { {index {}} } {
    my variable vars

    if { $index eq {} } {
      set itemno $vars(active.itemno)
    } else {
      set itemno [my _getindex $index]
    }
    if { $itemno == -1 } {
      return
    }

    set dictkey [my _getdictkey $itemno]
    if { [dict get $vars(items) $dictkey type] ne "cascade" } {
      return
    }

    set vars(dictkey.new) $dictkey
    set vars(itemno.new) $itemno
    my _setAction key.invoke
    my _processState
  }

  method type { index } {
    my variable vars

    set itemno [my _getindex $index]
    set dictkey [my _getdictkey $itemno]
    return [dict get $vars(items) $dictkey type]
  }

  method unpost { } {
    my variable vars

    my _setAction menu.unpost
    my _processState
  }

  method xposition { index } {
    my variable vars

    if { ! $vars(postprocess) } {
      my waitpostprocess [list my xposition $index]
      return
    }
    set itemno [my _getindex $index]
    set dictkey [my _getdictkey $itemno]
    return [dict get $vars(items) $dictkey x.left]
  }

  method yposition { index } {
    my variable vars

    if { ! $vars(postprocess) } {
      my waitpostprocess [list my yposition $index]
      return
    }
    set itemno [my _getindex $index]
    set dictkey [my _getdictkey $itemno]
    return [dict get $vars(items) $dictkey y.top]
  }
}
