#!/usr/bin/env tclsh
# Example usage of the CBOR package

# Add package to path
lappend auto_path [file dirname [info script]]
package require cbor

puts "=== CBOR Package Example ==="
puts ""

# Example 1: Simple values
puts "Example 1: Encoding simple values"
puts "-----------------------------------"
set values [list 42 "Hello, CBOR!" true false null 3.14]
foreach val $values {
    set encoded [cbor::encode $val]
    set decoded [cbor::decode $encoded]
    puts "Original: $val"
    puts "  Encoded (hex): [binary encode hex $encoded]"
    puts "  Decoded: $decoded"
    puts ""
}

# Example 2: Arrays
puts "Example 2: Encoding arrays"
puts "-----------------------------------"
set myArray [list 1 2 3 "four" 5.0]
set encoded [cbor::encode $myArray]
set decoded [cbor::decode $encoded]
puts "Original array: $myArray"
puts "Encoded (hex): [binary encode hex $encoded]"
puts "Decoded array: $decoded"
puts ""

# Example 3: Maps (dictionaries)
puts "Example 3: Encoding maps"
puts "-----------------------------------"
set myDict [dict create \
    "name" "Alice" \
    "age" 30 \
    "city" "San Francisco" \
    "active" true \
]
set encoded [cbor::encode $myDict]
set decoded [cbor::decode $encoded]
puts "Original dict:"
dict for {key value} $myDict {
    puts "  $key: $value"
}
puts "Encoded (hex): [binary encode hex $encoded]"
puts "Decoded dict:"
dict for {key value} $decoded {
    puts "  $key: $value"
}
puts ""

# Example 4: Complex nested structure
puts "Example 4: Complex nested structure"
puts "-----------------------------------"
set complexData [dict create \
    "users" [list \
        [dict create "name" "Alice" "age" 30 "admin" true] \
        [dict create "name" "Bob" "age" 25 "admin" false] \
    ] \
    "count" 2 \
    "timestamp" 1234567890 \
]
set encoded [cbor::encode $complexData]
set decoded [cbor::decode $encoded]
puts "Original structure:"
puts "  Users: [dict get $complexData users]"
puts "  Count: [dict get $complexData count]"
puts "Encoded size: [string length $encoded] bytes"
puts "Decoded successfully: [expr {$decoded eq $complexData}]"
puts ""

# Example 5: Saving to file
puts "Example 5: Saving to file"
puts "-----------------------------------"
set data [dict create \
    "message" "Hello from CBOR!" \
    "values" [list 1 2 3 4 5] \
    "timestamp" [clock seconds] \
]
set encoded [cbor::encode $data]

# Write to file
set fp [open "example.cbor" wb]
puts -nonewline $fp $encoded
close $fp
puts "Saved [string length $encoded] bytes to example.cbor"

# Read from file
set fp [open "example.cbor" rb]
set fileData [read $fp]
close $fp
set decoded [cbor::decode $fileData]
puts "Read from file and decoded:"
dict for {key value} $decoded {
    puts "  $key: $value"
}

# Clean up
file delete "example.cbor"
puts ""
puts "=== Examples Complete ==="
