# Demo: TI SensorTag
# May/June 2015 <chw@ch-werner.de>
# Hardware info: http://processors.wiki.ti.com/index.php/SensorTag_User_Guide

package require Borg
package require Ble
package require snack

font create ConsoleFont -family {DejaVu Sans Mono} -size 6
font configure TkDefaultFont -size 10

catch {
    font configure TkDefaultFont \
	-family [sdltk addfont \
	    [file join [file dirname [info script]] federation.ttf]] \
    font configure ConsoleFont \
	-family [sdltk addfont \
	    [file join [file dirname [info script]] beijing.ttf]] \
	-size 6
}

catch {
    set ::sensortag(sound) \
	[snack::sound -file \
	     [file join [file dirname [info script]] tricorder.wav]]
    $::sensortag(sound) stop
}

catch {
    set ::sensortag(sound_l) \
	[snack::sound -file \
	     [file join [file dirname [info script]] beep.wav]]
    $::sensortag(sound_l) stop
}

catch {
    set ::sensortag(sound_r) \
	[snack::sound -file \
	     [file join [file dirname [info script]] chirp.wav]]
    $::sensortag(sound_r) stop
}

catch {
    set ::sensortag(sound_t) \
	[snack::sound -file \
	     [file join [file dirname [info script]] transbeep.wav]]
    $::sensortag(sound_t) stop
}

proc play_sound {} {
    catch {$::sensortag(sound) play -blocking 0 -command play_sound}
    after cancel colorize
    after 100 colorize
}

proc play_sound_l {} {
    catch {$::sensortag(sound_l) play -blocking 0}
}

proc play_sound_r {} {
    catch {$::sensortag(sound_r) play -blocking 0}
}

proc play_sound_t {} {
    if {$::sensortag(play_t)} {
	return
    }
    if {![catch {$::sensortag(sound_t) play -blocking 0 -command {
	set ::sensortag(play_t) 0
    }}]} {
	set ::sensortag(play_t) 1
    }
}

proc stop_sound {} {
    catch {$::sensortag(sound) stop}
    .color0 configure -bg #B000B0
    .color1 configure -bg #B000B0
    after cancel play_sound
    after cancel play_sound_l
    after cancel play_sound_r
    after cancel play_sound_t
    after cancel colorize
}

proc colorize {} {
    set c [.color0 cget -bg]
    set g 0
    scan [string range $c 3 4] "%x" g
    incr g 16
    if {$g >= 160} {
	set g 16
    }
    set c [format "#00%02x00" $g]
    .color0 configure -bg $c
    .color1 configure -bg $c
    after 100 colorize
}

proc send_cmd {handle} {
    set cmds [ble userdata $handle]
    if {$cmds eq {}} {
	# all done
	return
    }
    set cmd [lindex $cmds 0]
    set cmds [lrange $cmds 1 end]
    if {[{*}$cmd] == 0} {
	# not done, keep it
	return
    }
    ble userdata $handle $cmds
}

proc ble_handler {event data} {
    if {$::sensortag(closed)} {
	return
    }
    dict with data {
	switch -- $event {
	    scan {
		if {$name eq "SensorTag"} {
		    # found the TI SensorTag, close scanner, then connect
		    ble close $handle
		    set ::sensortag(scanning) 0
		    after cancel colorize
		    after 100 colorize
		    # no autoreconnect in order to perform immediate connect!
		    ble connect $address ble_handler 0
		}
		# debug output
		if {[info exists scandata]} {
		    # make hex string from byte array
		    binary scan $scandata H* scandata
		    dict set data scandata $scandata
		}
	    }
	    connection {
		if {$state eq "disconnected"} {
		    # fall back to scanning
		    ble close $handle
		    ble start [ble scanner ble_handler]
		    set ::sensortag(found) 0
		    set ::sensortag(scanning) 1
		    stop_sound
		} elseif {$state eq "connected"} {
		    set addcmds [expr {!$::sensortag(found)}]
		    set ::sensortag(found) 1
		    after cancel play_sound
		    after idle play_sound
		    set cmds [ble userdata $handle]
		    if {$addcmds} {
			# Add commands to turn various sensors on.
			# Barometer needs two configurations to load
			# its calibration. Gyroscope has a bitmask
			# for various axes.
			set on1 [binary format H* "01"]
			set on2 [binary format H* "02"]
			set on7 [binary format H* "07"]
			foreach {suuid cuuid on} {
			    F000AA00 F000AA02 on1
			    F000AA10 F000AA12 on1
			    F000AA20 F000AA22 on1
			    F000AA30 F000AA32 on1
			    F000AA40 F000AA42 on2
			    F000AA50 F000AA52 on7
			    F000AA40 F000AA42 on1
			} {
			    lappend cmds \
				[list ble write $handle $suuid 0 $cuuid 0 \
				     [set $on]]
			}
			# read barometer calibration
			lappend cmds \
			    [list ble read $handle F000AA40 0 F000AA43 0]
			ble userdata $handle $cmds
		    }
		    # now send command
		    send_cmd $handle
		}
	    }
	    characteristic {
		if {($state eq "discovery") && ($properties & 0x10)} {
		    # later turn on notifications
		    set cmds [ble userdata $handle]
		    lappend cmds [list ble enable $handle $suuid $sinstance \
				      $cuuid $cinstance]
		    ble userdata $handle $cmds
		} elseif {$state eq "connected"} {
		    send_cmd $handle
		    if {($access eq "c") || ($access eq "r")} {
			# change notification or read request
			ble_new_value $cuuid $value
		    }
		}
	    }
	    descriptor {
		if {$state eq "connected"} {
		    send_cmd $handle
		}
	    }
	}
	# debug output
	if {[info exists value]} {
	    # make hex string from byte array
	    binary scan $value H* value
	    dict set data value $value
	}
	.t insert end "${event}: ${data}\n"
	# Prune text widget to max. 100 lines.
	scan [.t index end] "%d.%*d" lines
	if {$lines > 100} {
	    .t delete 1.0 [expr $lines - 100].0
	}
	.t yview end
    }
}

proc ble_new_value {uuid value} {
    switch -glob -- $uuid {
	F000AA01-* {
	    # temperature
	    set t0 0
	    set t1 0
	    binary scan $value s1s1 t0 t1
	    set tamb [expr {($t1 & 0xFFFF) / 128.0}]
	    set t0 [expr {$t0 * 0.00000015625}]
	    set tdie [expr {$tamb + 273.15}]
	    set s0 5.593e-14
	    set a1 1.75e-3
	    set a2 -1.678e-5
	    set b0 -2.94e-5
	    set b1 -5.7e-7
	    set b2 4.63e-9
	    set c2 13.4
	    set tref 298.15
	    set s [expr {$s0 * (1 + $a1 * ($tdie - $tref) + $a2 *
				pow(($tdie - $tref), 2))}]
	    set vos [expr {$b0 + $b1 * ($tdie - $tref) + $b2 *
			   pow(($tdie - $tref), 2)}]
	    set fobj [expr {($t0 - $vos) + $c2 * pow(($t0 - $vos), 2)}]
	    set tobj [expr {pow($tdie, 4) + ($fobj / $s)}]
	    if {$tobj < 0} {
		set tobj 0
	    }
	    set tobj [expr {pow($tobj, 0.25)}]
	    set tobj [expr {$tobj - 273.15}]
	    set ::sensortag(temp_object) [format "%.3f" $tobj]
	    set ::sensortag(temp_ambient) [format "%.3f" $tamb]
	}
	F000AA11-* {
	    # accelerometer
	    set x 0
	    set y 0
	    set z 0
	    binary scan $value c1c1c1 x y z
	    set ::sensortag(accel_x) [format "%.5f" [expr {$x / 64.0}]]
	    set ::sensortag(accel_y) [format "%.5f" [expr {$y / 64.0}]]
	    set ::sensortag(accel_z) [format "%.5f" [expr {$z / -64.0}]]
	    if {(abs($x) > 75) || (abs($y) > 75) || (abs($z) > 75)} {
		after cancel play_sound_t
		after idle play_sound_t
	    }
	}
	F000AA21-* {
	    # humidity
	    set a 0
	    set b 0
	    binary scan $value s1s1 a b
	    set b [expr {$b & 0xFFFC}]
	    set ::sensortag(humidity) \
		[format "%.2f" [expr {-6.0 + 125.0 * ($b / 65535.0)}]]
	}
	F000AA31-* {
	    # magnetic field
	    set x 0
	    set y 0
	    set z 0
	    binary scan $value s1s1s1 x y z
	    set ::sensortag(magnetic_x) \
		[format "%.5f" [expr {0 - $x * 2000.0 / 65536.0}]]
	    set ::sensortag(magnetic_y) \
		[format "%.5f" [expr {0 - $y * 2000.0 / 65536.0}]]
	    set ::sensortag(magnetic_z) \
		[format "%.5f" [expr {$z * 2000.0 / 65536.0}]]
	}
	F000AA41-* {
	    # barometer value
	    set tr 0
	    set pr 0
	    binary scan $value s1s1 tr pr
	    set pr [expr {$pr & 0xFFFF}]
	    array set c [array get ::sensortag "bar_c*"]
	    set s [expr {$c(bar_c2) + $c(bar_c3) * $tr / 131072.0 +
			 (($c(bar_c4) * $tr / 32768.0) * $tr) / 524288.0}]
	    set o [expr {$c(bar_c5) * 16384.0 + $c(bar_c6) * $tr / 8.0 +
			 (($c(bar_c7) * $tr / 32768.0) * $tr) / 16.0}]
	    set pa [expr {($s * $pr + $o) / 16384.0}]
	    set ::sensortag(bar_pressure) [format "%.2f" [expr {$pa / 100}]]
	}
	F000AA43-* {
	    # barometer calibration values
	    set v {}
	    set i 0
	    binary scan $value s8 v
	    foreach c $v {
		if {$i < 4} {
		    # c0..c3 are unsigned
		    set ::sensortag(bar_c$i) [expr {$c & 0xFFFF}]
		} else {
		    set ::sensortag(bar_c$i) $c
		}
		incr i
	    }
	}
	F000AA51-* {
	    # gyroscope
	    set x 0
	    set y 0
	    set z 0
	    binary scan $value s1s1s1 y x z
	    set ::sensortag(gyro_x) \
		[format "%.5f" [expr {$x * 500.0 / 65536.0}]]
	    set ::sensortag(gyro_y) \
		[format "%.5f" [expr {0 - $y * 500.0 / 65536.0}]]
	    set ::sensortag(gyro_z) \
		[format "%.5f" [expr {$z * 500.0 / 65536.0}]]
	}
	0000FFE1-* {
	    # buttons
	    set v 0
	    binary scan $value c1 v
	    set ::sensortag(button_l) [expr {($v & 2) != 0}]
	    set ::sensortag(button_r) [expr {($v & 1) != 0}]
	    if {$v & 2} {
		after cancel play_sound_l
		after idle play_sound_l
	    }
	    if {$v & 1} {
		after cancel play_sound_r
		after idle play_sound_r
	    }
	}
    }
    if {0} {
	# debug output
	parray ::sensortag
    }
}

wm attributes . -fullscreen 1
. configure -bg black
sdltk screensaver off
borg screenorientation portrait
bind all <Key-Break> exit
bind all <<DidEnterBackground>> sensortag_disconnect

foreach {w l bg row} {
    .ltemp	Temperature	#AA4444		0
    .laccel	Acceleration	#4444AA		2
    .lhumid	Humidity	#44AA44		5
    .lgyro	Gyroscope	#00AAAA		6
    .lmagn	Mag.Field	#AA44AA		9
    .lpres	Pressure	#AA4400		12
    .lbtns	Buttons		#AAAA44		13
    .ldate	Stardate	#4444AA		15
    .lscan	Scan		#44AAAA		16
} {
    label $w -text $l -width 12 -bg $bg
    grid $w -row $row -column 0 -pady 5
}

foreach {w l bg row} {
    .ltempa	amb		#AA4444		0
    .ltempo	obj		#AA4444		1
    .laccelx	X		#4444AA		2
    .laccely	Y		#4444AA		3
    .laccelz	Z		#4444AA		4
    .lhumid1	%		#44AA44		5
    .lgyrox	X		#00AAAA		6
    .lgyroy	Y		#00AAAA		7
    .lgyroz	Z		#00AAAA		8
    .lmagnx	X		#AA44AA		9
    .lmagny	Y		#AA44AA		10
    .lmagnz	Z		#AA44AA		11
    .lpres1	hPa		#AA4400		12
    .lbtnsl	left		#AAAA44		13
    .lbtnsr	right		#AAAA44		14
    .ldate0	{}		#4444AA		15
    .color0	{}		#E00000		16
} {
    label $w -text $l -width 5 -bg $bg
    grid $w -row $row -column 1 -padx 10 -pady 5
}

foreach {w v bg row} {
    .vtempa	::sensortag(temp_ambient)	#AA4444		0
    .vtempo	::sensortag(temp_object)	#AA4444		1
    .vaccelx	::sensortag(accel_x)		#4444AA		2
    .vaccely	::sensortag(accel_y)		#4444AA		3
    .vaccelz	::sensortag(accel_z)		#4444AA		4
    .vhumid	::sensortag(humidity)		#44AA44		5
    .vgyrox	::sensortag(gyro_x)		#00AAAA		6
    .vgyroy	::sensortag(gyro_y)		#00AAAA		7
    .vgyroz	::sensortag(gyro_z)		#00AAAA		8
    .vmagnx	::sensortag(magnetic_x)		#AA44AA		9
    .vmagny	::sensortag(magnetic_y)		#AA44AA		10
    .vmagnz	::sensortag(magnetic_z)		#AA44AA		11
    .vpres	::sensortag(bar_pressure)	#AA4400		12
    .vbtnsl	::sensortag(button_l)		#AAAA44		13
    .vbtnsr	::sensortag(button_r)		#AAAA44		14
    .vdate	::sensortag(stardate)		#4444AA		15
    .color1	::sensortag(no_var)		#E00000		16
} {
    label $w -textvariable $v -width 12 -bg $bg
    grid $w -row $row -column 2 -pady 5
}

text .t -width 40 -height 10 -bg black -fg #22AA22 \
    -font ConsoleFont -highlightthickness 0 -relief flat
grid .t -row 17 -column 0 -columnspan 3 -sticky nswe -pady 5
grid rowconfigure . 17 -weight 1
bindtags .t .t
bind .t <Double-1> {.t delete 1.0 end}

proc sensortag_scan {} {
    ble close all
    ble start [ble scanner ble_handler]
    stop_sound
    set ::sensortag(closed) 0
    set ::sensortag(found) 0
    set ::sensortag(scanning) 1
    .color0 configure -bg #B000B0
    .color1 configure -bg #B000B0
    after cancel colorize
}

proc sensortag_disconnect {} {
    set ::sensortag(closed) 1
    ble close all
    stop_sound
    set ::sensortag(found) 0
    set ::sensortag(scanning) 0
    .color0 configure -bg #E00000
    .color1 configure -bg #E00000
    after cancel colorize
}

proc sensortag_found {args} {
    set found 0
    if {[info exists ::sensortag(found)]} {
	set found $::sensortag(found)
    }
    set scanning 0
    if {[info exists ::sensortag(scanning)]} {
	set scanning $::sensortag(scanning)
    }
    if {$found} {
	if {!$scanning} {
	    .lscan configure -text "Disconnect"
	    bind .lscan <1> sensortag_disconnect
	} else {
	    .lscan configure -text "Disconnect"
	    bind .lscan <1> sensortag_scan
	}
    } else {
	if {$scanning} {
	    .lscan configure -text "Disconnect"
	    bind .lscan <1> sensortag_disconnect
	} else {
	    .lscan configure -text "Scan"
	    bind .lscan <1> sensortag_scan
	}
    }
}

proc sensortag_stardate {} {
    after 5000 sensortag_stardate
    set d [clock format [clock seconds] -format %Q]
    set ::sensortag(stardate) [lindex $d 1]
}

sensortag_stardate

trace add variable ::sensortag(found)    write sensortag_found
trace add variable ::sensortag(scanning) write sensortag_found

array set ::sensortag {
    closed   1
    found    0
    scanning 0
    button_l 0
    button_r 0
    bar_c0 45455
    bar_c1 27123
    bar_c2 45989
    bar_c3 33840
    bar_c4 3263
    bar_c5 -17400
    bar_c6 -12772
    bar_c7 1916
    play_t 0
}

# try to guess good font size according to display width
set d [expr {round([font measure TkDefaultFont "0 M"] / 3.0)}]
set q [expr {[dict get [borg displaymetrics] width] / $d}]
font configure TkDefaultFont -size [expr {round(10.0 * $q / 30)}]
