################################################################################
################################################################################
####                                  tclcurl.tcl
################################################################################
################################################################################
## Includes the tcl part of TclCurl
################################################################################
################################################################################
## (c) 2001-2011 Andres Garcia Garcia. fandom@telefonica.net
## See the file "license.terms" for information on usage and redistribution
## of this file and for a DISCLAIMER OF ALL WARRANTIES.
################################################################################
################################################################################

package provide TclCurl 7.22.0

namespace eval curl {

    variable cainfo [file join [file dirname [info script]] cert.pem]

################################################################################
# configure
#    Not available in AndroWish/undroidwish
################################################################################
proc ::curl::curlConfig {option} {
    error "This command is not available"
}

################################################################################
# setupCA
#    Depending on platform and installation, setup -cainfo/-capath,
#    and -cainfoblob options of the given handle.
################################################################################
proc ::curl::setupCA {curlHandle} {
    variable cainfo
    if {[info exists cainfo] && [file readable $cainfo]} {
        # Read it using Tcl due to VFS support.
        set f [open $cainfo rb]
        set blob [read $f]
        close $f
        $curlHandle configure -cainfoblob $blob
        return 1
    }
    if {$::tcl_platform(os) eq "Darwin"} {
        set certs /usr/local/etc/ca-certificates/cert.pem
        set ret 0
        if {[file readable $certs]} {
            $curlHandle configure -cainfo $certs
            set ret 1
        } else {
            set certs /etc/ssl/cert.pem
            if {[file readable $certs]} {
                $curlHandle configure -cainfo $certs
                set ret 1
            }
        }
        set certd /etc/ssl/certs
        if {[file isdirectory $certd]} {
            $curlHandle configure -capath $certd
            set ret 1
        }
        return $ret
    }
    if {$::tcl_platform(platform) ne "windows"} {
        set ret 0
        foreach certs {
            /etc/ssl/certs/ca-certificates.crt
            /etc/pki/tls/certs/ca-bundle.crt
            /etc/ssl/ca-bundle.pem
            /etc/pki/tls/cacert.pem
            /etc/pki/ca-trust/extracted/pem/tls-ca-bundle.pem
            /etc/ssl/cert.pem
        } {
            if {[file readable $certs]} {
                $curlHandle configure -cainfo $certs
                set ret 1
                break
            }
        }
        foreach certd {
            /etc/ssl/certs
            /system/etc/security/cacerts
            /usr/local/share/certs
            /etc/pki/tls/certs
            /etc/openssl/certs
            /var/ssl/certs
        } {
            if {[file isdirectory $certd]} {
                $curlHandle configure -capath $certd
                set ret 1
                break
            }
        }
        return $ret
    }
    return 0
}

################################################################################
# transfer
#    The transfer command is used for simple transfers in which you don't
#    want to request more than one file.
#
# Parameters:
#    Use the same parameters you would use in the 'configure' command to
#    configure the download and the same as in 'getinfo' with a 'info'
#    prefix to get info about the transfer.
################################################################################
proc ::curl::transfer {args} {
    variable getInfo
    variable curlBodyVar

    set i 0
    set newArgs ""
    catch {unset getInfo}

    if {[llength $args]==0} {
        error "No transfer configured"
    }

    if {[llength $args] % 2 != 0} {
        error "Missing value for option '[lindex $args end]'"
    }

    foreach {option value} $args {
        set noPassOption 0
        set block        1
        switch -regexp -- $option {
            -info.* {
                set noPassOption 1
                regsub -- {-info} $option {} option
                set getInfo($option) $value
            }
            -block {
                set noPassOption 1
                set block $value
            }
            -bodyvar {
                upvar $value curlBodyVar
                set    value curlBodyVar
            }
            -headervar {
                upvar $value curlHeaderVar
                set    value curlHeaderVar
            }
            -errorbuffer {
                upvar $value curlErrorVar
                set    value curlErrorVar
            }
        }
        if {$noPassOption==0} {
            lappend newArgs $option $value
        }
    }

    if {[catch {::curl::init} curlHandle]} {
        error "Could not init a curl session: $curlHandle"
    }
    catch {::curl::setupCA $curlHandle}

    if {[catch {eval $curlHandle configure $newArgs} result]} {
        $curlHandle  cleanup
        error $result
    }

    if {$block==1} {
        if {[catch {$curlHandle perform} result]} {
           $curlHandle cleanup
           error $result
        }
        if {[info exists getInfo]} {
            foreach {option var} [array get getInfo] {
                upvar $var info
                set info [eval $curlHandle getinfo $option]
            }
        }
        if {[catch {$curlHandle cleanup} result]} {
            error $result
        }
    } else {
        # We create a multiHandle
        set multiHandle [curl::multiinit]

        # We add the easy handle to the multi handle.
        $multiHandle addhandle $curlHandle

        # So now we create the event source passing the multiHandle as a parameter.
        curl::createEventSource $multiHandle

        # And we return, it is non blocking after all.
    }
    return 0
}

}
