package require Canvas3d

if {[info command borg] eq "borg"} {

    # many devices have an orientation sensor,
    # only few a rotation_vector

    proc init_sensor {} {
	foreach s [borg sensor list] {
	    if {[dict get $s type] eq "orientation"} {
		set i [dict get $s index]
		borg sensor enable $i
		bind . <<SensorUpdate>> {handle_sensor %x}
		break
	    }
	}
    }

    proc handle_sensor {i} {
	lassign [dict get [borg sensor get $i] values] x y z
	if {![info exists ::SX]} {
	    set ::SX $x
	    set ::SY $y
	    set ::SZ $z
	    return
	}
	set dx [expr {$::SX - $x}]
	set dy [expr {$::SY - $y}]
	set dz [expr {$::SZ - $z}]
	set ::SX $x
	set ::SY $y
	set ::SZ $z
	set dorot 0
	set rx 0
	set ry 0
	if {abs($dx) > 4} {
	    set rx $dx
	    incr dorot
	}
	if {abs($dz) > 4} {
	    set ry $dz
	    incr dorot
	}
	if {$dorot} {
	    rot_obj .c $rx $ry
	}
    }

}

set largewin 1
catch {
    if {[winfo screenwidth .] >= 480 || [winfo screenheight .] > 480} {
	borg screenorientation landscape
	sdltk root 1200 600
    } else {
	set largewin 0
    }
    wm attributes . -fullscreen 1
    sdltk touchtranslate 0
    if {$largewin} {
	sdltk vrmode 2 0.31 1.05
    }
    sdltk screensaver off
    # delayed sensor start, why is this necessary?
    bind . <Configure> {
	bind . <Configure> {}
	after idle init_sensor
    }
}

proc read_mtl_file {array filename grey} {
    upvar $array mtl
    if {[catch {set data $::file($filename)}]} {
	return
    }
    set name {}
    foreach line [split $data "\n"] {
	switch [lindex $line 0] {
	    "newmtl" {
		set name [lindex $line 1]
	    }
	    "Ka" {
		if {[string length $name]} {
		    if {$grey} {
			set v [tcl::mathfunc::max {*}[lrange $line 1 end]]
			set v [list $v $v $v]
		    } else {
			set v [lrange $line 1 end]
		    }
		    lappend mtl($name) -ambient $v
		}
	    }
	    "Kd" {
		if {[string length $name]} {
		    if {$grey} {
			set v [tcl::mathfunc::max {*}[lrange $line 1 end]]
			set v [list $v $v $v]
		    } else {
			set v [lrange $line 1 end]
		    }
		    lappend mtl($name) -diffuse $v
		}
	    }
	    "Ks" {
		if {[string length $name]} {
		    if {$grey} {
			set v [tcl::mathfunc::max {*}[lrange $line 1 end]]
			set v [list $v $v $v]
		    } else {
			set v [lrange $line 1 end]
		    }
		    lappend mtl($name) -specular $v
		}
	    }
	}
    }
}

proc read_obj_file {c tags filename} {
    set line_count 0
    set vertex_count 0
    set face_count 0
    set group {}
    set object {}
    set currmtl {}
    set grey [$c cget -enableanaglyph]
    if {[catch {set data $::file($filename)}]} {
	return
    }
    foreach line [split $data "\n"] {
	switch [lindex $line 0] {
	    "g" {
		# Faces can be broken up into groups
		# We implement them as tags
		set group [lrange $line 1 end]
	    }
	    "o" {
		# Modes can be broken up into multiple objects
		# We implement them as tags
		set object [lrange $line 1 end]
	    }
	    "v" {
		set vertex_xyz([incr vertex_count]) [lrange $line 1 end]
	    }
	    "f" {
		set coords {}
		foreach point [lrange $line 1 end] {
		    set vertex [lindex [split $point /] 0]
		    lappend coords {*}$vertex_xyz($vertex)
		}
		$c create polygon ${coords} {*}$currmtl \
		    -tags [list {*}$tags {*}$group {*}$object]
	    }
	    "mtllib" {
		read_mtl_file mtl [lindex $line 1] $grey
	    }
	    "usemtl" {
		set name [lindex $line 1]
		set currmtl {}
		if {[info exists mtl($name)]} {
		    set currmtl $mtl($name)
		}
	    }
	}
    }
}

proc show_obj {} {
    .c delete all
    .c create light {0 0 10.0}
    .c configure -cameralocation {0.1 0 10} -cameracenter {0 0 0} \
	-cameraup {0 1 0}
    read_obj_file .c x "chicken.obj"
    .c transform -camera light {lookat all}
}

proc rot_obj {c rx ry} {
    $c transform -camera type(light) [list orbitup $ry orbitleft $rx]
}

proc toggle_anaglyph {c} {
    if {!$::largewin} {
	return
    }
    $c configure -enableanaglyph [expr {![$c cget -enableanaglyph]}]
    if {[$c cget -enableanaglyph]} {
	catch {sdltk vrmode 0}
    } else {
	catch {sdltk vrmode 2 0.31 1.05}
    }
    show_obj
}

if {$largewin} {
    canvas3d .c -background black -width 1200 -height 600 -enablestereo 1
    pack .c
} else {
    canvas3d .c -background black -enablestereo 0
    pack .c -side top -fill both -expand 1
}

bind . <Break> exit
bind . <Escape> exit
bind . <Key-a> {toggle_anaglyph .c}

bind .c <B1-Motion> {
    set ry [expr {360.0 * (%y  - $::Y) / [%W cget -height]}]
    set rx [expr {360.0 * (%x  - $::X) / [%W cget -width]}]
    rot_obj %W $rx $ry
    set ::X %x
    set ::Y %y
}

bind .c <1> {
    set ::X %x
    set ::Y %y
}

bind .c <Double-1> {toggle_anaglyph %W}

proc joy_rot_obj {c s x} {
    if {$s == 1 || $s == 3} {
	rot_obj $c [expr {($x / 20.0) / [$c cget -width]}] 0
    } elseif {$s == 2 || $s == 4} {
	rot_obj $c 0 [expr {($x / 20.0) / [$c cget -height]}]
    }
}

bind . <<JoystickMotion>> {
    joy_rot_obj .c %s %x
}

set file(chicken.mtl) {
# Blender3D MTL File: chicken.blend
# Material Count: 5
newmtl eyes
Ns 96.078431
Ka 0.000000 0.000000 0.000000
Kd 0.117363 0.117363 0.117363
Ks 0.500000 0.500000 0.500000
Ni 1.000000
d 1.000000
illum 2


newmtl feet
Ns 96.078431
Ka 0.000000 0.000000 0.000000
Kd 0.509348 0.389802 0.115481
Ks 0.500000 0.500000 0.500000
Ni 1.000000
d 1.000000
illum 2


newmtl beak
Ns 96.078431
Ka 0.000000 0.000000 0.000000
Kd 0.521597 0.352739 0.038392
Ks 0.500000 0.500000 0.500000
Ni 1.000000
d 1.000000
illum 2


newmtl flubber
Ns 96.078431
Ka 0.000000 0.000000 0.000000
Kd 0.460350 0.072652 0.072652
Ks 0.500000 0.500000 0.500000
Ni 1.000000
d 1.000000
illum 2


newmtl 02_-_Default
Ns 96.078431
Ka 0.000000 0.000000 0.000000
Kd 0.693039 0.633920 0.422487
Ks 0.500000 0.500000 0.500000
Ni 1.000000
d 1.000000
illum 2


}

set file(chicken.obj) {
# Blender3D v249 OBJ File: chicken.blend
# www.blender3d.org
mtllib chicken.mtl
v 0.034038 0.336502 -0.207066
v 0.002368 0.368957 -0.249688
v -0.015878 0.345512 -0.203677
v -0.087455 -0.184990 0.028988
v 0.020751 0.280472 -0.223484
v -0.116189 -0.008853 0.083153
v 0.066224 -0.332425 -0.038374
v -0.118637 -0.033103 0.104995
v 0.078726 -0.245316 0.003428
v -0.061926 0.107597 -0.170888
v 0.069751 0.092412 -0.067492
v -0.006159 0.270325 -0.083952
v 0.005510 0.136323 0.325650
v -0.034941 -0.316193 -0.008483
v 0.006193 0.324965 -0.292320
v 0.072705 -0.219535 0.033495
v -0.018897 0.099674 0.327705
v 0.052758 -0.331068 -0.003765
v 0.013657 -0.106682 -0.180133
v 0.043284 -0.330746 0.005108
v -0.071694 -0.251848 0.025811
v -0.084042 -0.131955 -0.032834
v 0.005929 0.341544 -0.275875
v -0.018491 0.344070 -0.210999
v 0.113157 -0.018361 -0.138992
v 0.034027 0.337078 -0.214636
v 0.033374 0.326178 -0.217323
v 0.045986 -0.110764 0.139410
v 0.071686 0.038595 0.211916
v -0.004055 0.291340 -0.235740
v -0.003847 0.375262 -0.148975
v -0.037284 0.132236 -0.079647
v -0.069044 -0.327295 -0.022381
v 0.002507 0.319543 -0.273534
v 0.099929 -0.104237 0.058256
v 0.018557 0.380909 -0.181428
v 0.055341 0.141723 -0.090996
v -0.030609 0.067374 0.077000
v -0.087625 0.038996 0.065307
v 0.060417 0.135979 -0.087573
v -0.055604 -0.320152 0.038628
v -0.030683 0.269390 -0.125598
v 0.005967 0.084164 0.019040
v -0.001672 -0.327029 -0.045622
v -0.057548 -0.312726 0.059116
v 0.060873 -0.334278 0.035628
v 0.123169 0.050471 -0.090398
v -0.024350 0.077718 0.122307
v 0.065627 0.024545 0.211141
v -0.029430 -0.103161 -0.167826
v 0.079001 -0.297208 0.033604
v -0.070997 -0.002312 0.188743
v -0.076387 0.006594 0.053575
v 0.074692 0.108151 -0.114545
v 0.033427 0.334079 -0.215406
v 0.012713 0.277662 -0.223639
v 0.081618 -0.093098 0.097536
v 0.106735 -0.337705 -0.051095
v 0.086546 -0.222970 0.010507
v 0.018093 0.270325 -0.083952
v 0.031488 0.074479 0.090290
v -0.022107 0.324442 -0.207433
v 0.040052 0.331510 -0.156740
v -0.090466 -0.090033 0.014560
v -0.022093 0.337078 -0.214636
v -0.045206 -0.338124 0.013247
v -0.023678 -0.161585 0.016543
v -0.074597 -0.303060 0.003302
v 0.031533 0.323111 -0.212540
v 0.076635 0.026734 -0.202349
v 0.083801 -0.101058 -0.058306
v 0.005933 0.293349 -0.255870
v -0.037833 0.078728 0.069429
v 0.082500 -0.337369 0.009722
v 0.005953 0.367355 -0.275310
v -0.043406 0.141723 -0.090996
v -0.057836 -0.339351 0.044924
v -0.022208 0.342343 -0.211063
v 0.017366 0.096321 0.340692
v 0.080211 -0.315595 -0.102796
v -0.060770 -0.219535 0.033495
v -0.007328 -0.318385 -0.035348
v -0.021439 0.326178 -0.217323
v -0.098708 -0.090184 -0.058489
v -0.005909 0.096515 0.340268
v 0.014894 0.360064 -0.243686
v 0.005967 0.121211 0.190218
v -0.031349 -0.330746 0.005108
v -0.004171 -0.325195 -0.040646
v 0.086313 -0.220392 0.030712
v 0.032771 0.334925 -0.200503
v 0.088321 0.006594 0.053576
v 0.082097 -0.154062 -0.042181
v -0.055532 -0.297438 -0.001088
v -0.090778 0.069745 -0.016701
v -0.026412 0.112314 -0.049649
v 0.031584 0.082500 0.147739
v -0.018173 -0.053554 -0.226374
v -0.036667 -0.318823 -0.012652
v -0.018249 0.378339 -0.176009
v -0.008048 0.007508 0.285081
v 0.011450 -0.068020 0.206278
v 0.009521 0.319737 -0.273472
v 0.034112 0.328170 -0.208027
v 0.100596 -0.042118 0.067529
v 0.030623 0.283870 -0.103171
v 0.030581 0.359528 -0.138149
v 0.057029 -0.191884 -0.011932
v 0.030890 0.375726 -0.171554
v 0.057536 -0.307264 0.004453
v -0.110609 -0.324138 -0.076013
v -0.084358 -0.181210 -0.003896
v 0.013594 0.268662 -0.224388
v -0.067066 -0.297208 0.033604
v -0.070565 -0.337369 0.009722
v 0.005746 0.408385 -0.199183
v -0.016555 0.319639 -0.203084
v -0.072559 -0.012206 0.167156
v 0.043674 0.138084 -0.088535
v 0.126930 -0.311262 -0.067727
v 0.026493 0.067619 0.335235
v 0.122544 -0.324138 -0.076013
v -0.082730 -0.017866 0.162090
v -0.027719 0.082812 0.131289
v 0.061275 0.128915 -0.134655
v 0.031725 0.096557 0.326967
v -0.015357 0.261919 -0.205324
v 0.092238 0.028715 -0.169768
v -0.017672 0.154884 -0.073451
v 0.100900 -0.318827 -0.016908
v 0.071344 -0.261013 0.031435
v -0.022757 0.329909 -0.204120
v 0.005967 0.163423 -0.071916
v 0.069366 0.049157 0.049385
v -0.049961 -0.316310 -0.011958
v 0.093139 0.074567 -0.175274
v -0.106737 -0.314596 -0.072322
v -0.122830 0.016744 -0.004884
v -0.017508 0.320885 -0.215227
v 0.118192 -0.308784 -0.046903
v 0.065518 -0.323993 -0.029590
v 0.095720 -0.068562 0.068925
v 0.005967 0.281230 -0.253760
v -0.029228 0.005361 -0.243214
v 0.005918 0.372132 -0.274035
v 0.081163 -0.125483 0.070576
v 0.066070 -0.237362 0.015104
v -0.110590 0.050471 -0.090399
v 0.086532 -0.303060 0.003302
v 0.004784 0.388531 -0.247505
v 0.061155 0.086531 0.051482
v -0.024658 -0.059156 0.203945
v 0.034454 0.070134 -0.247518
v -0.059410 -0.261013 0.031435
v -0.065019 -0.320075 -0.027236
v -0.021662 0.323526 -0.212286
v -0.085196 -0.025762 0.147355
v 0.072295 -0.211456 -0.001702
v -0.031461 0.088384 -0.010780
v -0.094421 -0.332961 -0.011757
v -0.035704 0.190977 -0.200296
v -0.052748 -0.107663 -0.112249
v -0.092609 -0.086826 0.021437
v -0.042703 0.066594 0.208294
v 0.072359 -0.060889 0.160716
v 0.091680 -0.337390 0.029882
v -0.031652 0.075460 0.025556
v -0.017443 -0.157842 -0.094316
v -0.017761 -0.017489 0.228744
v -0.019553 0.074479 0.090290
v 0.096929 -0.321481 -0.024020
v 0.095309 -0.017866 0.162090
v -0.070163 -0.154062 -0.042181
v 0.087623 -0.330431 -0.003507
v -0.074212 -0.177661 0.052392
v -0.087994 -0.104237 0.058256
v -0.028117 0.331510 -0.156740
v -0.007309 0.357676 -0.125266
v 0.016106 -0.325195 -0.040646
v -0.015229 0.114827 0.205036
v 0.005967 -0.091119 0.183653
v 0.072470 -0.294741 0.035694
v -0.063445 -0.324019 0.043753
v -0.021983 0.038858 0.289766
v -0.100578 -0.018361 -0.138992
v -0.060424 -0.060889 0.160716
v -0.084990 0.033380 -0.057307
v -0.080303 0.028715 -0.169768
v -0.091996 -0.123705 -0.009207
v 0.034408 0.028485 0.282944
v -0.055114 -0.234688 0.006110
v 0.028517 0.344483 -0.203781
v -0.079151 -0.308980 0.007828
v 0.005796 0.299037 -0.312285
v 0.069483 -0.312726 0.059116
v -0.019004 0.082500 0.147739
v -0.121724 -0.331309 -0.089430
v -0.026407 -0.163900 -0.011459
v -0.020622 0.322757 -0.208215
v 0.032782 0.333505 -0.223307
v 0.038477 -0.163949 -0.011449
v -0.022519 0.070134 -0.247518
v 0.084382 0.108298 -0.164177
v -0.084278 0.007673 -0.160183
v -0.059107 0.038595 0.211916
v 0.086147 -0.177661 0.052392
v -0.115236 -0.094151 -0.003396
v -0.023028 -0.329088 -0.032236
v 0.005967 0.402355 -0.231542
v 0.128769 -0.008853 0.083153
v -0.114405 -0.051588 0.102358
v -0.064701 0.026734 -0.202349
v -0.048576 0.086531 0.051482
v 0.073527 0.128688 -0.129476
v 0.009045 0.360513 -0.254909
v 0.013322 0.070291 0.344481
v -0.034051 -0.110764 0.139410
v -0.071208 -0.335627 0.041754
v 0.083629 -0.251848 0.025811
v 0.017636 -0.054713 -0.232824
v 0.075379 -0.324019 0.043753
v -0.020504 0.370551 -0.195870
v -0.021181 0.327138 -0.201714
v -0.062032 -0.326847 -0.120290
v -0.069228 -0.125483 0.070575
v -0.057816 0.092412 -0.067492
v -0.088632 -0.040574 -0.023633
v -0.037375 0.067553 0.131649
v 0.038347 0.112314 -0.049649
v -0.065181 0.113767 -0.147235
v 0.007006 0.384884 -0.251320
v 0.089797 0.038581 -0.172186
v 0.005981 0.270796 -0.241370
v 0.104543 -0.086826 0.021437
v 0.005948 0.348626 -0.276577
v 0.096858 0.007674 -0.160183
v -0.047838 0.135979 -0.087573
v 0.063731 -0.301233 0.024694
v 0.111288 -0.090184 -0.058489
v -0.089114 -0.095860 -0.014670
v 0.108810 -0.114196 0.025798
v -0.011809 -0.146262 0.100830
v 0.095976 -0.131955 -0.032834
v 0.103045 -0.090033 0.014561
v 0.048601 -0.318823 -0.012652
v 0.018207 -0.169144 -0.051979
v 0.051282 0.136555 -0.102399
v 0.050413 0.078728 0.069429
v 0.021778 0.347618 -0.211219
v 0.005973 0.347821 -0.276545
v -0.057061 -0.314400 -0.103322
v 0.085210 0.010057 0.205177
v 0.139249 -0.038033 -0.004855
v 0.064990 -0.323738 -0.103890
v 0.032411 0.319697 -0.211067
v 0.013606 -0.327029 -0.045622
v -0.012657 0.058696 0.309583
v 0.067466 -0.297438 -0.001088
v 0.018379 0.135255 0.281520
v -0.021333 0.331909 -0.219049
v 0.100204 0.038996 0.065307
v 0.019969 -0.320806 -0.040092
v 0.043189 0.067374 0.077000
v 0.027769 0.325096 -0.220103
v -0.034420 0.263483 -0.162437
v -0.042309 0.049109 0.206779
v -0.053047 0.024545 0.211141
v 0.060370 0.080788 0.085590
v 0.122917 -0.095600 -0.028266
v 0.002425 0.372507 -0.206590
v 0.021982 -0.145316 0.103388
v 0.038508 0.303712 -0.178651
v 0.027865 0.336463 -0.195080
v 0.036809 -0.154991 0.044241
v -0.016234 0.339624 -0.220553
v -0.018688 0.283870 -0.103171
v 0.002723 0.317787 -0.291290
v 0.027292 0.261919 -0.205324
v 0.083495 -0.323712 -0.103500
v -0.080560 0.074567 -0.175274
v 0.067048 -0.234688 0.006110
v 0.131883 -0.325202 -0.066416
v -0.074611 -0.222970 0.010507
v 0.087140 -0.075649 -0.088092
v 0.046354 0.263483 -0.162437
v 0.034691 0.329908 -0.204120
v -0.091996 0.024315 -0.167524
v -0.014570 0.090193 0.144979
v -0.078558 -0.090076 -0.057344
v 0.091137 -0.090076 -0.057344
v 0.006189 0.408385 -0.199183
v 0.011545 0.008182 0.292184
v -0.072631 0.010057 0.205177
v 0.129967 -0.324450 -0.072133
v -0.066928 -0.334720 -0.029316
v 0.050904 -0.167467 -0.044726
v 0.118672 -0.314596 -0.072322
v 0.054889 0.049109 0.206779
v 0.043586 0.075460 0.025556
v 0.033596 0.323526 -0.212286
v 0.041162 0.005361 -0.243214
v -0.037617 0.083017 0.144929
v -0.015431 0.121054 0.289608
v -0.032047 0.334655 -0.188293
v 0.068790 0.041086 0.166858
v 0.005957 0.305182 -0.290117
v -0.095720 0.008665 -0.112923
v 0.075864 -0.326271 -0.107584
v 0.005967 0.069584 -0.256747
v -0.061885 -0.043083 -0.190029
v 0.005967 0.120488 -0.045222
v 0.027608 0.204885 -0.218413
v -0.071863 -0.317685 -0.068044
v 0.079309 -0.307546 -0.004226
v 0.099231 -0.020539 -0.141060
v 0.005957 0.364492 -0.276036
v -0.118033 -0.324450 -0.072133
v 0.118433 0.023789 -0.132762
v 0.044041 0.088384 -0.010780
v -0.049298 -0.220612 0.027630
v 0.036929 0.077718 0.122307
v 0.016994 0.371694 -0.140969
v 0.071024 -0.312344 -0.065026
v -0.105854 0.023789 -0.132762
v 0.101694 -0.095860 -0.014670
v 0.050197 0.083017 0.144929
v -0.005791 0.357892 -0.240166
v 0.005969 0.079932 0.316141
v -0.083140 -0.068562 0.068925
v 0.097776 -0.025762 0.147355
v -0.016327 0.339268 -0.196540
v -0.060947 0.128688 -0.129476
v -0.026573 0.303712 -0.178650
v -0.054290 -0.332425 -0.038373
v 0.011672 0.369550 -0.210656
v 0.005973 0.367221 -0.275396
v 0.005967 0.407251 -0.195810
v 0.023509 0.362851 -0.217112
v 0.034142 0.342343 -0.211063
v 0.103358 0.069745 -0.016702
v 0.024689 0.060357 0.311732
v -0.096985 0.000981 0.157991
v 0.099696 -0.027664 -0.142028
v 0.057568 -0.098019 -0.153402
v 0.006012 0.403657 -0.216291
v 0.009772 0.318716 -0.280282
v -0.077218 0.038581 -0.172186
v 0.084982 -0.199570 -0.004710
v -0.025597 0.080121 0.157453
v 0.026382 -0.021753 0.228305
v -0.065683 -0.317850 0.058738
v 0.131217 -0.033104 0.104995
v 0.083577 -0.002312 0.188743
v 0.005832 0.299170 -0.312542
v 0.034880 0.329463 -0.200372
v -0.097699 -0.021071 0.156992
v 0.040299 0.082812 0.131289
v -0.048696 0.128915 -0.134655
v 0.028124 0.336323 -0.221558
v -0.084995 -0.321481 -0.024020
v 0.043600 0.069759 0.049319
v 0.133658 -0.331309 -0.089430
v 0.126985 -0.051588 0.102358
v -0.021913 0.336369 -0.214406
v 0.029762 0.343716 -0.213583
v 0.044911 0.050200 0.271147
v 0.049309 0.067553 0.131649
v 0.054143 -0.032899 0.210341
v -0.071866 -0.101059 -0.058306
v -0.088910 -0.071723 -0.093229
v 0.061232 -0.220612 0.027630
v 0.106356 -0.332961 -0.011757
v -0.010359 -0.330892 -0.022344
v 0.019263 -0.318385 -0.035348
v 0.030587 -0.161237 -0.086637
v -0.048938 -0.334278 0.035628
v -0.020848 0.333504 -0.223307
v -0.110337 -0.095600 -0.028266
v -0.079746 -0.337390 0.029882
v 0.004419 0.141627 0.239632
v -0.017026 0.328985 -0.195531
v -0.054135 -0.237362 0.015104
v 0.135409 0.016744 -0.004884
v 0.027163 0.114827 0.205036
v -0.023194 0.332570 -0.206993
v -0.067374 -0.307546 -0.004226
v 0.074505 0.107597 -0.170887
v -0.051797 -0.301233 0.024694
v -0.030591 0.108799 -0.054112
v 0.067539 -0.320152 0.038628
v -0.094800 -0.337705 -0.051095
v 0.029606 0.154885 -0.073451
v -0.001824 0.267412 -0.224226
v -0.020477 0.319697 -0.211066
v 0.032438 0.370551 -0.195870
v -0.096875 -0.114196 0.025798
v -0.106257 -0.308784 -0.046903
v -0.030937 0.099115 -0.037150
v 0.077115 0.113769 -0.147235
v -0.126669 -0.038033 -0.004855
v 0.103931 -0.123705 -0.009207
v 0.101489 -0.071723 -0.093229
v 0.096293 -0.181210 -0.003896
v 0.046876 -0.316193 -0.008483
v -0.068276 -0.315595 -0.102796
v 0.013333 0.349628 -0.116855
v -0.060536 -0.294741 0.035694
v -0.071802 0.108298 -0.164177
v -0.040737 -0.153043 0.068605
v -0.049874 0.056999 0.199525
v 0.039265 0.096317 0.256850
v 0.038176 0.080121 0.157453
v 0.073967 -0.326847 -0.120290
v -0.040709 -0.167322 -0.044673
v -0.000289 0.277177 -0.223693
v -0.040823 -0.331068 -0.003765
v 0.005741 0.276629 -0.216090
v -0.028344 -0.160248 0.044086
v -0.053056 -0.323738 -0.103890
v 0.008769 0.317726 -0.286154
v -0.005562 -0.154111 0.024999
v -0.071560 -0.323712 -0.103500
v -0.066791 -0.245316 0.003428
v -0.086651 -0.020539 -0.141060
v -0.062757 0.108152 -0.114545
v 0.049863 0.132236 -0.079647
v 0.006088 0.306448 -0.284021
v 0.043516 0.099115 -0.037150
v -0.119948 -0.325202 -0.066416
v -0.008816 0.280471 -0.223484
v 0.086408 -0.309989 0.022956
v -0.031020 0.069759 0.049319
v 0.068996 -0.314400 -0.103322
v -0.045094 -0.191884 -0.011932
v 0.033060 0.325330 -0.221220
v 0.007603 0.306652 -0.303064
v -0.075689 -0.330431 -0.003507
v 0.127816 -0.094151 -0.003396
v 0.080979 -0.327295 -0.022381
v 0.069771 -0.339351 0.044924
v -0.087761 -0.027664 -0.142028
v -0.015673 0.204885 -0.218413
v 0.061896 -0.316310 -0.011958
v 0.033268 0.331909 -0.219049
v 0.042617 0.269390 -0.125598
v 0.005967 0.272728 -0.079925
v 0.043982 0.334655 -0.188293
v 0.077618 -0.317850 0.058738
v -0.114084 -0.077455 0.049774
v 0.047639 0.190978 -0.200296
v 0.009584 0.038697 0.306195
v 0.006504 0.379688 -0.197941
v -0.022475 0.329221 -0.212843
v 0.104575 0.024315 -0.167524
v 0.096924 0.033380 -0.057308
v -0.074474 -0.309989 0.022956
v 0.083797 -0.317685 -0.068043
v 0.005967 0.001573 -0.256096
v 0.023429 0.088428 0.137527
v 0.015989 0.291340 -0.235740
v 0.062454 0.056999 0.199525
v 0.125068 -0.334629 -0.062218
v -0.088965 -0.318827 -0.016909
v 0.022294 -0.330892 -0.022344
v -0.018646 0.359528 -0.138149
v -0.057431 0.049157 0.049385
v 0.078863 -0.334720 -0.029316
v -0.088662 -0.042118 0.067529
v 0.017839 0.126440 0.321436
v 0.006176 0.298344 -0.311869
v -0.114995 -0.311262 -0.067727
v 0.005967 0.409267 -0.233998
v -0.032976 0.050200 0.271147
v 0.091086 -0.308980 0.007828
v 0.043171 0.108799 -0.054112
v 0.012460 -0.028187 0.225501
v -0.016947 0.329160 -0.221790
v -0.059090 -0.312344 -0.065026
v -0.022946 0.329463 -0.200372
v -0.063929 -0.326271 -0.107584
v 0.005967 0.075587 0.060652
v -0.022845 0.336293 -0.207658
v 0.035613 -0.161585 0.016543
v 0.029717 0.323029 -0.199959
v -0.031094 0.138084 -0.088535
v -0.074560 -0.075649 -0.088092
v 0.107655 0.008665 -0.112923
v 0.126664 -0.077455 0.049774
v -0.060360 -0.211456 -0.001702
v 0.057141 -0.338124 0.013247
v 0.034962 -0.329088 -0.032236
v 0.110279 -0.021071 0.156992
v -0.021876 0.337603 -0.202802
v -0.113133 -0.334629 -0.062218
v -0.073048 -0.199570 -0.004710
v 0.027307 0.318569 -0.206329
v 0.099389 -0.184990 0.028988
v -0.003391 0.071074 0.329279
v -0.045601 -0.307264 0.004453
v 0.005967 0.244707 -0.226705
v 0.100566 -0.040574 -0.023633
v 0.109564 0.000981 0.157991
v 0.083143 -0.335627 0.041754
v 0.005967 0.178583 -0.231481
v -0.047791 0.080788 0.085590
v -0.021125 0.325330 -0.221220
v 0.084494 -0.012206 0.167156
v 0.076954 -0.320075 -0.027236
v -0.008034 -0.320806 -0.040092
v -0.069684 -0.093098 0.097536
v -0.053583 -0.323993 -0.029590
v 0.034779 0.336293 -0.207658
v 0.073819 -0.043083 -0.190029
v 0.034041 0.324441 -0.207433
v -0.074378 -0.220392 0.030712
v 0.052687 -0.152837 0.068734
v -0.038703 0.136555 -0.102399
v -0.042495 -0.008247 0.223072
v 0.002823 0.314817 -0.291392
v -0.010557 0.364126 -0.216556
v -0.015412 0.069051 0.336768
vn 0.954283 0.293466 0.056307
vn -0.981597 0.122044 -0.146764
vn -0.740562 0.595935 0.310465
vn -0.962188 -0.218848 0.162084
vn 0.618397 -0.276009 -0.735771
vn -0.931852 0.330668 0.149297
vn -0.758354 -0.626331 -0.180425
vn -0.975890 -0.042329 0.213965
vn 0.207678 -0.066713 -0.975890
vn -0.183752 0.613208 -0.768242
vn 0.806482 0.504746 0.307840
vn -0.493576 0.206397 0.844813
vn -0.105014 0.823939 0.556810
vn 0.325785 0.932005 -0.158666
vn 0.605487 0.513535 -0.607959
vn 0.017731 -0.298257 0.954314
vn -0.880581 0.265145 0.392743
vn 0.445204 -0.721000 -0.530961
vn 0.060427 -0.782495 -0.619678
vn -0.852962 -0.014191 0.521714
vn -0.857204 0.015931 0.514695
vn -0.911069 -0.065279 -0.407025
vn -0.462264 0.520585 -0.717826
vn -0.839412 0.533433 -0.103824
vn 0.707816 -0.440840 -0.551897
vn 0.989776 0.037690 -0.137455
vn 0.995697 -0.017945 -0.090884
vn 0.536271 -0.661367 0.524338
vn 0.439161 0.346141 0.829005
vn -0.846126 -0.353313 -0.398999
vn -0.216132 0.898923 0.381054
vn 0.291971 0.887936 0.355358
vn -0.988342 -0.021485 -0.150670
vn -0.961211 0.041963 -0.272561
vn 0.903012 -0.122227 0.411786
vn 0.129917 0.981445 -0.140934
vn 0.789788 0.298715 0.535661
vn 0.772149 0.620685 -0.136052
vn -0.782647 0.595447 0.181219
vn 0.303812 0.936796 0.173406
vn 0.841182 -0.146489 0.520463
vn -0.954039 0.176702 0.241951
vn 0.000000 0.925718 0.378185
vn 0.863033 0.404584 -0.302347
vn 0.531693 0.384289 0.754692
vn -0.837153 0.092257 0.539079
vn 0.936125 0.344615 -0.069613
vn 0.528275 0.808222 -0.260079
vn -0.158605 -0.301126 0.940275
vn -0.529344 -0.670492 -0.519791
vn 0.608814 0.351665 0.711081
vn 0.060030 -0.794671 0.604053
vn -0.938353 0.337565 0.074068
vn 0.930082 0.310800 0.195776
vn 0.919553 0.030580 -0.391705
vn 0.479263 -0.263680 -0.837092
vn 0.817042 -0.390820 0.423841
vn -0.650655 -0.642537 -0.404675
vn 0.941984 -0.190039 -0.276650
vn 0.491562 0.205115 0.846309
vn 0.430830 0.900632 -0.056520
vn 0.013062 -0.814112 0.580493
vn 0.989166 0.119114 0.085665
vn 0.001617 -0.957854 0.287179
vn -0.989776 0.037690 -0.137455
vn 0.252358 -0.961638 -0.107334
vn 0.755669 -0.649098 0.087191
vn -0.675619 0.431928 -0.597400
vn 0.893490 -0.427015 -0.138890
vn 0.832789 0.018677 -0.553240
vn 0.817988 -0.272744 -0.506424
vn -0.000366 -0.564989 -0.825068
vn 0.672506 0.735282 0.084017
vn 0.241676 -0.946379 -0.214331
vn -0.975951 0.050722 -0.211951
vn -0.789788 0.298715 0.535691
vn 0.278420 -0.798395 0.533860
vn -0.950621 0.165929 -0.262185
vn 0.436964 0.259438 0.861232
vn 0.637471 0.671957 -0.376904
vn -0.017640 -0.298257 0.954314
vn 0.596576 0.792688 0.125217
vn -0.995666 -0.017975 -0.090884
vn -0.261422 -0.886471 -0.381817
vn -0.326609 0.217627 0.919736
vn 0.886929 0.366985 -0.280435
vn -0.017060 0.906766 -0.421216
vn 0.852962 -0.014161 0.521714
vn 0.428205 -0.853603 -0.296548
vn 0.739982 -0.316660 0.593371
vn 0.893246 0.124027 0.432051
vn 0.940794 0.330241 0.075991
vn 0.590899 -0.381207 -0.710959
vn 0.130741 0.346873 -0.928739
vn -0.774743 0.607501 0.175207
vn -0.433699 0.690451 0.578906
vn -0.187292 0.978973 -0.080721
vn -0.379650 -0.521592 -0.764031
vn -0.455550 0.507492 -0.731376
vn -0.641377 0.767144 0.010376
vn -0.470779 -0.646260 0.600543
vn 0.067934 -0.562029 0.824305
vn 0.958647 -0.001526 -0.284524
vn 0.976653 -0.204657 0.064913
vn 0.988189 0.046297 0.146001
vn 0.798181 0.240547 0.552263
vn 0.805200 0.403394 0.434584
vn -0.448286 -0.570910 -0.687796
vn 0.773247 0.620655 0.129795
vn -0.915464 0.401990 -0.017060
vn 0.459548 -0.703665 -0.541887
vn -0.878719 -0.284524 -0.383251
vn 0.283761 0.010987 -0.958800
vn -0.608814 0.351665 0.711081
vn -0.241676 -0.946379 -0.214331
vn 0.991089 0.131474 -0.020905
vn -0.701498 -0.649525 0.293191
vn -0.962157 -0.027833 0.270974
vn 0.172704 0.873745 0.454634
vn 0.704794 0.684133 -0.187475
vn 0.354839 -0.772881 0.526048
vn -0.459548 -0.703665 -0.541887
vn -0.074892 -0.843654 0.531602
vn 0.307016 0.946989 -0.094424
vn 0.153874 0.933775 -0.323069
vn 0.899930 0.253243 0.354930
vn -0.891842 -0.016602 -0.451979
vn 0.941221 0.066073 -0.331248
vn -0.421339 0.320811 0.848231
vn 0.184973 0.973327 -0.135624
vn -0.393292 0.013398 0.919279
vn -0.996307 -0.063234 -0.057772
vn 0.000000 0.315744 0.948820
vn 0.734123 0.656911 0.171697
vn 0.367992 0.524827 -0.767541
vn 0.559618 0.171606 -0.810755
vn 0.687002 0.482070 -0.543657
vn -0.923185 0.368389 0.109439
vn -0.783135 -0.555834 -0.278787
vn 0.266884 0.934446 0.235694
vn -0.836085 0.541887 -0.085360
vn -0.026276 -0.897031 0.441115
vn -0.000275 -0.499222 -0.866451
vn -0.598529 -0.159795 -0.784997
vn 0.318522 0.709067 -0.629078
vn 0.570025 -0.380566 0.728141
vn -0.980956 -0.194006 -0.008057
vn -0.936125 0.344615 -0.069613
vn 0.675619 0.431928 -0.597400
vn -0.883969 0.367779 -0.288644
vn 0.158391 0.941465 0.297525
vn -0.408277 -0.568834 0.713920
vn 0.522935 0.112583 -0.844874
vn 0.393292 0.013398 0.919279
vn -0.281442 0.950591 -0.130955
vn -0.951170 -0.306467 -0.036164
vn 0.035127 -0.879879 0.473861
vn -0.044252 -0.283181 -0.958037
vn 0.958861 -0.269051 -0.090396
vn -0.792566 -0.538163 0.286660
vn -0.860530 0.183935 -0.474990
vn -0.782495 -0.532334 -0.322947
vn -0.991668 0.111332 -0.064638
vn -0.927152 0.369518 0.061708
vn 0.799066 -0.423170 0.427045
vn 0.782434 -0.506516 0.362194
vn -0.307718 0.921323 0.237495
vn -0.344218 -0.830683 -0.437513
vn -0.461654 -0.560106 0.687826
vn -0.353313 0.931791 -0.082919
vn -0.799585 0.564959 -0.203589
vn 0.074892 -0.843654 0.531602
vn -0.594378 -0.379833 -0.708792
vn -0.415876 -0.570666 -0.708060
vn -0.499496 -0.328593 0.801569
vn -0.903012 -0.122227 0.411786
vn -0.988861 0.122868 0.083804
vn -0.372295 0.570482 0.732047
vn -0.428175 -0.853603 -0.296548
vn -0.698294 0.688070 -0.197272
vn 0.052217 -0.764061 0.642994
vn -0.272805 0.480056 0.833705
vn -0.237098 -0.211951 0.948057
vn -0.659505 -0.557054 0.504685
vn -0.707816 -0.440840 -0.551897
vn -0.776452 -0.430830 0.459822
vn -0.940947 0.306131 0.144383
vn -0.941221 0.066073 -0.331248
vn -0.976592 0.003296 -0.215033
vn 0.599628 -0.546220 0.584826
vn 0.584552 -0.108249 -0.804071
vn 0.742180 0.633869 0.217627
vn -0.934385 0.334147 0.123447
vn 0.054048 0.600879 -0.797479
vn -0.531693 0.384289 0.754692
vn 0.187292 0.978973 -0.080721
vn -0.058382 -0.105564 -0.992676
vn 0.466994 -0.882260 -0.058992
vn -0.873287 -0.486557 0.024323
vn 0.956053 0.100345 -0.275460
vn -0.572222 -0.818110 -0.056825
vn -0.522935 0.112583 -0.844874
vn 0.763787 0.521226 -0.380688
vn -0.139195 -0.453017 -0.880551
vn -0.439161 0.346141 0.829005
vn 0.499863 -0.328501 0.801355
vn -0.515946 -0.848842 0.115024
vn -0.503494 -0.507584 -0.699118
vn 0.012360 0.919889 0.391949
vn 0.931852 0.330668 0.149297
vn -0.560076 -0.699179 0.444319
vn -0.832789 0.018677 -0.553240
vn -0.158391 0.941465 0.297525
vn 0.503433 0.849239 -0.159001
vn 0.957915 0.117832 -0.261696
vn 0.075991 -0.556047 0.827662
vn -0.524613 -0.675436 0.518174
vn -0.636647 -0.071688 0.767785
vn 0.857204 0.015931 0.514695
vn 0.306497 -0.533769 -0.788110
vn 0.237098 -0.211951 0.948057
vn -0.798151 0.502823 -0.331767
vn -0.876431 -0.280129 0.391583
vn -0.056581 -0.105258 -0.992828
vn -0.569781 -0.380627 0.728294
vn -0.806482 0.504746 0.307840
vn -0.998779 0.007508 -0.048311
vn -0.741539 0.664846 -0.089633
vn 0.433699 0.690451 0.578906
vn -0.972289 0.224952 -0.063417
vn 0.881191 0.336528 -0.331980
vn -0.061434 -0.182745 -0.981231
vn 0.005615 -0.986694 -0.162328
vn 0.991668 0.111332 -0.064638
vn -0.007538 -0.080355 -0.996734
vn 0.139195 -0.453017 -0.880551
vn -0.303812 0.936796 0.173406
vn -0.929136 0.116306 0.350902
vn 0.261422 -0.886471 -0.381817
vn 0.017823 -0.999786 -0.007782
vn 0.998108 0.015259 0.059358
vn -0.162725 -0.867519 0.469955
vn 0.911069 -0.065279 -0.407025
vn -0.001617 -0.957854 0.287179
vn 0.455519 0.507492 -0.731376
vn 0.008087 -0.999817 -0.016327
vn 0.164922 0.986145 0.016236
vn -0.672506 0.735282 0.084017
vn 0.435408 0.900082 -0.015168
vn -0.992920 -0.020997 -0.116733
vn 0.588855 0.760582 -0.273293
vn 0.407727 -0.478927 0.777398
vn 0.997497 -0.054689 -0.044252
vn -0.917722 -0.366008 -0.154180
vn 0.948729 -0.229621 -0.217170
vn -0.863063 0.404584 -0.302347
vn 0.103519 -0.658071 0.745781
vn -0.130741 0.346873 -0.928739
vn 0.748527 0.661184 0.050111
vn -0.991943 -0.027467 -0.123447
vn 0.782647 0.595447 0.181219
vn 0.499557 0.385327 -0.775811
vn -0.772149 0.620685 -0.136052
vn 0.789117 -0.191992 -0.583422
vn -0.971007 0.137364 -0.195532
vn 0.138707 0.518906 0.843471
vn 0.158605 -0.301126 0.940275
vn -0.482772 0.875118 -0.032655
vn 0.763115 -0.607379 -0.220649
vn -0.966399 0.034486 0.254585
vn 0.128819 -0.897549 0.421613
vn 0.961913 -0.083682 -0.260109
vn 0.754295 0.179418 0.631489
vn -0.618244 -0.762749 0.189550
vn -0.774041 0.330210 -0.540147
vn -0.804987 0.234840 0.544786
vn -0.966399 0.163366 -0.198370
vn 0.890683 -0.019684 -0.454146
vn 0.915738 -0.179876 -0.359203
vn -0.559618 0.171606 -0.810755
vn -0.584552 -0.108249 -0.804071
vn 0.991150 -0.058596 0.119022
vn -0.941984 -0.190039 -0.276650
vn -0.012238 -0.833583 -0.552232
vn 0.971007 0.137364 -0.195532
vn 0.996307 -0.063265 -0.057802
vn -0.491348 -0.273782 -0.826777
vn -0.372234 0.862911 -0.341716
vn 0.021027 -0.959868 -0.279550
vn -0.021027 -0.959868 -0.279550
vn -0.993713 0.109195 -0.023957
vn 0.169042 -0.657857 0.733879
vn -0.407727 -0.478927 0.777398
vn 0.804559 -0.550493 -0.222724
vn -0.485733 -0.870052 -0.083834
vn 0.290994 -0.820643 -0.491775
vn -0.687033 0.482070 -0.543657
vn -0.138707 0.518906 0.843471
vn 0.307718 0.921323 0.237495
vn 0.951231 -0.306284 -0.036256
vn 0.590777 -0.163396 -0.790094
vn -0.209174 0.965728 0.153569
vn -0.824274 0.562822 0.061281
vn -0.982391 0.002441 -0.186743
vn 0.900052 0.435377 0.017365
vn -0.950835 -0.298227 0.083224
vn -0.990905 0.131382 -0.028535
vn 0.263649 -0.917478 -0.297769
vn 0.000000 0.109104 -0.994018
vn -0.799738 -0.329142 -0.502029
vn 0.000000 0.720878 0.693045
vn 0.581622 0.127018 -0.803461
vn -0.889553 0.453780 0.052370
vn 0.301492 0.600299 -0.740745
vn 0.173437 -0.601581 -0.779717
vn 0.979308 -0.006867 -0.202124
vn -0.804559 -0.550493 -0.222694
vn 0.965484 0.063875 -0.252449
vn -0.958861 -0.269051 -0.090396
vn 0.813898 -0.360088 0.455947
vn -0.528306 0.808222 -0.260079
vn 0.217872 0.850703 0.478347
vn -0.078555 0.993652 0.080264
vn -0.965484 0.063875 -0.252449
vn -0.017823 -0.999786 -0.007782
vn 0.209174 0.965728 0.153569
vn -0.821192 0.462203 -0.334574
vn 0.000092 -0.840052 0.542497
vn 0.026276 -0.897031 0.441115
vn -0.035127 -0.879879 0.473861
vn -0.744835 0.353038 0.566149
vn -0.503433 0.849239 -0.159001
vn -0.961913 -0.083682 -0.260109
vn 0.758354 -0.626362 -0.180425
vn 0.840358 0.506149 0.193823
vn -0.369549 -0.252052 0.894345
vn 0.235908 0.439039 0.866909
vn 0.567217 0.721580 -0.396893
vn 0.950957 0.160558 -0.264290
vn 0.774743 0.607501 0.175207
vn -0.352947 -0.615772 0.704428
vn -0.898495 0.336497 0.281838
vn 0.930387 -0.284188 -0.231513
vn 0.652913 -0.606281 -0.453993
vn 0.827387 0.560594 0.033082
vn 0.993194 0.106326 -0.047182
vn 0.061434 -0.182745 -0.981231
vn 0.415235 -0.471145 -0.778161
vn 0.008423 0.952116 0.305551
vn 0.515458 -0.650990 0.557176
vn -0.491226 -0.469131 0.733879
vn 0.975890 -0.042329 0.213965
vn -0.060030 -0.794671 0.604053
vn 0.115848 -0.790521 0.601306
vn 0.973052 -0.147954 -0.176672
vn -0.466811 -0.668844 0.578509
vn -0.307016 0.946989 -0.094424
vn -0.153874 0.933775 -0.323069
vn 0.774895 0.143895 -0.615467
vn 0.799585 0.564959 -0.203589
vn -0.969726 0.242500 0.028230
vn 0.058351 -0.105564 -0.992676
vn 0.560076 -0.699179 0.444319
vn -0.919034 0.235420 -0.316050
vn 0.786370 0.593097 -0.172674
vn 0.909177 -0.177587 0.376598
vn 0.730277 0.677328 -0.088687
vn 0.520951 -0.533860 0.666005
vn -0.832942 -0.306406 -0.460738
vn -0.229621 -0.734825 -0.638142
vn -0.814020 -0.364391 0.452223
vn 0.792566 -0.538163 0.286660
vn 0.565142 -0.808863 0.162175
vn -0.596606 0.792688 0.125217
vn 0.190161 -0.913297 -0.360118
vn 0.837153 0.092257 0.539079
vn -0.965514 0.076174 -0.248909
vn -0.763115 -0.607379 -0.220649
vn -0.782434 -0.506516 0.362194
vn -0.101444 0.983215 -0.151494
vn -0.774621 -0.133488 0.618122
vn 0.980895 -0.194220 -0.007874
vn 0.923185 0.368389 0.109439
vn 0.705130 0.684713 -0.184118
vn -0.997162 -0.047487 0.058016
vn -0.301492 0.600299 -0.740745
vn 0.183752 0.613208 -0.768242
vn 0.929136 0.116306 0.350902
vn 0.487686 0.703085 0.517472
vn -0.841182 -0.146519 0.520463
vn 0.650655 -0.642537 -0.404675
vn 0.421339 0.320811 0.848231
vn -0.350993 -0.002594 -0.936369
vn -0.948729 -0.229621 -0.217170
vn 0.761742 0.573626 -0.301126
vn -0.998108 0.015259 0.059358
vn -0.266884 0.934446 0.235694
vn 0.948759 0.283181 0.140080
vn 0.972289 0.224952 -0.063417
vn -0.997497 -0.054689 -0.044252
vn 0.976592 0.003296 -0.215033
vn 0.229621 -0.734825 -0.638142
vn 0.878719 -0.284524 -0.383251
vn -0.325785 0.932005 -0.158666
vn -0.637471 0.671957 -0.376904
vn 0.350261 0.505020 0.788812
vn 0.272805 0.480056 0.833705
vn -0.763787 0.521226 -0.380688
vn -0.025697 -0.821528 0.569567
vn -0.470595 0.771508 0.428114
vn 0.934385 0.344646 0.090152
vn -0.008423 0.952116 0.305551
vn 0.056581 -0.105228 -0.992828
vn -0.178594 -0.853542 -0.489425
vn -0.397046 -0.197119 -0.896359
vn -0.445204 -0.721000 -0.530961
vn -0.018677 -0.845698 0.533280
vn 0.697165 -0.631428 0.339457
vn 0.917722 -0.366008 -0.154180
vn 0.962920 0.178594 -0.202124
vn 0.205664 -0.960997 0.184729
vn -0.915738 -0.179876 -0.359203
vn -0.207678 -0.066713 -0.975890
vn -0.173437 -0.601581 -0.779717
vn -0.930082 0.310800 0.195776
vn -0.291971 0.887936 0.355358
vn -0.002564 -0.995849 0.090762
vn -0.948759 0.283181 0.140080
vn -0.991150 -0.058596 0.119022
vn -0.628407 -0.272958 -0.728385
vn 0.906339 0.107547 0.408582
vn 0.969726 0.242500 0.028230
vn -0.588855 0.760582 -0.273293
vn 0.442854 -0.566546 -0.694876
vn 0.960173 -0.189611 -0.205145
vn 0.966277 0.141392 -0.215155
vn 0.415876 -0.570666 -0.708029
vn 0.515976 -0.848811 0.115024
vn 0.988342 -0.021485 -0.150670
vn -0.278420 -0.798395 0.533860
vn -0.940642 -0.254097 -0.224921
vn -0.581439 0.127476 -0.803522
vn -0.367992 0.524827 -0.767510
vn 0.991943 -0.027467 -0.123417
vn 0.954039 0.176702 0.241951
vn -0.040132 0.289071 0.956450
vn 0.982604 0.000153 -0.185614
vn 0.491256 -0.469131 0.733848
vn -0.625202 -0.723746 0.291940
vn 0.860530 0.183935 -0.474990
vn 0.153539 -0.366131 0.917783
vn 0.601337 -0.336131 0.724815
vn -0.931639 -0.215369 -0.292642
vn 0.491348 -0.273782 -0.826777
vn 0.940947 0.306131 0.144383
vn -0.906339 0.107547 0.408612
vn 0.889553 0.453780 0.052370
vn -0.026643 -0.166631 -0.985626
vn 0.253609 0.920194 -0.298166
vn 0.844234 -0.354137 -0.402264
vn 0.470595 0.771508 0.428114
vn 0.440077 -0.861568 -0.252937
vn -0.184973 0.973327 -0.135624
vn -0.565142 -0.808863 0.162175
vn -0.807031 0.430494 0.404126
vn -0.734123 0.656911 0.171697
vn 0.485733 -0.870052 -0.083834
vn -0.988189 0.046297 0.146001
vn 0.727470 0.533708 0.431166
vn 0.923887 -0.257607 -0.282846
vn -0.704794 0.684133 -0.187475
vn -0.461592 0.570940 -0.678915
vn -0.905179 -0.184942 0.382672
vn 0.934385 0.334147 0.123447
vn -0.487686 0.703085 0.517472
vn 0.049715 -0.731529 0.679983
vn -0.821650 -0.087741 -0.563128
vn 0.078555 0.993652 0.080233
vn -0.973052 -0.147984 -0.176641
vn -0.263649 -0.917478 -0.297769
vn 0.019868 0.999664 0.015503
vn -0.997650 0.068117 0.005310
vn -0.992035 -0.123997 0.021790
vn 0.856807 -0.392285 0.334605
vn -0.172704 0.873745 0.454634
vn 0.012238 -0.833583 -0.552232
vn 0.990905 0.131382 -0.028535
vn 0.625202 -0.723746 0.291940
vn 0.044252 -0.283181 -0.958037
vn -0.252358 -0.961638 -0.107334
vn 0.503494 -0.507584 -0.699118
vn 0.466811 -0.668844 0.578509
vn -0.899045 0.300699 0.318186
vn -0.440077 -0.861568 -0.252937
vn -0.417341 -0.469619 -0.777978
vn 0.798975 -0.590930 0.111423
vn 0.962188 -0.218848 0.162084
vn 0.239357 -0.970275 -0.035371
vn 0.915464 0.401990 -0.017060
vn -0.004761 0.043458 -0.999023
vn 0.998779 0.007508 -0.048311
vn 0.898495 0.336497 0.281838
vn 0.636647 -0.071688 0.767785
vn 0.000000 0.177557 -0.984100
vn 0.482772 0.875118 -0.032655
vn -0.960112 -0.189642 -0.205298
vn 0.966216 0.000732 0.257668
vn 0.281442 0.950591 -0.130955
vn -0.499588 0.385327 -0.775811
vn -0.817103 -0.390698 0.423841
vn 0.836085 0.541887 -0.085360
vn 0.997650 0.068117 0.005310
vn 0.792199 -0.298685 -0.532090
vn -0.013276 -0.814081 0.580554
vn -0.739982 -0.316660 0.593371
vn 0.113498 -0.738212 0.664937
vn -0.164922 0.986145 0.016236
vn -0.644765 -0.409406 0.645466
vn -0.963683 -0.257973 0.068758
vn -0.522263 0.789270 -0.322855
vn -0.383374 -0.641194 0.664724
g chicken_modified_chicken_modified_02_-_Default
usemtl 02_-_Default
s 1
f 399 487 128
f 303 380 180
f 285 272 63
f 425 226 76
f 505 73 213
f 513 70 128
f 36 520 100
f 187 307 227
f 270 2 327
f 126 121 366
f 356 157 123
f 502 252 492
f 22 189 163
f 61 367 134
f 460 435 255
f 180 228 164
f 387 203 136
f 403 348 93
f 206 35 146
f 42 465 177
f 495 423 283
f 375 246 414
f 394 30 430
f 214 203 387
f 458 220 98
f 324 185 378
f 395 338 36
f 473 118 518
f 96 226 466
f 375 19 344
f 95 237 332
f 78 304 222
f 8 211 356
f 311 43 229
f 8 400 449
f 465 100 177
f 123 52 293
f 384 380 259
f 257 328 498
f 301 513 220
f 516 146 57
f 73 38 432
f 393 127 430
f 367 305 92
f 333 127 265
f 428 248 426
f 196 48 124
f 473 521 17
f 505 302 124
f 410 39 342
f 349 196 124
f 286 512 355
f 39 6 342
f 342 293 205
f 169 473 518
f 312 278 450
f 70 450 128
f 159 398 432
f 378 400 324
f 421 274 409
f 90 497 206
f 377 260 65
f 309 202 504
f 147 108 281
f 303 180 164
f 450 285 399
f 442 500 504
f 133 12 129
f 447 109 63
f 505 213 302
f 400 138 148
f 414 168 375
f 320 382 67
f 81 320 409
f 393 500 442
f 507 105 92
f 34 506 377
f 483 201 147
f 181 271 28
f 324 287 185
f 178 12 446
f 232 387 136
f 121 498 328
f 198 421 67
f 506 83 377
f 343 487 501
f 229 392 133
f 500 393 113
f 312 113 278
f 37 11 54
f 384 367 459
f 153 301 458
f 113 415 56
f 62 156 479
f 366 121 190
f 5 460 255
f 458 309 153
f 5 255 278
f 217 242 181
f 175 515 81
f 178 322 31
f 368 102 181
f 280 10 347
f 6 400 8
f 409 320 418
f 169 152 102
f 121 341 190
f 16 90 206
f 43 299 229
f 109 447 395
f 241 401 234
f 106 107 406
f 356 342 8
f 318 253 269
f 414 434 495
f 445 107 106
f 207 240 64
f 392 60 133
f 43 481 299
f 495 283 112
f 17 164 473
f 92 105 501
f 164 118 473
f 332 237 517
f 302 213 39
f 434 198 382
f 241 35 497
f 37 106 60
f 76 226 96
f 255 300 355
f 243 93 71
f 259 126 411
f 173 495 112
f 181 28 368
f 54 455 487
f 368 350 476
f 285 63 445
f 310 441 188
f 57 35 105
f 501 455 92
f 278 355 272
f 501 71 343
f 239 269 325
f 36 109 395
f 213 237 95
f 352 210 502
f 252 49 353
f 133 446 12
f 127 393 442
f 161 442 202
f 42 425 76
f 510 225 409
f 191 434 382
f 369 162 414
f 42 265 230
f 42 177 265
f 222 304 100
f 348 158 108
f 414 162 168
f 357 97 412
f 414 495 173
f 239 402 25
f 247 119 426
f 202 309 458
f 202 442 504
f 25 454 318
f 133 311 229
f 257 101 292
f 34 30 506
f 32 389 485
f 285 278 272
f 466 226 187
f 516 206 146
f 324 280 287
f 102 476 169
f 50 310 98
f 483 274 201
f 53 164 228
f 32 398 389
f 303 17 13
f 520 36 335
f 27 200 444
f 486 289 84
f 449 211 8
f 255 355 278
f 394 506 30
f 172 330 492
f 303 164 17
f 252 353 172
f 366 411 126
f 151 248 268
f 507 165 57
f 119 475 426
f 355 447 272
f 292 476 350
f 505 124 73
f 230 307 425
f 438 488 244
f 356 329 157
f 129 12 76
f 178 465 276
f 521 257 498
f 377 65 78
f 415 393 430
f 227 22 163
f 451 341 328
f 92 305 507
f 326 151 268
f 163 189 396
f 344 19 220
f 424 185 287
f 409 516 271
f 267 205 293
f 253 383 210
f 246 201 198
f 175 176 4
f 332 408 148
f 31 322 36
f 148 408 324
f 153 450 70
f 190 341 451
f 96 129 76
f 121 126 79
f 445 106 37
f 261 151 326
f 382 198 67
f 252 29 49
f 507 305 411
f 339 86 338
f 276 465 42
f 98 220 19
f 318 203 47
f 214 40 340
f 225 175 409
f 207 378 240
f 188 441 307
f 409 271 242
f 348 9 158
f 132 62 479
f 84 370 486
f 210 383 261
f 60 446 133
f 61 459 367
f 207 449 400
f 318 136 203
f 339 512 26
f 26 444 200
f 112 283 4
f 489 423 495
f 399 285 445
f 220 513 344
f 410 342 205
f 37 392 229
f 466 167 96
f 469 13 79
f 29 298 49
f 255 435 300
f 378 207 400
f 87 384 459
f 78 327 377
f 481 61 299
f 368 28 165
f 39 213 95
f 50 168 162
f 517 358 332
f 378 185 84
f 426 40 247
f 96 167 43
f 395 447 339
f 481 459 61
f 142 488 363
f 517 237 32
f 487 343 128
f 96 311 133
f 50 162 310
f 107 109 322
f 101 257 184
f 446 60 406
f 178 31 465
f 177 304 333
f 412 326 357
f 479 156 394
f 293 356 123
f 29 461 298
f 248 357 268
f 211 449 329
f 343 344 513
f 296 108 201
f 214 125 247
f 112 22 173
f 181 242 271
f 152 518 186
f 357 248 263
f 502 261 461
f 328 341 121
f 501 105 234
f 234 243 501
f 90 59 497
f 152 169 518
f 176 225 510
f 272 447 63
f 366 507 411
f 230 161 188
f 369 22 227
f 253 438 269
f 302 410 349
f 411 305 384
f 234 401 243
f 103 215 200
f 167 481 43
f 212 202 144
f 127 479 394
f 247 40 214
f 377 327 34
f 454 236 232
f 29 502 461
f 214 340 47
f 406 322 178
f 284 402 239
f 310 162 441
f 79 216 121
f 98 310 144
f 520 78 222
f 229 11 37
f 188 307 230
f 454 136 318
f 32 485 517
f 483 371 274
f 305 367 384
f 37 54 445
f 163 176 468
f 98 144 458
f 355 514 286
f 11 455 54
f 152 217 181
f 180 380 87
f 412 461 326
f 12 276 76
f 215 86 200
f 504 312 153
f 265 127 161
f 57 146 35
f 450 278 285
f 105 35 234
f 228 288 170
f 414 173 369
f 134 367 92
f 269 25 318
f 201 274 421
f 227 307 441
f 338 86 335
f 53 228 466
f 375 168 19
f 458 144 202
f 455 11 134
f 216 498 121
f 78 482 479
f 184 473 169
f 133 129 96
f 29 252 502
f 48 38 124
f 414 246 198
f 506 156 83
f 213 73 32
f 244 325 438
f 466 187 53
f 47 253 318
f 102 152 181
f 298 461 412
f 234 35 241
f 85 521 216
f 395 339 338
f 349 410 266
f 369 173 22
f 333 304 479
f 227 468 53
f 142 363 492
f 184 257 521
f 190 292 350
f 274 516 409
f 466 228 170
f 136 454 232
f 31 36 100
f 229 299 134
f 5 113 56
f 37 60 392
f 189 22 112
f 502 210 261
f 276 12 178
f 479 482 132
f 5 278 113
f 212 161 202
f 113 393 415
f 516 16 206
f 441 369 227
f 47 203 214
f 32 237 213
f 57 165 28
f 504 500 312
f 257 451 328
f 315 25 402
f 357 263 321
f 101 476 292
f 347 204 287
f 216 521 498
f 497 403 401
f 124 302 349
f 352 253 210
f 147 201 108
f 466 170 167
f 355 300 514
f 312 500 113
f 492 363 352
f 217 186 510
f 34 327 2
f 447 355 339
f 175 81 409
f 501 487 455
f 329 449 64
f 92 455 134
f 47 383 253
f 476 102 368
f 325 269 438
f 76 276 42
f 424 486 370
f 266 410 205
f 368 507 366
f 335 270 520
f 63 107 445
f 520 327 78
f 296 344 343
f 356 293 342
f 343 71 296
f 200 86 339
f 127 394 430
f 434 489 495
f 396 176 163
f 315 236 454
f 134 11 229
f 200 435 103
f 352 363 488
f 322 406 107
f 201 246 296
f 451 292 190
f 177 333 265
f 410 302 39
f 253 488 438
f 72 103 460
f 267 266 205
f 230 425 42
f 151 40 426
f 186 217 152
f 175 4 515
f 244 488 142
f 426 248 151
f 239 290 284
f 370 185 424
f 188 212 310
f 39 95 138
f 350 366 190
f 54 487 399
f 63 109 107
f 217 409 242
f 310 212 144
f 497 59 403
f 401 241 497
f 239 25 269
f 259 469 126
f 93 348 296
f 6 138 400
f 28 516 57
f 387 125 214
f 78 479 304
f 186 518 118
f 418 320 67
f 230 265 161
f 488 253 352
f 301 70 513
f 31 100 465
f 301 153 70
f 361 263 248
f 475 428 426
f 507 368 165
f 504 153 309
f 510 186 118
f 188 161 212
f 501 243 71
f 357 326 268
f 93 243 403
f 402 284 315
f 168 50 19
f 71 93 296
f 161 127 442
f 4 176 396
f 325 290 239
f 189 112 4
f 347 287 280
f 153 312 450
f 259 380 13
f 172 492 252
f 361 428 319
f 368 366 350
f 516 371 16
f 126 469 79
f 384 259 411
f 198 434 414
f 281 108 158
f 342 6 8
f 191 423 489
f 489 434 191
f 307 187 425
f 406 60 106
f 340 383 47
f 383 340 261
f 36 322 109
f 4 283 515
f 458 301 220
f 357 321 97
f 356 211 329
f 292 451 257
f 84 289 240
f 184 521 473
f 53 187 227
f 324 400 148
f 200 27 435
f 435 460 103
f 185 370 84
f 446 406 178
f 170 288 459
f 260 377 83
f 177 100 304
f 425 187 226
f 27 300 435
f 330 142 492
f 87 459 288
f 13 17 85
f 156 506 394
f 332 358 10
f 403 243 401
f 339 26 200
f 19 50 98
f 327 520 270
f 288 180 87
f 59 9 348
f 348 108 296
f 445 54 399
f 148 138 95
f 128 343 513
f 296 246 375
f 271 516 28
f 38 73 124
f 169 101 184
f 355 512 339
f 409 418 421
f 57 105 507
f 118 468 510
f 352 502 492
f 287 204 424
f 468 176 510
f 10 408 332
f 167 170 481
f 454 25 315
f 85 17 521
f 101 169 476
f 65 482 78
f 333 479 127
f 280 408 10
f 4 396 189
f 261 326 461
f 216 79 85
f 403 59 348
f 96 43 311
f 340 151 261
f 170 459 481
f 85 79 13
f 303 13 380
f 248 428 361
f 510 409 217
f 408 280 324
f 432 398 73
f 53 468 118
f 469 259 13
f 497 35 206
f 163 468 227
f 32 73 398
f 421 418 67
f 340 40 151
f 338 335 36
f 148 95 332
f 483 147 371
f 288 228 180
f 240 378 84
f 274 371 516
f 158 9 281
f 162 369 441
f 299 61 134
f 52 267 293
f 198 201 421
f 87 380 384
f 100 520 222
f 225 176 175
f 296 375 344
f 118 164 53
f 64 449 207
f 30 34 72
f 39 138 6
f 128 450 399
g chicken_modified_chicken_modified_flubber
usemtl flubber
f 30 72 143
f 215 23 235
f 145 75 2
f 291 345 270
f 215 316 145
f 452 270 335
f 345 209 150
f 233 30 143
f 86 231 345
f 250 2 235
f 215 235 316
f 56 417 233
f 30 415 430
f 150 231 145
f 145 231 215
f 337 291 270
f 30 233 415
f 345 337 452
f 56 233 460
f 209 231 472
f 2 270 345
f 231 209 345
f 86 345 335
f 5 56 460
f 75 316 2
f 472 231 150
f 336 75 145
f 345 150 2
f 452 337 270
f 233 417 415
f 145 316 336
f 150 145 2
f 345 116 337
f 143 460 233
f 23 2 250
f 143 72 460
f 472 150 209
f 215 231 86
f 56 415 417
f 235 2 316
f 345 452 335
g chicken_modified_chicken_modified_eyes
usemtl eyes
f 364 24 275
f 273 484 91
f 24 493 3
f 1 104 55
f 493 381 331
f 385 493 364
f 223 199 117
f 192 273 91
f 223 117 381
f 192 365 249
f 364 493 24
f 264 55 69
f 453 477 139
f 1 91 104
f 1 55 365
f 364 275 477
f 385 199 223
f 55 264 359
f 493 331 3
f 365 55 359
f 69 484 496
f 199 139 117
f 223 493 385
f 69 55 104
f 484 69 104
f 199 453 139
f 104 91 484
f 453 364 477
f 385 364 453
f 91 1 192
f 1 365 192
f 199 385 453
f 493 223 381
g chicken_modified_chicken_modified_beak
usemtl beak
f 15 420 436
f 427 72 34
f 346 470 436
f 436 420 346
f 23 420 15
f 15 194 277
f 306 519 470
f 103 346 34
f 34 519 427
f 2 23 34
f 23 277 34
f 519 277 194
f 23 215 420
f 103 420 215
f 470 346 427
f 427 346 103
f 470 194 15
f 15 436 470
f 72 427 103
f 23 15 277
f 519 354 470
f 306 427 519
f 194 354 519
g chicken_modified_chicken_modified_feet
usemtl feet
f 437 160 391
f 191 94 423
f 68 283 423
f 294 462 122
f 160 379 193
f 467 440 490
f 439 508 314
f 137 197 471
f 467 457 439
f 149 219 474
f 33 295 115
f 314 130 174
f 457 323 508
f 193 456 21
f 74 174 372
f 360 397 463
f 508 141 443
f 130 474 372
f 405 422 313
f 439 174 74
f 388 499 382
f 429 160 397
f 149 314 258
f 154 320 81
f 208 89 373
f 429 397 471
f 140 372 282
f 120 140 282
f 88 376 77
f 77 66 88
f 443 141 7
f 456 114 21
f 174 130 171
f 314 149 130
f 443 245 404
f 499 191 382
f 297 362 122
f 122 362 294
f 41 45 351
f 131 51 219
f 308 413 80
f 282 462 294
f 494 160 429
f 147 238 131
f 297 122 58
f 386 155 33
f 149 258 9
f 423 94 68
f 491 262 245
f 386 94 135
f 457 279 80
f 443 258 314
f 478 251 405
f 154 81 21
f 431 448 221
f 33 313 295
f 82 509 99
f 494 111 391
f 443 314 508
f 245 18 491
f 374 179 464
f 448 195 390
f 33 155 313
f 195 238 390
f 413 433 80
f 474 166 372
f 372 166 74
f 140 130 372
f 171 58 174
f 182 195 51
f 21 114 154
f 390 110 46
f 41 388 45
f 499 376 88
f 416 208 373
f 110 281 258
f 130 149 474
f 21 515 283
f 130 140 171
f 373 66 416
f 94 386 68
f 135 499 14
f 467 279 457
f 94 191 499
f 295 77 115
f 308 467 7
f 379 218 456
f 221 503 431
f 208 416 99
f 80 433 323
f 515 21 81
f 419 224 251
f 407 388 154
f 390 238 110
f 193 68 463
f 140 297 171
f 317 494 429
f 58 372 174
f 135 511 155
f 140 120 297
f 21 283 68
f 422 405 480
f 221 390 440
f 195 448 51
f 433 413 254
f 88 373 82
f 88 66 373
f 323 141 508
f 82 14 88
f 110 258 443
f 499 388 41
f 114 456 351
f 59 90 219
f 33 437 386
f 120 362 297
f 9 258 281
f 490 7 467
f 155 478 313
f 413 308 254
f 58 122 462
f 308 80 279
f 391 111 137
f 111 494 317
f 171 297 58
f 374 262 256
f 219 51 431
f 374 256 179
f 155 386 135
f 464 490 20
f 390 221 448
f 77 218 379
f 89 44 82
f 405 224 480
f 74 440 467
f 282 294 120
f 77 376 41
f 511 334 419
f 14 99 135
f 115 77 379
f 66 334 135
f 114 45 407
f 66 135 416
f 308 7 254
f 254 7 141
f 440 503 221
f 18 245 443
f 20 490 440
f 179 262 491
f 99 14 82
f 295 422 480
f 323 457 80
f 114 407 154
f 154 388 382
f 131 182 51
f 374 464 20
f 219 431 474
f 374 404 245
f 390 46 440
f 18 490 464
f 457 508 439
f 58 462 372
f 110 404 20
f 407 45 388
f 443 7 490
f 141 323 433
f 88 14 499
f 448 431 51
f 317 197 111
f 137 471 397
f 474 431 166
f 256 262 179
f 166 440 74
f 405 251 224
f 405 313 478
f 131 238 182
f 99 509 208
f 464 179 491
f 360 137 397
f 313 422 295
f 282 372 462
f 440 46 20
f 111 197 137
f 245 262 374
f 294 362 120
f 160 494 391
f 147 110 238
f 334 511 135
f 437 391 360
f 89 509 44
f 193 21 68
f 419 480 224
f 9 59 149
f 166 503 440
f 351 183 41
f 44 509 82
f 115 379 160
f 419 334 480
f 382 320 154
f 160 437 115
f 16 371 131
f 131 371 147
f 404 110 443
f 511 419 251
f 135 94 499
f 74 467 439
f 77 41 183
f 114 351 45
f 376 499 41
f 160 193 463
f 334 295 480
f 431 503 166
f 155 511 478
f 463 68 386
f 183 218 77
f 20 404 374
f 219 16 131
f 308 279 467
f 464 491 18
f 456 218 183
f 314 174 439
f 16 219 90
f 208 509 89
f 20 46 110
f 471 197 317
f 66 77 295
f 360 463 437
f 295 334 66
f 18 443 490
f 160 463 397
f 437 463 386
f 135 99 416
f 251 478 511
f 238 195 182
f 391 137 360
f 373 89 82
f 433 254 141
f 471 317 429
f 147 281 110
f 149 59 219
f 115 437 33
f 183 351 456
f 379 456 193
f 34 277 519
}

show_obj
